#' Occurrence records
#'
#' The most common form of data stored by ALA are observations of
#' individual life forms, known as 'occurrences'. This function allows the
#' user to search for occurrence records that match their specific criteria,
#' and return them as a \code{data.frame} for analysis. Optionally,
#' the user can also request a DOI for a given download to facilitate citation and
#' re-use of specific data resources.
#'
#' @param taxa \code{data.frame}: generated by a call to \code{\link{select_taxa}()}. This
#' argument also accepts a vector of unique species identifiers.
#' @param filters \code{data.frame}: generated by a call to \code{\link{select_filters}()}
#' @param locations \code{string}: generated by a call to \code{\link{select_locations}()}
#' @param columns \code{data.frame}: generated by a call to \code{\link{select_columns}()}
#' @param mint_doi \code{logical}: by default no DOI will be generated. Set to
#' true if you intend to use the data in a publication or similar
#' @param doi \code{string}: this argument enables retrieval of occurrence
#' records previously downloaded from the ALA, using the DOI generated by the data.
#' Specifying this argument means all other arguments will be ignored.  
#' @details
#' Note that unless care is taken, some queries can be particularly large.
#' While most cases this will simply take a long time to process, if the number
#' of requested records is >50 million the call will not return any data. Users
#' can test whether this threshold will be reached by first calling
#' \code{\link{ala_counts}()} using the same arguments that they intend to pass to
#' \code{ala_occurrences}(). It may also be beneficial when requesting a large
#' number of records to show a progress bar by setting \code{verbose = TRUE} in
#' \code{\link{ala_config}()}.
#' @return A \code{data.frame} of occurrences, columns as specified
#' by \code{\link{select_columns}()}.
#' @examples \dontrun{
#' # Search for occurrences matching a taxon identifier
#' occ <- ala_occurrences(taxa = select_taxa("Reptilia"))
#'
#' # Search for occurrences in a year range
#' occ <- ala_occurrences(filters = select_filters(year = seq(2010, 2020)))
#'
#' # Search for occurrences in a WKT-specified area
#' polygon <- "POLYGON((146.24960 -34.05930,146.37045 -34.05930,146.37045 \
#' -34.152549,146.24960 -34.15254,146.24960 -34.05930))"
#' occ <- ala_occurrences(locations = select_locations(polygon))
#' }
#' @export ala_occurrences

ala_occurrences <- function(taxa, filters, locations, columns,
                            mint_doi = FALSE, doi) {

  config_verbose <- getOption("galah_config")$verbose
  assert_that(is.logical(mint_doi))
  assert_that(is.logical(config_verbose))
  
  if (!missing(doi)) {
    # search for data using DOI
    data_path <- doi_download(doi)
    df <- read.csv(unz(data_path, "data.csv"), stringsAsFactors = FALSE)
    attr(df, "doi") <- doi
    return(df)
  }
  
  if (missing(taxa)) { taxa <- NULL }
  if (missing(filters)) { filters <- NULL }
  if (missing(locations)) { locations <- NULL }
  
  if(missing(columns)) {
    if (config_verbose) {
      message("No columns specified, default columns will be returned.")
    }
    columns <- select_columns(group = "basic")
  }
  
  query <- build_query(taxa, filters, locations, columns)

  # Add columns after getting record count
  
  # handle caching
  caching <- getOption("galah_config")$caching

  if (caching) {
    cache_file <- cache_filename(c(getOption("galah_server_config")$
                                     base_url_biocache,
                                   path = "ws/occurrences/offline/download",
                                   params = unlist(query)), ext = ".zip")
    if (file.exists(cache_file)) {
      if (config_verbose) { message("Using existing file") }
      # look for file using query parameters
      data <- read.csv(unz(cache_file, "data.csv"), stringsAsFactors = FALSE)
      #TODO: Add DOI here
      return(data)
    }
  } else {
    cache_file <- tempfile(fileext = ".zip")
  }

  count <- record_count(query)
  check_count(count, config_verbose)
  
  assertion_cols <- columns[columns$type == "assertions", ]
  query$fields <- build_columns(columns[columns$type != "assertions", ])
  query$qa <- build_columns(assertion_cols)

  if (mint_doi) {
    query$mintDoi <- "true"
  }
  query$emailNotify <- email_notify()

  # Get data
  url <- getOption("galah_server_config")$base_url_biocache
  query <- c(query, email = user_email(), reasonTypeId = download_reason(),
             dwcHeaders = "true", sourceId = 2004)

  download_path <- wait_for_download(url, query, config_verbose)
  data_path <- ala_download(url = url,
                       path = download_path,
                       cache_file = cache_file, ext = ".zip")

  tryCatch(
    df <- read.csv(unz(data_path, "data.csv"), stringsAsFactors = FALSE),
    error = function(e) {
      message("There was an error reading the occurrence data; possibly no data was returned. This may be because
no valid column names have been provided. To check whether column names are valid, use `search_fields()`")
    }
  )

  # rename cols so they match requested cols
  names(df) <- rename_columns(names(df), type = "occurrence")

  # replace 'true' and 'false' with boolean
  if (nrow(assertion_cols) > 0) {
    df <- fix_assertion_cols(df, assertion_cols$name)
  }

  # add DOI as attribute
  attr(df, "doi") <- get_doi(mint_doi, data_path)

  return(df)
}


get_doi <- function(mint_doi, data_path) {
  doi <- NA
  if (as.logical(mint_doi)) {
    tryCatch(
      doi <- as.character(
        read.table(unz(data_path, "doi.txt"))$V1),
      warning = function(e) {
        e$message <- "No DOI was generated for download. The DOI server may
                        be down. Please try again later"
      })
  }
  return(doi)
}


wait_for_download <- function(url, query, verbose) {
  status <- ala_GET(url, "ws/occurrences/offline/download",
                    params = query, on_error = occ_error_handler)

  status_url <- parse_url(status$statusUrl)
  status <- ala_GET(url, path = status_url$path)

  # create a progress bar
  if (verbose) {
    pb <- txtProgressBar(max = 1, style = 3)
  }

  while(status$status == "inQueue") {
    status <- ala_GET(url, path = status_url$path)
  }

  while (tolower(status$status) == "running") {
    val <- (status$records / status$totalRecords)
    if (verbose) {
      setTxtProgressBar(pb, val)
    }
    status <- ala_GET(url, path = status_url$path)
    Sys.sleep(2)
  }
  if (verbose) {
    setTxtProgressBar(pb, value = 1)
    close(pb)
  }
  parse_url(status$downloadUrl)$path
}

check_count <- function(count, config_verbose) {
  if (count == 0) {
    stop("This query does not match any records.")
  } else if (count > 50000000) {
    stop("A maximum of 50 million records can be retrieved at once.",
         " Please narrow the query and try again.")
  } else {
    if (config_verbose) { message("This query will return ", count, " records") }
  }
}

doi_download <- function(doi, caching) {
  # strip useful part of DOI
  doi_str <- str_split(doi, "ala.")[[1]][2]
  if (is.na(doi_str)) {
    stop("DOI has not been generated by the ALA. ALA DOIs created by the ALA
         have a prefix of 10.26197")
  }
  url <- getOption("galah_config")$base_url_doi
  #cache_file <- cache_filename(c(url, doi_str))
  path <- ala_download(url, path = paste0("/doi/", doi_str, "/download"),
                       ext = ".zip", cache_file = tempfile())
  return(path)
}

download_reason <- function() {
  reason <- getOption("galah_config")$download_reason_id
  if (reason == "") {
    reason <- 4
  }
  reason
}


email_notify <- function() {
  notify <- as.logical(getOption("galah_config")$send_email)
  if (is.na(notify)) {
    notify <- FALSE
  }
  # ala api requires lowercase
  ifelse(notify, "true", "false")
}

user_email <- function() {
  email <- getOption("galah_config")$email
  if (email == "") {
    email <- Sys.getenv("email")
  }
  if (email == "") {
    stop("To download occurrence records you must provide a valid email ",
         "address registered with the ALA using `ala_config(email = )`")
  }
  email
}

occ_error_handler <- function(code) {
  if (code == 403) {
    stop("Status code 403 was returned for this occurrence download request. This may be because
  the email you provided is not registered with the ALA. Please check and try again.")
  }
}
