% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/general_operations.R
\name{spa_eval}
\alias{spa_eval}
\title{Evaluate the membership degree of a point in a \code{pgeometry} object}
\usage{
spa_eval(pgo, point)
}
\arguments{
\item{pgo}{A \code{pgeometry} object of any type.}

\item{point}{An \code{sfg} object of the type \code{POINT}.}
}
\value{
A numeric value between 0 and 1 that indicates the membership degree of a point (i.e., \code{sfg} object) in a spatial plateau object (i.e., \code{pgeometry} object).
}
\description{
\code{spa_eval()} evaluates the membership degree of a given point in a spatial plateau object of any type.
It returns a value in [0, 1] that indicates to which extent the point belongs to the \code{pgeometry} object.
}
\details{
The \code{spa_eval()} returns the membership degree of a simple point object (i.e., \code{sfg} object) in a given spatial plateau object (i.e., \code{pgeometry} object).
This evaluation depends on the following basic cases:
\itemize{
\item if the simple point object belongs to the interior or boundary of \emph{one} component of the spatial plateau object, it returns the membership degree of that component.
\item if the simple point object intersects more components (e.g., boundaries of region components, or different line components), it returns the maximum membership degree of all intersected components.
\item if the simple point object is disjoint to the support of the spatial plateau object, it returns 0.
}
}
\examples{
library(sf)

# Point components
pcp1 <- create_component("POINT(0 0)", 0.3)
pcp2 <- create_component("MULTIPOINT((2 2), (2 4), (2 0))", 0.5)
pcp3 <- create_component("MULTIPOINT((1 1), (3 1), (1 3), (3 3))", 0.9)
pcp4 <- create_component("MULTIPOINT((1 2), (2 1), (3 2))", 1)
pcp5 <- create_component("MULTIPOINT((0 0.5), (2 3))", 0.7)
pcp6 <- create_component("MULTIPOINT((0 1), (3 3.5))", 0.85)
pcp7 <- create_component("MULTIPOINT((1 0), (4 2))", 0.4)
# Line components
lcp1 <- create_component("LINESTRING(0 0, 1 1.5)", 0.2)
lcp2 <- create_component("LINESTRING(1 3, 1 2, 2 0.5)", 0.5)
lcp3 <- create_component("LINESTRING(2 1.2, 3 1.6, 4 4)", 0.7)
lcp4 <- create_component("LINESTRING(1 1.5, 2 1.2)", 1.0)
lcp5 <- create_component("LINESTRING(-1 1, 2 2)", 0.9)
# Polygon components
rcp1 <- create_component("POLYGON((0 0, 1 4, 2 2, 0 0))", 0.4)
rcp2 <- create_component("POLYGON((2 0.5, 4 1, 4 0, 2 0.5))", 0.8)

# Creating spatial plateau objects
ppoint <- create_pgeometry(list(pcp1, pcp2, pcp3, pcp4, pcp5), "PLATEAUPOINT")
pline <- create_pgeometry(list(lcp1, lcp2, lcp3), "PLATEAULINE")
pregion <- create_pgeometry(list(rcp1, rcp2), "PLATEAUREGION")
pcomp <- create_pgeometry(list(pcp6, pcp7, lcp4, lcp5), "PLATEAUCOMPOSITION")
pcol <- create_pgeometry(list(ppoint, pline, pregion, pcomp), "PLATEAUCOLLECTION")

point <- st_point(c(0, 0))

spa_eval(ppoint, point)
spa_eval(pline, point)
spa_eval(pregion, point)
spa_eval(pcomp, point)
spa_eval(pcol, point)
}
\references{
Formal definitions of this function are described in:
\itemize{
\item \href{https://ieeexplore.ieee.org/document/9882658}{Carniel, A. C.; Galdino, F.; Schneider, M. Evaluating Region Inference Methods by Using Fuzzy Spatial Inference Models. In Proceedings of the 2022 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2022), pp. 1-8, 2022.}
\item \href{https://ieeexplore.ieee.org/document/8491565}{Carniel, A. C.; Schneider, M. Spatial Plateau Algebra: An Executable Type System for Fuzzy Spatial Data Types. In Proceedings of the 2018 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2018), pp. 1-8, 2018.}
}
}
