% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/return_error.R
\name{return_error}
\alias{return_error}
\title{Compute forecast error}
\usage{
return_error(data_results, data_test = NULL, test_indices = NULL,
  metrics = c("mae", "mape", "mdape", "smape"), models = NULL,
  horizons = NULL, windows = NULL, group_filter = NULL)
}
\arguments{
\item{data_results}{An object of class 'training_results' or 'forecast_results' from running
\code{\link[=predict.forecast_model]{predict}} on a trained model.}

\item{data_test}{Required for forecast results only. If \code{data_results} is an object of class 'forecast_results', a data.frame used to
assess the accuracy of a 'forecast_results' object. \code{data_test} should have the outcome/target columns
and any grouping columns.}

\item{test_indices}{Required if \code{data_test} is given. A vector or 1-column data.frame of numeric
row indices or dates (class 'Date') with length nrow(data_test).}

\item{metrics}{Common forecast error metrics. See the Error Metrics section below for details. The
default behavior is to return all metrics.}

\item{models}{Optional. A character vector of user-defined model names supplied to \code{train_model()}.}

\item{horizons}{Optional. A numeric vector to filter results by horizon.}

\item{windows}{Optional. A numeric vector to filter results by validation window number.}

\item{group_filter}{Optional. A string for filtering plot results for grouped time-series
(e.g., \code{"group_col_1 == 'A'"}). The results are passed to \code{dplyr::filter()} internally.}
}
\value{
An S3 object of class 'validation_error' or 'forecast_error': A list of data.frames
of error metrics for the validation datasets or forecast dataset depending
on the \code{data_test} argument. \cr

A list containing: \cr

\itemize{
  \item Error metrics by horizon + validation window
  \item Error metrics by horizon, collapsed across validation windows
  \item Global error metrics collapsed across horizons and validation windows
}
}
\description{
Compute forecast error metrics on the validation datasets or a test dataset.
}
\section{Error Metrics}{


\itemize{
  \item \code{mae}: Mean absolute error
  \item \code{mape}: Mean absolute percentage error
  \item \code{mdape}: Median absolute percentage error
  \item \code{smape}: Symmetrical mean absolute percentage error
}
}

\section{Methods and related functions}{


The output of \code{return_error()} has the following generic S3 methods

\itemize{
  \item \code{\link[=plot.validation_error]{plot}} from \code{return_error()}
}
}

\examples{
# Sampled Seatbelts data from the R package datasets.
data("data_seatbelts", package = "forecastML")

# Example - Training data for 2 horizon-specific models w/ common lags per predictor.
horizons <- c(1, 12)
lookback <- 1:15

data_train <- create_lagged_df(data_seatbelts, type = "train", outcome_col = 1,
                               lookback = lookback, horizon = horizons)

windows <- create_windows(data_train, window_length = 12)

# User-define model - LASSO
# A user-defined wrapper function for model training that takes the following
# arguments: (1) a horizon-specific data.frame made with create_lagged_df(..., type = "train")
# (e.g., my_lagged_df$horizon_h) and, optionally, (2) any number of additional named arguments
# which are passed as '...' in train_model().
library(glmnet)
model_function <- function(data, my_outcome_col) {

  x <- data[, -(my_outcome_col), drop = FALSE]
  y <- data[, my_outcome_col, drop = FALSE]
  x <- as.matrix(x, ncol = ncol(x))
  y <- as.matrix(y, ncol = ncol(y))

  model <- glmnet::cv.glmnet(x, y, nfolds = 3)
  return(model)
}

# my_outcome_col = 1 is passed in ... but could have been defined in model_function().
model_results <- train_model(data_train, windows, model_name = "LASSO", model_function,
                             my_outcome_col = 1)

# User-defined prediction function - LASSO
# The predict() wrapper takes two positional arguments. First,
# the returned model from the user-defined modeling function (model_function() above).
# Second, a data.frame of predictors--identical to the datasets returned from
# create_lagged_df(..., type = "train"). The function can return a 1- or 3-column data.frame
# with either (a) point forecasts or (b) point forecasts plus lower and upper forecast
# bounds (column order and column names do not matter).
prediction_function <- function(model, data_features) {

  x <- as.matrix(data_features, ncol = ncol(data_features))

  data_pred <- data.frame("y_pred" = predict(model, x, s = "lambda.min"))
  return(data_pred)
}

# Predict on the validation datasets.
data_valid <- predict(model_results, prediction_function = list(prediction_function),
                      data = data_train)

# Forecast error metrics for validation datasets.
data_error <- return_error(data_valid)
}
