# Package Code 'fishmove'
# 
# Author: Johannes Radinger
###############################################################################


#Fishmove main default
fishmove <- function(species=NA,L=NA,AR=NA,SO=6,T=30,interval="confidence",rep=50,...){
	
	
	####### VALIDATION of INPUT ##########
	#Validation of Arguments
	if(missing(species) && missing(L)) stop("No fish length or species provided")
	if(missing(species) && missing(AR)) stop("No aspect ratio or species provided")
	if(!missing(species) && !is.element(species,speciesfishmove$SPECIES)) stop("Fish species is not in speciesfishmove")
	# if Species given than take arguments from species
	if(!missing(species) && is.element(species,speciesfishmove$SPECIES) && missing(L)){
		L.tmp <- speciesfishmove$LENGTH[speciesfishmove$SPECIES==species]
	}	
	if(!missing(species) && is.element(species,speciesfishmove$SPECIES) && missing(AR)){
		AR.tmp <- speciesfishmove$ASPECT.RATIO[speciesfishmove$SPECIES==species]
	}
	# if length or aspect ratio given than overwrite species arguments
	if(!missing(species) && is.element(species,speciesfishmove$SPECIES) && !missing(L)){
		warning("new.length will be used, not fish length from species-data",call. = FALSE)
		L.tmp <- L
	}
	if(!missing(species) && is.element(species,speciesfishmove$SPECIES) && !missing(AR)){
		warning("new.aspect.ratio will be used, not aspect ratio from species",call. = FALSE)
		AR.tmp <- AR
	}	
	# if fish length and aspect ratio are given but no species
	if(missing(species) && !missing(AR) && !missing(L)){
		L.tmp <- L
		AR.tmp <- AR
	}
	#warning if the argument value is outside the range of original regression
	if(min(datafishmove$LENGTH) > min(L.tmp) | max(L.tmp) > max(datafishmove$LENGTH)) warning("Fish length is outside the range of original regression",call. = FALSE)
	if(min(datafishmove$ASPECT.RATIO) > min(AR.tmp) | max(AR.tmp) > max(datafishmove$ASPECT.RATIO)) warning("Aspect Ratio is outside the range of original regression",call. = FALSE)
	if(min(datafishmove$STREAM.ORDER) > min(SO) | max(SO) > max(datafishmove$STREAM.ORDER)) warning("Stream order is outside the range of original regression",call. = FALSE)
	if(min(datafishmove$TIME) > min(T) | max(T) > max(datafishmove$TIME)) warning("Time is outside the range of original regression",call. = FALSE)
	#Validation of values
	if(!is.numeric(rep)) stop("rep is not numeric")
	if(rep>5000) stop("rep to large")
	if(!is.numeric(L.tmp)) stop("L is not numeric")
	#if(new.length.tmp<=0) stop("Fish length <= 0")
	if(!is.numeric(AR.tmp)) stop("AR is not numeric")
	#if(new.aspect.ratio.tmp<=0) stop("Aspect ratio < 0")
	if(!is.numeric(SO)) stop("SO is not numeric")
	#if(new.stream.order<=0) stop("new.stream.order < 0")
	if(!is.numeric(T)) stop("T is not numeric")
	#if(new.time<=0) stop("Time < 0")
	if(interval != "prediction" && interval !="confidence") stop("Interval must either be 'confidence' or 'prediction'")
	
	#transform from tmp to variable
	L  <- L.tmp
	AR <- AR.tmp
	
	
	# Getting string of regression coefficients
	coefs = function(model){
		c(coef(model),									#vector of coefficients
				summary(model)$coefficients[-1,4],		#p-values for all except Intercept
				pf(summary(model)$fstatistic[1], summary(model)$fstatistic[2],
						summary(model)$fstatistic[3], lower.tail = FALSE),		#overall p-value
				summary(model)$r.squared)				#r.squared
	}
	
	
	# Creating empty arrays for regression coefs and predicted values
	reg <- array(NA, dim = c(rep, 11, 2), dimnames=list(NULL,NULL,c("sigma_stat","sigma_mob")))
	pred <- array(NA, dim = c(rep, 3, 2,length(L),length(AR),length(SO),length(T)), 
			dimnames=list(NULL,NULL,c("sigma_stat","sigma_mob"),paste("L", L, sep="="),paste("AR", AR, sep="="),paste("SO", SO, sep="="),paste("T", T, sep="=")))
	
	
	for (i in 1:rep) {
		REP = NULL # needed to get visible binding for global var in ddply
		subsample <- ddply(datafishmove, .(REP), function(x){
					x[sample(nrow(x), 1), ]}) # subsample to correct for effect of pseudo replication
		model.fishmove.stat <- lm(log(SIGMA_STAT) ~log(LENGTH)+ASPECT.RATIO+sqrt(STREAM.ORDER)+log(TIME),data=subsample)
		model.fishmove.mob <- lm(log(SIGMA_MOB) ~log(LENGTH)+ASPECT.RATIO+sqrt(STREAM.ORDER)+log(TIME),data=subsample)
		reg[i,,"sigma_stat"] <- coefs(model.fishmove.stat) # regression parameters for stationary componenent
		reg[i,,"sigma_mob"] <- coefs(model.fishmove.mob) # regression parameters for mobile componenent
		for (j in 1:length(L)){
			for (k in 1:length(AR)){
				for (l in 1:length(SO)){
					for (m in 1:length(T)){
						newdata <- data.frame(LENGTH=L[j],ASPECT.RATIO=AR[k],STREAM.ORDER=SO[l],TIME=T[m])
						pred[i,,"sigma_stat",j,k,l,m] <- exp(predict.lm(model.fishmove.stat,newdata=newdata,interval=interval))
						pred[i,,"sigma_mob",j,k,l,m] <- exp(predict.lm(model.fishmove.mob,newdata=newdata,interval=interval))
					}}}}}
	
	
	
	coef.fishmove <- apply(reg,c(2,3),mean,na.rm=TRUE) #Calc mean of matrix columns/reg. param.
	rownames(coef.fishmove) <- c("Intercept","log(LENGTH)","ASPECT.RATIO","sqrt(STREAM.ORDER)","log(TIME)",
			"p-value log(LENGTH)","p-value ASPECT.RATIO","p-value sqrt(STREAM.ORDER)","p-value log(TIME)",
			"overall p value","R.squared")
	
	pred.fishmove <- apply(pred,c(2,3,4,5,6,7),mean,na.rm=TRUE) #Mean predicted values for given newdata
	rownames(pred.fishmove) <- c("fit","lwr","upr")
	
	out <- list(coef.fishmove=coef.fishmove,
			pred.fishmove =pred.fishmove
	)
	
	class(out) <- "fishmove"
	
	return(out)
}

# Print method for fishmove 
print.fishmove=function(x,...){
	cat("Predicted movement for selected parameters:\n")
	print(x$pred.fishmove)
}



# Summary method
summary.fishmove <- function(object,...){
	cat("Summary\n")
	cat("Regression coefficients:\n")
	print(object$coef.fishmove)
	cat("Predicted movement for selected parameters:\n")
	print(object$pred.fishmove)
}

# Plotting dispersal kernel 
pdk <-function(fishmove,p=0.66,...){
	
	#validation of arguments
	if(!is.numeric(p)) stop("p is not numeric")
	if(0 > p | p > 1) stop("p must be a value between 0 and 1")
	#if(length(fishmove$pred.fishmove["fit","sigma_stat",,,,])>1) stop("for plotting: please provide only single variables in fishmove")
	
	
	#dimesion to plot
	if(length(fishmove$pred.fishmove["fit","sigma_stat",,,,])>1 && !hasArg(dim)){
		warning("Multiple values supplied in fishmove. Only plotted for first values.",call. = FALSE)
	} 
	
	if(hasArg(dim)){
		print("Specifying dimensions not implemented yet")
		#fishmove.array <- fishmove$pred.fishmove[dim]
	} else{
		fishmove.array <- fishmove$pred.fishmove[,,1,1,1,1]
	}
	
	
	# General equation of heterogeneous movement kernel (two superimposed normal distributions)
	eq <- function(x,sigma_stat,sigma_mob,p) {
		(dnorm(x, sd = sigma_stat)*p)+(dnorm(x, sd = sigma_mob)*(1-p))
	}
	
	# Equations for mean, upper and lower movement kernel
	eq_fit <- function(x) {
		eq(x,fishmove.array["fit","sigma_stat"],fishmove.array["fit","sigma_mob"],p)
	}
	eq_lwr <- function(x) {
		eq(x,fishmove.array["lwr","sigma_stat"],fishmove.array["lwr","sigma_mob"],p)
	}
	eq_upr <- function(x) {
		eq(x,fishmove.array["upr","sigma_stat"],fishmove.array["upr","sigma_mob"],p)
	}
	
	# Creating dataframe for plotting, indicating min and max of x axis
	x <- seq(-1.5*fishmove.array["fit","sigma_mob"],1.5*fishmove.array["fit","sigma_mob"],1)
	y <- eq(x,fishmove.array["fit","sigma_stat"],fishmove.array["fit","sigma_mob"],p)
	tmp <- data.frame(x=x, y=y)
	
	# Plot commands to ggplot2
	pdk <- ggplot(tmp, aes(x=x, y=y))+
			stat_function(fun=eq_lwr,aes(colour="lower/upper bound"),n=500)+
			stat_function(fun=eq_upr,aes(colour="lower/upper bound"),n=300)+
			stat_function(fun=eq_fit,aes(colour="fitted mean"),size=0.8,n=300)+
			scale_y_continuous("Probability",limits=c(0,eq(0,fishmove.array["fit","sigma_stat"],fishmove.array["fit","sigma_mob"],p)))+
			scale_x_continuous("Movement Distance (m)")+
			scale_colour_manual(name="",values=c("fitted mean"="green4","lower/upper bound"="grey65"))+
			theme_bw()+
			theme(	legend.position = c(0.95,0.85),
					legend.justification ="right",
					legend.title = element_blank(),
					legend.background = element_rect(colour = 'grey', fill = 'white'),
					panel.grid.minor = element_blank())
	
	pdk #plot final density plot
}
