% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/first.R
\name{totalsobol.knn}
\alias{totalsobol.knn}
\title{Estimating Total Sobol' Indices from Data}
\usage{
totalsobol.knn(
  X,
  y,
  noise = TRUE,
  n.knn = 2,
  n.mc = nrow(X),
  mc.option = "random",
  rescale = TRUE,
  parl = NULL
)
}
\arguments{
\item{X}{a matrix or data frame containing the inputs.}

\item{y}{a vector containing the outputs.}

\item{noise}{a logical indicating whether the outputs are noisy.}

\item{n.knn}{number of nearest-neighbors for the inner loop conditional variance estimation.}

\item{n.mc}{number of Monte Carlo samples for the outer loop expectation estimation.}

\item{mc.option}{option for the Monte Carlo samples. The options are random(default)/twinning. See below for details.}

\item{rescale}{a logical indicating whether continuous inputs is rescaled to zero mean unit variance.}

\item{parl}{number of cores on which to parallelize the computation. If \code{NULL}, then no parallelization is done.}
}
\value{
A numeric vector of the total Sobol' indices estimation.
}
\description{
\code{totalsobol.knn} implements the estimation of the total Sobol' indices directly from scattered data.
Parallel computation is available to accelerate the estimation.
For categorical inputs, please convert them to factor before calling this function.
For large datasets, we support the use of subsample to reduce the computational cost.
}
\details{
\code{totalsobol.knn} provides consistent estimation of the total Sobol' indices (Sobol, 1993) from scattered data.
When the output is clean/noiseless (\code{noise=FALSE}), \code{totalsobol.knn} implements the nearest-neighbor estimator proposed in Broto et al. (2020).
See \code{shapleysobol_knn} from the \pkg{sensitivity} package for another implementation of the nearest-neighbor estimator.
When the output is noisy (\code{noise=TRUE}), \code{totalsobol.knn} implements the Noise-Adjusted Nearest-Neighbor (NANNE) estimator (Huang and Joseph, 2024).
NANNE estimator can correct the estimation bias of the nearest-neighbor estimator caused by the random noise.
Please see Huang and Joseph (2024) for a more detailed discussion and comparison.

For integer/numeric output, \code{n.knn=2} nearest-neighbors is recommended for the noisy data (Huang and Joseph, 2024),
and \code{n.knn=3} nearest-neighbors is suggested for the clean/noiseless data (Broto et al., 2020).
For numeric inputs, it is recommended to standardize them via setting the argument \code{rescale=TRUE}.
Categorical inputs are transformed via one-hot encoding for the nearest-neighbor search.
To speed up the nearest-neighbor search, k-d tree from the \pkg{FNN} package is used.
Also, parallel computation is also supported via the \pkg{parallel} package.

Last, for large datasets, we support the use of subsamples for further acceleration.
Use argument \code{n.mc} to specify the number of subsamples.
Two options (argument \code{mc.option}) are available for finding the subsamples: random and twinning (Vakayil and Joseph, 2022).
Twinning is able to find subsamples that better represent the big data, i.e.,
providing a more accurate estimation, but at a slightly higher computational cost.
For more details, please see the \pkg{twinning} package.
}
\examples{
ishigami <- function(x) {
  x <- -pi + 2*pi*x
  y <- sin(x[1]) + 7*sin(x[2])^2 + 0.1*x[3]^4*sin(x[1])
  return (y)
}

set.seed(123)
n <- 10000
p <- 3
X <- matrix(runif(n*p), ncol=p)
y <- apply(X,1,ishigami) + rnorm(n)
tsi <- totalsobol.knn(X, y, noise=TRUE, n.knn=2, rescale=FALSE)
print(round(tsi,3)) # Analytical Total Sobol' Indices: 0.558, 0.442, 0.244

}
\references{
Broto, B., Bachoc, F., & Depecker, M. (2020). Variance reduction for estimation of Shapley effects and adaptation to unknown input distribution. SIAM/ASA Journal on Uncertainty Quantification, 8(2), 693-716.

Huang, C., & Joseph, V. R. (2024). Factor Importance Ranking and Selection using Total Indices. arXiv preprint arXiv:2401.00800.

Sobol', I. Y. M. (1990). On sensitivity estimation for nonlinear mathematical models. Matematicheskoe modelirovanie, 2(1), 112-118.

Vakayil, A., & Joseph, V. R. (2022). Data twinning. Statistical Analysis and Data Mining: The ASA Data Science Journal, 15(5), 598-610.
}
\author{
Chaofan Huang \email{chaofan.huang@gatech.edu} and V. Roshan Joseph \email{roshan@gatech.edu}
}
