% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cached_read.R
\name{cached_read}
\alias{cached_read}
\alias{cached_read_csv}
\title{Read files via cache of file list and contents}
\usage{
cached_read(
  files,
  label,
  read_fn,
  cache = NULL,
  type = NULL,
  force = FALSE,
  skip_file_info = FALSE
)

cached_read_csv(
  files,
  label,
  read_type = NULL,
  cache = NULL,
  type = NULL,
  skip_file_info = FALSE,
  force = FALSE
)
}
\arguments{
\item{files}{A file or files to read with \code{read_fn}.}

\item{label}{A string to use as the name of the file to cache.}

\item{read_fn}{A function which takes file(s) as its first parameter and
reads them. To use a single-input read function such as
\code{arrow::read_csv_arrow()} with multiple files, use \code{\link[=vectorize_reader]{vectorize_reader()}},
e.g. \code{read_fn = vectorize_reader(arrow::read_csv_arrow)}.}

\item{cache}{One of the following:
\itemize{
\item The path to an existing directory to use for caching.
If \code{NULL} (default) uses the current path, using \code{\link[here:here]{here::here()}}.
\item An existing cache object as generated by \code{\link[=file_cache]{file_cache()}}.
}}

\item{type}{A string describing the type of cache.
Must be \code{NULL} or one of 'rds', 'parquet', or 'csv'.
If \code{NULL} (default), uses 'rds'.}

\item{force}{If \code{TRUE}, forces evaluation even if the cache exists.}

\item{skip_file_info}{Whether to skip saving and/or checking the file info.
Use this when just querying the file system (without opening files) is slow.}

\item{read_type}{Type of csv read function to use. One of:
\itemize{
\item "readr": \code{readr::read_csv()}
\item "arrow": \code{vectorize_reader(arrow::read_csv_arrow)()}
\item "data.table": \code{vectorize_reader(data.table::fread)()}
\item "base": \code{vectorize_reader(utils::read.csv)()}
\item \code{NULL} (default): uses the first installed.
}}
}
\value{
The result of \code{read_fn(files)}.
}
\description{
Reads data and save to a local file for easier management and re-reading.

By default, also saves the file info to determine whether the cache
is valid, or whether the contents need to be updated because the files
have been modified. To skip this, or force reading from scratch, set
\code{skip_file_info=TRUE} or \code{force=TRUE}, respectively.

If updating is called for, all the files are re-read.

\code{cached_read_csv()} is a convenience function using a csv read function
based on \code{read_type}.
}
\examples{
# Create a temporary directory for the cache.
tf <- tempfile()
dir.create(tf)

# A function that logs when it's called.
read_csv_log <- function(files) {
  message("Reading from file ...")
  return(vectorize_reader(read.csv)(files, stringsAsFactors = TRUE))
}

# `iris` data frame separated into multiple subset files.
iris_files <- system.file("extdata", package = "filecacher") |>
  list.files(pattern = "_only[.]csv$", full.names = TRUE)

# 1) First time, the message is shown.
iris_files |>
  cached_read("mtcars", read_csv_log, cache = tf) |>
  all.equal(iris)

# 2) Second time, no message is shown since the data is pulled from cache.
iris_files |>
  cached_read("mtcars", read_csv_log, cache = tf) |>
  all.equal(iris)

# 3) If desired, reloading can be forced using `force = TRUE`.
iris_files |>
  cached_read("mtcars", read_csv_log, cache = tf, force = TRUE) |>
  all.equal(iris)


unlink(tf, recursive = TRUE)
}
\seealso{
\code{\link[=vectorize_reader]{vectorize_reader()}} to convert a single-input read function into a
multiple-input function.
}
