#' Plot Model Assessment for NS-FFA
#'
#' @description
#' Creates a normalized and detrended quantile–quantile plot (a worm plot) comparing 
#' empirical annual maximum series data to quantile estimates from a fitted, parametric, 
#' and nonstationary model. Confidence intervals are also provided. The worm plot can
#' be interpreted using the following heuristics:
#'
#' - For a satisfactory fit, the worm (red data points) should be within the 95% 
#'   confidence interval (dashed black lines).
#' - If the worm (red points) is passes above/below the origin, the model mean
#'   is too small/large respectively.
#' - If the worm has a clear positive/negative slope, the model variance is too 
#'   small/large respectively.
#' - If the worm has a U-shape or inverted U-shape, the model is too skew to the
#'   left/right respectively.
#'
#' @param results List; model assessment results generated by [model_assessment()].
#'
#' @param ... Optional named arguments: 'title', 'xlabel', and 'ylabel'.
#'
#' @return \code{ggplot}; a plot containing:
#' - A black line representing a model with no deviation from the empirical quantiles.
#' - Red points denoting the estimated quantiles against the empirical quantiles.
#' - A dashed black line showing the 95% confidence intervals of the residuals.
#'
#' @references
#' van Buuren, S., Fredriks, M. Worm plot: a simple diagnostic device for modelling 
#' growth reference curves. Statistics in Medicine 20 (8), 1259-1277. 
#' \doi{10.1002/sim.746}
#'
#' @examples
#' # Initialize example data and params
#' data <- rnorm(n = 100, mean = 100, sd = 10) + seq(from = 1, to = 100)
#' years <- seq(from = 1901, to = 2000)
#' structure <- list(location = TRUE, scale = FALSE)
#'
#' # Evaluate model diagnostics
#' results <- model_assessment(data, "NOR", "MLE", NULL, years, structure)
#'
#' # Generate a model assessment plot
#' plot_nsffa_assessment(results)
#'
#' @importFrom stats dnorm
#' @import ggplot2
#' @export

plot_nsffa_assessment <- function(results, ...) {

	# Compute CIs and create a dataframe for the plot
	n <- length(results$data)
	x <- results$q_theoretical
	y <- results$q_empirical - x
	ci <- (1 / dnorm(x)) * sqrt(pnorm(x) * (1 - pnorm(x)) / n)
	df <- data.frame(x = x, y = y, lower = -ci, upper = ci)

	# Capture optional arguments
	args <- list(...)

    # Set default values, overriding if necessary
    title <- args$title %||% "NS-FFA Model Assessment"
    xlabel <- args$xlabel %||% expression(Normalized ~ Theoretical ~ Quantiles)
	ylabel <- args$ylabel %||% expression(Normalized ~ Residuals)

	# Generate the plot
	p1 <- ggplot(data = df) +
		geom_hline(yintercept = 0, linewidth = 1.1) +
		geom_point(
			aes(x = .data$x, y = .data$y), 
			color = "red",
			size = 3,
			alpha = 0.5
		) + 
		geom_line(
			aes(x = .data$x, y = .data$lower), 
			linetype = "dashed",
			linewidth = 1.1,
		) + 
		geom_line(
			aes(x = .data$x, y = .data$upper), 
			linetype = "dashed",
			linewidth = 1.1,
		) + 
		labs(
			title = title,
			x = xlabel,
			y = ylabel
		)

	# Add theme and scales and return
	add_theme(add_scales(p1))

}
