#' Block-Bootstrap Mann-Kendall Test for Trend Detection
#'
#' Performs a bootstrapped version of the Mann-Kendall trend test to adjust
#' for serial correlation in annual maximum series data. The BBMK test uses 
#' Spearman’s serial correlation test to identify the least insignificant lag, 
#' then applies a shuffling procedure to obtain the empirical p-value and confidence 
#' bounds for the Mann-Kendall test statistic.
#'
#' @inheritParams param-data
#' @inheritParams param-alpha
#' @inheritParams param-samples
#'
#' @return A list containing the test results, including:
#' - `data`: The `data` argument.
#' - `alpha`: The significance level as specified in the `alpha` argument.
#' - `null_hypothesis`: A string describing the null hypothesis.
#' - `alternative_hypothesis`: A string describing the alternative hypothesis.
#' - `statistic`: The Mann-Kendall S-statistic computed on the original series.
#' - `bootstrap`: Vector of bootstrapped Mann-Kendall test statistics.
#' - `p_value`: Empirical two-sided p-value derived from the bootstrap distribution.
#' - `bounds`: Empirical confidence interval bounds from the bootstrap distribution.
#' - `reject`: If `TRUE`, the null hypothesis was rejected at significance `alpha`.
#'
#' @details
#' The block size for reshuffling is equal to the `least_lag` as estimated in 
#' [eda_spearman_test()]. Bootstrap samples are generated by resampling blocks 
#' of the original data *without replacement*. This procedure effectively removes 
#' serial correlation from the data. 
#'
#' @seealso [plot_bbmk_test()], [eda_mk_test()], [eda_spearman_test()]
#'
#' @examples
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#' eda_bbmk_test(data, samples = 1000L)
#'
#' @references
#' Bayazit, M., 2015. Nonstationarity of hydrological records and recent trends in trend 
#' analysis: a state-of-the-art review. Environmental Processes 2 (3), 527–542. 
#' \doi{10.1007/s40710-015-0081-7}
#'
#' Khaliq, M.N., Ouarda, T.B.M.J., Gachon, P., Sushama, L., St-Hilaire, A., 2009. 
#' Identification of hydrological trends in the presence of serial and cross correlations: 
#' a review of selected methods and their application to annual flow regimes of Canadian 
#' rivers. Journal Hydrolology 368 (1–4), 117–130. \doi{10.1016/j.jhydrol.2009.01.035}
#'
#' Sonali, P., Nagesh Kumar, D., 2013. Review of trend detection methods and their 
#' application to detect temperature changes in India. Journal Hydrology 476, 212–227. 
#' \doi{10.1016/j.jhydrol.2012.10.034}
#'
#' @importFrom stats quantile
#' @export

eda_bbmk_test <- function(data, alpha = 0.05, samples = 10000L) {

	data <- validate_numeric("data", data, bounds = c(0, Inf))
	alpha <- validate_float("alpha", alpha, bounds = c(0.01, 0.1))
	samples <- validate_integer("samples", samples, bounds = c(1, Inf))

	least_lag <- eda_spearman_test(data, alpha)$least_lag
	statistic  <- eda_mk_test(data, alpha)$statistic

	# Blocks have size of least_lag to remove serial correlation 
	n <- length(data)
	size <- least_lag
	n_blocks <- ceiling(n / size)
	blocks <- split(data[1:(n_blocks * size)], rep(1:n_blocks, each = size))

	bootstrap <- sapply(1:samples, function(i) { 

		# Sample blocks without replacement
		sampled_blocks <- sample(blocks, n_blocks, replace = FALSE)
		sampled_data <- unlist(sampled_blocks, use.names = FALSE)

		# Compute the Mann-Kendall statistic for the bootstrap
		m <- outer(sampled_data, sampled_data, `-`)                   
  		sum(sign(m[lower.tri(m)]), na.rm = TRUE)

	})

	# The p-value is computed empirically using the bootstrap distribution
	p_value <- ifelse(
		statistic < 0, 
		2 * mean(statistic >= bootstrap),
		2 * mean(statistic <= bootstrap)
	)

	reject <- (p_value <= alpha)

	# The confidence interval bounds are determined empirically from the bootstrap
	bounds <- quantile(bootstrap, c(alpha / 2, 1 - (alpha / 2)))

	list(
		data = data,
		alpha = alpha,
		null_hypothesis = "There is no monotonic trend in the mean of the data.",
		alternative_hypothesis = "There is a monotonic trend in the mean of the data.",
		statistic = statistic,
		bootstrap = bootstrap,
		p_value = p_value,
		bounds = bounds,
		reject = reject
	)

}

