% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calcAB.R
\name{calcAB}
\alias{calcAB}
\title{Determination of optimal coefficients for computing weights of evidence in logistic regression}
\usage{
calcAB(W, mu = NULL)
}
\arguments{
\item{W}{Vector of coefficients of length (d+1), where d is the number of features, in the
case of M=2 classes, or (d+1,M) matrix of coefficients (or connection weights) in the case
of M>2 classes.}

\item{mu}{Optional vector containing the means of the d features.}
}
\value{
A list with two elements:
  \describe{
  \item{A}{Vector of length d (M=2) or matrix of size (d,M) (for M>2) of coefficients alpha.}
  \item{B}{Vector of length d (M=2) or matrix of size (d,M) (for M>2) of coefficients beta.}
 }
}
\description{
\code{calcAB} computes optimal coefficients alpha and beta needed to transform coefficients
from logistic regression (or connections weights between the last hidden layer and the output
layer of multilayer neural networks) into weights of evidence. These weights of evidence
can then be used to express the outputs of logistic regression or multilayer neural networks
as "latent" mass functions.
}
\examples{
## Example with 2 classes and logistic regression
data(ionosphere)
x<-ionosphere$x[,-2]
y<-ionosphere$y-1
fit<-glm(y ~ x,family='binomial')
AB<-calcAB(fit$coefficients,colMeans(x))
AB
## Example with K>2 classes and multilayer neural network
library(nnet)
data(glass)
K<-max(glass$y)
d<-ncol(glass$x)
n<-nrow(x)
x<-scale(glass$x)
y<-as.factor(glass$y)
p<-3 # number of hidden units
fit<-nnet(y~x,size=p)  # training a neural network with 3 hidden units
W1<-matrix(fit$wts[1:(p*(d+1))],d+1,p) # Input-to-hidden weights
W2<-matrix(fit$wts[(p*(d+1)+1):(p*(d+1) + K*(p+1))],p+1,K) # hidden-to-output weights
a1<-cbind(rep(1,n),x)\%*\%W1  # hidden unit activations
o1<-1/(1+exp(-a1)) # hidden unit outputs
AB<-calcAB(W2,colMeans(o1))
AB
}
\references{
T. Denoeux. Logistic Regression, Neural Networks and Dempster-Shafer Theory: a New Perspective.
Knowledge-Based Systems, Vol. 176, Pages 54–67, 2019.
}
\seealso{
\code{\link{calcm}}
}
\author{
Thierry Denoeux.
}
