% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateRM.R
\name{simulateRM}
\alias{simulateRM}
\alias{simulateIRM}
\alias{simulatePCRM}
\alias{rRM_Kiani}
\title{Simulation of confidence ratings and RTs in race confidence models}
\usage{
simulateRM(paramDf, n = 10000, model = "IRM", time_scaled = FALSE,
  gamma = FALSE, agg_simus = FALSE, stimulus = c(1, 2), delta = 0.01,
  maxrt = 15, seed = NULL)

rRM_Kiani(paramDf, n = 10000, time_scaled = FALSE, gamma = FALSE,
  agg_simus = FALSE, stimulus = c(1, 2), delta = 0.01, maxrt = 15,
  seed = NULL)
}
\arguments{
\item{paramDf}{a list or data frame with one row. Column names should match the names of
IRM and PCRM model parameter names. For different stimulus quality/mean drift rates, names should
be v1, v2, v3,.... The function allows also the process noise to vary (using
s1, s2, ...), which is otherwise not required and set to 1.
Additionally, the confidence thresholds should be given by names with
thetaUpper1, thetaUpper2,..., thetaLower1,... or, for symmetric thresholds only by theta1, theta2,....}

\item{n}{integer. The number of samples (per condition and stimulus direction) generated.
Total number of samples is \code{n*nConditions*length(stimulus)}.}

\item{model}{character scalar. One of "IRM" or "PCRM". ("IRMt" and "PCRMt" will also be accepted. In that case,
time_scaled is set to TRUE.)}

\item{time_scaled}{logical. Whether a time_scaled transformation for the confidence measure should
be used.}

\item{gamma}{logical. If TRUE, the gamma correlation between confidence ratings, rt and accuracy is
computed.}

\item{agg_simus}{logical. Simulation is done on a trial basis with rts outcome. If TRUE,
the simulations will be aggregated over RTs to return only the distribution of response and
confidence ratings. Default: FALSE.}

\item{stimulus}{numeric vector. Either 1, 2 or c(1, 2) (default).
Together with condition represents the experimental situation. In a binary decision task the presented
stimulus belongs to one of two categories. In the default setting trials with
both categories presented are simulated but one can choose to simulate only trials with the
stimulus coming from one category (each associated with positive drift in one of two accumulators).}

\item{delta}{numerical. Size of steps for the discretized simulation (see details).}

\item{maxrt}{numerical. Maximum reaction time to be simulated (see details). Default: 15.}

\item{seed}{numerical. Seeding for non-random data generation. (Also possible outside of the function.)}
}
\value{
Depending on \code{gamma} and \code{agg_simus}.

If \code{gamma} is \code{FALSE}, returns a \code{data.frame} with columns: \code{condition},
\code{stimulus}, \code{response}, \code{correct}, \code{rt}, \code{conf} (the continuous confidence
measure) and \code{rating} (the discrete confidence rating) or
(if \code{agg_simus=TRUE}): \code{condition}, \code{stimulus},\code{response}, \code{correct},
\code{rating} and \code{p} (for the probability of a response and rating, given
the condition and stimulus).

If \code{gamma} is \code{TRUE}, returns a \code{list} with elements:
\code{simus} (the simulated data frame) and \code{gamma}, which is again a \code{list} with elements
\code{condition}, \code{rt} and \code{correct}, each a \code{tibble} with two columns (see details for more
information).
}
\description{
Simulates the decision responses, reaction times and state of the loosing accumulator
together with a discrete confidence judgment  in the independent and partially anti-correlated
race model (IRM and PCRM) (Hellmann et al., preprint), given specific parameter constellations.
See \link{RaceModels} for more information about
parameters. Also computes the Gamma rank correlation between the confidence
ratings and condition (task difficulty), reaction times and accuracy in the
simulated output. Basically, this function is a wrapper for \code{\link{rIRM}}
and \code{\link{rPCRM}} for application in confidence experiments with
manipulation of specific parameters.
\code{rRM_Kiani} simulates a different version of race models, presented in
Kiani et al. (2014), but without a confidence measure.
}
\details{
The simulation is done by simulating normal variables in discretized steps until
one process reaches the boundary. If no boundary is met within the maximum time, response is
set to 0. The output of the fitting function \code{\link{fitRTConf}} with the respective model
fits the argument \code{paramDf} for simulation. The Gamma coefficients are computed separately for
correct/incorrect responses for the correlation of confidence ratings with condition and rt
and separately for conditions for the correlation of accuracy and confidence. The resulting
tibbles in the output thus have two columns. One for the grouping variable and one for the
Gamma coefficient.
}
\note{
Different parameters for different conditions are only allowed for drift rate, \code{v},
and process variability, \code{s}. All other parameters are used for all conditions.
}
\examples{
# Examples for "PCRM" model (equivalent applicable for "IRM" model)
# 1. Define some parameter set in a data.frame
paramDf <- data.frame(a=2,b=2, v1=0.5, v2=1, t0=0.1,st0=0,
                      wx=0.6, wint=0.2, wrt=0.2,
                      theta1=4)

# 2. Simulate trials for both stimulus categories and all conditions (2)
simus <- simulateRM(paramDf, n=300,model="PCRM", time_scaled=TRUE)
head(simus)
# equivalent:
simus <- simulateRM(paramDf, model="PCRMt")
\donttest{
  library(ggplot2)
  simus <- simus[simus$response!=0,]
  simus$rating <- factor(simus$rating, labels=c("unsure", "sure"))
  ggplot(simus, aes(x=rt, group=interaction(correct, rating),
                    color=as.factor(correct), linetype=rating))+
    geom_density(size=1.2)+
    facet_grid(rows=vars(condition), labeller = "label_both")
}

# automatically aggregate simulation distribution
# to get only accuracy x confidence rating distribution for
# all conditions
agg_simus <- simulateRM(paramDf, n = 200, model="PCRMt", agg_simus = TRUE)
head(agg_simus)
\donttest{
  agg_simus$rating <- factor(agg_simus$rating, labels=c("unsure", "sure"))
  library(ggplot2)
  ggplot(agg_simus, aes(x=rating, group=correct, fill=as.factor(correct), y=p))+
    geom_bar(stat="identity", position="dodge")+
    facet_grid(cols=vars(condition), labeller = "label_both")
}

\donttest{
  # Compute Gamma correlation coefficients between
  # confidence and other behavioral measures
  # output will be a list
  simu_list <- simulateRM(paramDf, model="IRMt", gamma=TRUE)
  simu_list
}

}
\references{
Hellmann, S., Zehetleitner, M., & Rausch, M. (preprint). Simultaneous modeling of choice,
confidence and response time in visual perception. https://osf.io/9jfqr/

Kiani, R., Corthell, L., & Shadlen, M.N. (2014) Choice certainty is informed
by both evidence and decision time.
Neuron, 84(6), 1329-1342. doi:10.1016/j.neuron.2014.12.015
}
\author{
Sebastian Hellmann.
}
