% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/moments.R
\name{moments}
\alias{moments}
\title{Compute distributional moments from the parameters}
\usage{
moments(par, fam_name, what = "mean", ex_fun = NULL)
}
\arguments{
\item{par}{Parameters of the modeled distribution in a data.frame form. Can
be Output of \code{\link{preds}}, for example.}

\item{fam_name}{Name of the used family in character form. Can be one of
\code{distreg.vis::dists$dist_name}. All gamlss.dist and exported bamlss
families are supported. To obtain the family from a model in character
form, use \code{\link{fam_obtainer}}.}

\item{what}{One of \code{"mean"}, \code{"upperlimit"}, \code{"lowerlimit"}.
If it is \code{mean} (which is also the default), then the mean of the
parameter samples is calculated. 2.5% and 97.5% quantiles are calculated
for \code{lowerlimit} and \code{upperlimit}, respectively.}

\item{ex_fun}{An external function \code{function(par) {...}} which
calculates a measure, whose dependency from a certain variable is of
special interest.}
}
\description{
This function takes (predicted) parameters of a response distribution and
calculates the corresponding distributional moments from it. Furthermore, you
can specify own functions that calculate measures depending on distributional
parameters.
}
\details{
With the exception of \link{betareg}, the distributional families behind the
estimation of the distributional regression models are represented by own
objects, e.g. \link[gamlss.dist]{GA} or \link[bamlss]{lognormal_bamlss}. We
worked together with both the authors of \link{gamlss} and \link{bamlss} such
that the functions to compute the moments from the parameters of the
underlying distribution is already implemented in the family functon itself.
As an example, try out \code{gamlss.dist::BE()$mean}, which shows one
example. The function \code{moments()} utilizes this fact and ensures that
the outcome is always in the right format: Two columns named `Expected_Value`
and `Variance` detailing the first two moments. One exception appears when an
external function is specified, at which point there are three columns.

Each row details one `scenario` meaning one covariate combination for which
to predict the moments. \code{moments()} is heavily used in
\link{plot_moments}, where moments are calculated over the entire range of
one variable.

If target distribution stems from a \link{bamlss} model, \code{moments()} can
also utilize the samples from the \link{preds} function to transform them.
This is important for correct estimates, as just taking the mean of the
samples and then using those means to estimate the moments can lead to
inaccurate results. \code{moments()} knows when samples of predicted
parameters were specified in the \code{par} argument, and then transforms the
samples to the moments, before taking averages. Only through this procedure
we even get credible intervals for the expected moments (see "upperlimit" and
"lowerlimit" as possible outcomes of argument \code{what}).
}
\examples{

# Get some artificial data
gamma_data <- model_fam_data(fam_name = "gamma")

# Estimate model
library("bamlss")
model <- bamlss(list(gamma ~ norm2 + binomial1,
                     sigma ~ norm2 + binomial1),
                     data = gamma_data,
                     family = gamma_bamlss())

# Get some predicted parameters in sample and without sample form
pred_params <- preds(model, vary_by = "binomial1")
pred_params_samples <- preds(model, vary_by = "binomial1", what = "samples")

# Now calculate moments - with samples more correct estimates come out
moments(pred_params, fam_name = "gamma", what = "mean")
moments(pred_params_samples, fam_name = "gamma", what = "mean")

# Now with specifying an external function
my_serious_fun <- function(par) {
  return(par[["mu"]] + 3*par[["sigma"]])
}
moments(pred_params_samples,
        what = "mean",
        fam_name = "gamma",
        ex_fun = "my_serious_fun")

}
