library(testthat)

context("Triangular distribution")

test_that("constructor",{
  expect_silent(Triangular$new())
  expect_error(Triangular$new(lower = 5, upper = 3))
  expect_error(Triangular$new(lower = 5, upper = 5))
  expect_error(Triangular$new(lower = -Inf))
  expect_error(Triangular$new(upper = Inf))
  expect_silent(Triangular$new(lower = 2.3, upper = 3.8))
  expect_silent(Triangular$new(lower = 2, upper = 8, mode = 6))
  expect_message(Triangular$new(lower = 2, upper = 8, mode = 1, symmetric = T, verbose = T))
  expect_equal(Triangular$new(symmetric = F)$.__enclos_env__$private$.type,"asymmetric")
  expect_equal(Triangular$new(symmetric = T)$.__enclos_env__$private$.type,"symmetric")
})

test_that("parameters", {
  expect_equal(Triangular$new(lower = 2.3, upper = 3.5)$getParameterValue("lower"),2.3)
  expect_equal(Triangular$new(lower = 2.3, upper = 3.5)$getParameterValue("upper"),3.5)
  expect_equal(Triangular$new(lower = 2.3, upper = 3.5)$getParameterValue("mode"),2.9)
  expect_equal(Triangular$new(lower = 2.3, upper = 3.5, symmetric = T, mode = 4)$getParameterValue("mode"),2.9)
  expect_equal(Triangular$new(lower = 2.3, upper = 3.5, mode = 2.6)$getParameterValue("mode"),2.6)
})

test_that("properties & traits",{
  expect_equal(Triangular$new()$valueSupport(), "continuous")
  expect_equal(Triangular$new()$variateForm(), "univariate")
  expect_equal(Triangular$new(symmetric = T)$symmetry(), "symmetric")
  expect_equal(Triangular$new(symmetric = F)$symmetry(), "symmetric")
  expect_equal(Triangular$new(symmetric = F, mode = 0.2)$symmetry(), "asymmetric")
  expect_equal(Triangular$new()$sup(), 1)
  expect_equal(Triangular$new()$inf(), 0)
  expect_equal(Triangular$new()$dmax(), 1)
  expect_equal(Triangular$new()$dmin(), 0)
})

t = Triangular$new()
test_that("statistics",{
  expect_equal(t$mean(), 0.5)
  expect_equal(t$variance(), 0.75/18)
  expect_equal(t$skewness(), 0)
  expect_equal(t$kurtosis(T), -0.6)
  expect_equal(t$kurtosis(F), 2.4)
  expect_equal(t$entropy(), 0.5 + log(0.5,2))
  expect_equal(t$mgf(0),  NaN)
  expect_equal(t$mgf(1),  2 * (0.5 - exp(0.5) + 0.5*exp(1)) / (0.25))
  expect_equal(t$cf(0),  as.complex(NaN))
  expect_equal(t$cf(1),  (-2 * (0.5 - exp(0.5i) + 0.5*exp(1i))) / (0.25))
  expect_equal(t$mode(),0.5)
  expect_equal(t$pgf(1), NaN)
  expect_equal(t$pdf(1), extraDistr::dtriang(1,0,1,0.5))
  expect_equal(t$cdf(1), extraDistr::ptriang(1,0,1,0.5))
  expect_equal(t$quantile(0.324), extraDistr::qtriang(0.324,0,1,0.5))
  expect_equal(t$cdf(t$quantile(0.324)), 0.324)
  expect_silent(t$rand(10))
})

t = Triangular$new(lower=1,upper=5,symmetric=F)
test_that("update parameters",{
  expect_equal(Triangular$new(lower = 2, upper = 3, symmetric = T)$setParameterValue(lst = list(mode = 5))$getParameterValue("mode"),2.5)
  expect_equal(Triangular$new(lower = 2, upper = 3, symmetric = F)$setParameterValue(lst = list(mode = 2.1))$getParameterValue("mode"),2.1)
  expect_error(t$setParameterValue(lst = list(upper = 0)))
  expect_silent(t$setParameterValue(lst = list(upper = 2)))
  expect_error(t$setParameterValue(lst = list(lower = 3)))
  expect_error(t$setParameterValue(lst = list(mode = 6)))
  expect_error(t$setParameterValue(lst = list(mode = -Inf)))
  expect_silent(t$setParameterValue(lst = list(lower = 1)))
  expect_silent(t$setParameterValue(lst = list(lower = 1, upper = 3)))
  expect_error(t$setParameterValue(lst = list(lower = 1, upper = 0)))
})
