#' ctKalman 
#'
#' Outputs predicted, updated, and smoothed estimates of manifest indicators and latent states, 
#' with covariances, for specific subjects from data fit with \code{\link{ctStanFit}}, 
#' based on medians of parameter distribution.
#' 
#' @param fit fit object as generated by \code{\link{ctStanFit}}.
#' @param datalong Optional long format data object as used by \code{\link{ctStanFit}}. 
#' If not included, data from fit will used. 
#' @param timerange Either 'asdata' to just use the observed data range, or a numeric vector of length 2 denoting start and end of time range, 
#' allowing for estimates outside the range of observed data. Currently unused for ctStan fits.
#' @param timestep Either 'asdata' to just use the observed data 
#' (which also requires 'asdata' for timerange) or a positive numeric value
#' indicating the time step to use for interpolating values. Lower values give a more accurate / smooth representation,
#' but take a little more time to calculate. Currently unavailable for ctStan fits.
#' @param subjects vector of integers denoting which subjects (from 1 to N) to plot predictions for. 
#' @param removeObs Logical. If TRUE, observations (but not covariates)
#' are set to NA, so only expectations based on
#' parameters and covariates are returned. 
#' @param plot Logical. If TRUE, plots output instead of returning it. 
#' See \code{\link{plot.ctKalman}} for the possible arguments.
#' @param ... additional arguments to pass to \code{\link{plot.ctKalman}}.
#' @return Returns a list containing matrix objects etaprior, etaupd, etasmooth, y, yprior, 
#' yupd, ysmooth, prederror, time, loglik,  with values for each time point in each row. 
#' eta refers to latent states and y to manifest indicators - y itself is thus just 
#' the input data. 
#' Covariance matrices etapriorcov, etaupdcov, etasmoothcov, ypriorcov, yupdcov, ysmoothcov,  
#' are returned in a row * column * time array. 
#' Some outputs are unavailable for ctStan fits at present.
#' If plot=TRUE, nothing is returned but a plot is generated.
#' @examples
#' \donttest{
#' #Basic
#' ctKalman(ctstantestfit, timerange=c(0,60), timestep=.5, plot=TRUE)
#' 
#' #Multiple subjects, y and yprior, showing plot arguments
#' ctKalman(ctstantestfit, timerange=c(0,60), timestep=.1, plot=TRUE,
#'   subjects=2:3, 
#'   kalmanvec=c('y','yprior'),
#'   errorvec=c(NA,'ypriorcov')) #'auto' would also have achieved this
#'   }
#' @export

ctKalman<-function(fit, datalong=NULL, timerange='asdata', timestep=sd(fit$standata$time,na.rm=TRUE)/50,
  subjects=1, removeObs = FALSE, plot=FALSE, ...){
  type=NA
  if('ctStanFit' %in% class(fit)) type='stan' 
  if('ctsemFit' %in% class(fit)) type ='omx'
  if(is.na(type)) stop('fit object is not from ctFit or ctStanFit!')
  
  subjects <- sort(subjects) #in case not entered in ascending order
  
  if(type=='stan'){
    if(all(timerange == 'asdata')) timerange <- range(fit$standata$time[fit$standata$subject %in% subjects])
    if(timestep != 'asdata' && fit$ctstanmodel$continuoustime) {
      if(fit$ctstanmodel$continuoustime != TRUE) stop('Discrete time model fits must use timestep = "asdata"')
      times <- seq(timerange[1],timerange[2],timestep)
      fit$standata <- standataFillTime(fit$standata,times)
    } 
    idstore <- fit$standata$subject
    if(!class(fit$stanfit) %in% 'stanfit') {
      fit$standata <- standatact_specificsubjects(fit$standata, subjects = subjects)
      idstore <- as.integer(subjects[fit$standata$subject])
    }
    if(class(fit$stanfit) %in% 'stanfit') fit$standata$dokalmanrows <-
        as.integer(fit$standata$subject %in% subjects |
            as.logical(match(unique(fit$standata$subject),fit$standata$subject)))

    if(removeObs){
      sapply(c('nobs_y','nbinary_y','ncont_y','whichobs_y','whichbinary_y','whichcont_y'),
        function(x) fit$standata[[x]][] <<- 0L)
      fit$standata$Y[] <- 99999
    }
    out <- ctStanKalman(fit,collapsefunc=mean) #extract state predictions
    out$id <- idstore #as.integer(subjects[out$id]) #get correct subject indicators
    out <- meltkalman(out)
    out=out[!(out$Subject %in% subjects) %in% FALSE,]
  }
  
  if(type !='stan'){
    
    out<-list()
    if(timerange[1] != 'asdata' & timestep[1] == 'asdata') stop('If timerange is not asdata, a timestep must be specified!')
    
    if(!is.null(datalong)) { #adjust ids and colnames as needed
      
      datalong <- makeNumericIDs(datalong, fit$ctstanmodel$subjectIDname,fit$ctstanmodel$timeName) #ensure id's are appropriate
      colnames(datalong)[colnames(datalong)==fit$ctstanmodel$subjectIDname] <- 'subject'
      colnames(datalong)[colnames(datalong)==fit$ctstanmodel$timeName] <- 'time'
    }
    
    if(is.null(datalong)) { #get relevant data

      if(is.null(fit$mxobj$expectation$P0)) { #if not fit with kalman filter then data needs rearranging
        datalong=suppressMessages(ctWideToLong(datawide = fit$mxobj$data$observed[subjects,,drop=FALSE],
          Tpoints=fit$ctmodelobj$Tpoints,
          n.manifest=fit$ctmodelobj$n.manifest,manifestNames = fit$ctmodelobj$manifestNames,
          n.TDpred=fit$ctmodelobj$n.TDpred,TDpredNames = fit$ctmodelobj$TDpredNames,
          n.TIpred = fit$ctmodelobj$n.TIpred, TIpredNames = fit$ctmodelobj$TIpredNames))
        datalong <- suppressMessages(ctDeintervalise(datalong = datalong,id = 'id',dT = 'dT'))
        datalong[,'id'] <- subjects[datalong[,'id'] ]
      } else {
        datalong=fit$mxobj$data$observed
        datalong <- suppressMessages(ctDeintervalise(datalong = datalong,id = 'id',dT = 'dT1'))
      }
      colnames(datalong)[colnames(datalong) == 'id'] <- 'subject'
      
      
    }
    
    
    
    if(!all(subjects %in% datalong[,'subject'])) stop('Invalid subjects specified!')
    
    for(subjecti in subjects){
      #setup subjects data, interpolating and extending as necessary
      sdat=datalong[datalong[,'subject'] == subjecti,,drop=FALSE]
      if(timestep != 'asdata' || timerange[1] != 'asdata') {
        if(timerange[1]=='asdata') stimerange <- range(sdat[,'time']) else {
          stimerange <- timerange
          if(timerange[1] > min(sdat[,'time']) || timerange[2] < max(sdat[,'time']) ) stop('Specified timerange must contain all subjects time ranges!')
        }
        snewtimes <- seq(stimerange[1],stimerange[2],timestep)
        snewdat <- array(NA,dim=c(length(snewtimes),dim(sdat)[-1]),dimnames=list(c(),dimnames(sdat)[[2]]))
        snewdat[,'time'] <- snewtimes
        snewdat[,fit$ctmodelobj$TDpredNames] <- 0
        sdat <- rbind(sdat,snewdat)
        sdat[,'time'] <- round(sdat[,'time'],10)
        sdat<-sdat[!duplicated(sdat[,'time']),,drop=FALSE]
        sdat <- sdat[order(sdat[,'time']),,drop=FALSE]
        sdat[,c(fit$ctmodelobj$manifestNames,fit$ctmodelobj$TDpredNames)] [sdat[,c(fit$ctmodelobj$manifestNames,fit$ctmodelobj$TDpredNames)]==99999] <- NA
        sdat[,'subject'] <- subjecti
      }
      
      #get parameter matrices
      # 
      model <- summary(fit)
      # model <- model$
      
      #get kalman estimates
      
      out[[paste('subject',subjecti)]]<-Kalman(kpars=model,
        datalong=sdat,
        manifestNames=fit$ctmodelobj$manifestNames,
        latentNames=fit$ctmodelobj$latentNames,
        TDpredNames=fit$ctmodelobj$TDpredNames,
        idcol='subject',
        timecol='time')
    }
    class(out) <- c('ctKalman',class(out))
  }
  
  if(plot) {
    plot(x=out,subjects=subjects,...)
  } else return(out)
}



#' Plots Kalman filter output from ctKalman.
#'
#' @param x Output from \code{\link{ctKalman}}. In general it is easier to call 
#' \code{\link{ctKalman}} directly with the \code{plot=TRUE} argument, which calls this function.
#' @param subjects vector of integers denoting which subjects (from 1 to N) to plot predictions for. 
#' @param kalmanvec string vector of names of any elements of the output you wish to plot, 
#' the defaults of 'y' and 'yprior' plot the original data, 'y', 
#' and the prior from the Kalman filter for y. Replacing 'y' by 'eta' will 
#' plot latent variables instead (though 'eta' alone does not exist) and replacing 'prior' 
#' with 'upd' or 'smooth' respectively plotting updated (conditional on all data up to current time point)
#' or smoothed (conditional on all data) estimates.
#' @param errorvec vector of names of covariance elements to use for uncertainty indication 
#' around the kalmanvec items. 'auto' uses the latent covariance when plotting
#' latent states, and total covariance when plotting expectations of observed states. 
#' Use NA to skip uncertainty plotting.
#' @param errormultiply Numeric denoting the multiplication factor of the std deviation of errorvec objects. 
#' Defaults to 1.96, for 95\% intervals.
#' @param ltyvec vector of line types, varying over dimensions of the kalmanvec object.
#' @param colvec color vector, varying either over subject if multiple subjects, or otherwise over 
#' the dimensions of the kalmanvec object.
#' @param lwdvec vector of line widths, varying over the kalmanvec objects. 
#' @param subsetindices Either NULL, or vector of integers to use for subsetting the (columns) of kalmanvec objects.
#' @param pchvec vector of symbol types, varying over the dimensions of the kalmanvec object.
#' @param typevec vector of plot types, varying over the kalmanvec objects. 'auto' plots lines for
#' any  'prior', 'upd', or 'smooth' objects, and points otherwise.
#' @param grid Logical. Plot a grid?
#' @param add Logical. Create a new plot or update existing plot?
#' @param plotcontrol List of graphical arguments (see \code{\link{par}}), 
#' though lty,col,lwd,x,y, will all be ignored.
#' @param legend Logical, whether to include a legend if plotting.
#' @param legendcontrol List of arguments to the \code{\link{legend}} function.
#' @param polygoncontrol List of arguments to the \code{\link{ctPoly}} function for filling the uncertainty region.
#' @param polygonalpha Numeric for the opacity of the uncertainty region.
#' @param ... not used.
#' @return Generates plots.
#' @method plot ctKalman
#' @export
#' @examples
#' \donttest{
#' data(AnomAuth) 
#' AnomAuthmodel <- ctModel(LAMBDA = matrix(c(1, 0, 0, 1), nrow = 2, ncol = 2), 
#'   Tpoints = 5, n.latent = 2, n.manifest = 2,  TRAITVAR = NULL) 
#' AnomAuthfit <- ctFit(AnomAuth, AnomAuthmodel)
#' ctKalman(AnomAuthfit,subjects=1,plot=TRUE)
#' }
plot.ctKalman<-function(x, subjects=1, kalmanvec=c('y','yprior'),
  errorvec='auto', errormultiply=1.96,
  ltyvec="auto",colvec='auto', lwdvec='auto', 
  subsetindices=NULL,pchvec='auto', typevec='auto',grid=FALSE,add=FALSE, 
  plotcontrol=list(ylab='Value',xlab='Time',xaxs='i',lwd=2,mgp=c(2,.8,0)),
  polygoncontrol=list(steps=20),polygonalpha=.1,
  legend=TRUE, legendcontrol=list(x='topright',bg='white',cex=.7),...){
  
  if(!'ctKalman' %in% class(x)) stop('not a ctKalman object')
  
  
  
  if('ctKalman' %in% class(x)){ #base plots
    if(length(subjects) > 1 & colvec[1] =='auto') colvec = rainbow(length(subjects),v=.9)
    
    if(lwdvec[1] %in% 'auto') lwdvec=rep(2,length(kalmanvec))
    
    if(is.null(plotcontrol$ylab)) plotcontrol$ylab='Variable'
    if(is.null(plotcontrol$xlab)) plotcontrol$xlab='Time'
    
    if(typevec[1] %in% 'auto') typevec=c('p','l')[grepl("prior|upd|smooth|eta",kalmanvec)+1]
    
    if(errorvec[1] %in% 'auto') {
      errorvec=rep(NA,length(kalmanvec))
      errorvec[grepl("prior|upd|smooth",kalmanvec)]<-paste0(
        kalmanvec[grepl("prior|upd|smooth",kalmanvec)],'cov')
    }
    
    if(is.null(plotcontrol$xlim)) plotcontrol$xlim <- range(sapply(x,function(x) x$time))
    if(is.null(plotcontrol$ylim)) {
      plotcontrol$ylim <- range(unlist(lapply(x,function(x) { #for every subject
        if(!is.null(x)){
          ret<-c()
          for(kveci in 1:length(kalmanvec)){
            est<-x[[kalmanvec[kveci]]] [,
              if(is.null(subsetindices)) 1:dim(x[[kalmanvec[kveci]]])[2] else subsetindices]
            
            if(!is.na(errorvec[kveci])) err <- sqrt(abs(c(apply(x[[errorvec[kveci]]][
              (if(is.null(subsetindices)) 1:dim(x[[errorvec[kveci]]])[2] else subsetindices),
              (if(is.null(subsetindices)) 1:dim(x[[errorvec[kveci]]])[2] else subsetindices),
              ,drop=FALSE],3,diag))))
            
            if(is.na(errorvec[kveci])) err <- 0
            
            esthigh <- est + err*errormultiply
            estlow <- est - err*errormultiply
            ret <- c(ret,esthigh,estlow)
          }
          return(ret)}})),na.rm=TRUE)
      if(legend) plotcontrol$ylim[2] <- plotcontrol$ylim[2] + sd(plotcontrol$ylim)/4
    }
    
    
    
    
    legendtext<-c()
    legendcol <- c()
    legendlty<-c()
    legendpch<-c()
    
    #when not set to auto, must define 'new' vector as the user specified vector
    colvecnew <- colvec
    ltyvecnew<-ltyvec
    pchvecnew <- pchvec
    
    for(si in 1:length(subjects)){#subjects
      subjecti = subjects[si]
      subiname=paste('subject',subjecti)
      names(x)[si] <-subiname
      plist<-plotcontrol
      if(length(subjects) > 1) {
        plist$col = colvec[si] #set colour based on subject if multiple subjects
      }
      
      for(kveci in 1:length(kalmanvec)){ #kalman output types
        kvecdims=1:dim(x[[subiname]][[kalmanvec[kveci]]])[-1]
        if(length(subjects) == 1 & colvec[1] =='auto') colvecnew = rainbow(length(kvecdims),v=.9)
        if(any(subsetindices > max(kvecdims))) stop('subsetindices contains a value greater than relevant dimensions of object in kalmanvec!')
        if(!is.null(subsetindices)) kvecdims=kvecdims[subsetindices]
        if(rl(ltyvec[1]=='auto')) ltyvecnew <- 1:length(kvecdims) else ltyvecnew <- ltyvec
        
        if(rl(pchvec[1] =='auto')) pchvecnew = 1:(length(kvecdims)) else pchvecnew <- pchvec
        
        for(dimi in 1:length(kvecdims)){ #dimensions of kalman matrix
          kdimi <- kvecdims[dimi]
          plist$x=x[[subiname]]$time
          plist$y=x[[subiname]][[kalmanvec[kveci]]][,kdimi] 
          plist$lwd=lwdvec[kveci]
          plist$lty=ltyvecnew[dimi] 
          plist$pch=pchvecnew[dimi]
          plist$type=typevec[kveci]
          if(length(subjects)==1) plist$col=colvecnew[dimi]
          
          
          if(subjecti == subjects[1] & kveci==1 && dimi == 1 && !add) {
            
            do.call(graphics::plot.default,plist) 
            if(grid) {
              grid()
              par(new=TRUE)
              do.call(graphics::plot.default,plist) 
              par(new=FALSE)
            }
          } else do.call(graphics::points.default,plist) 
          
          if(!is.na(errorvec[kveci])){
            if(is.null(x[[subiname]][[errorvec[kveci]]])) stop('Invalid errorvec specified!')
            backwardstimesindex=order(plist$x,decreasing=TRUE)
            
            ctpolyargs<-polygoncontrol
            ctpolyargs$x=c(plist$x)
            ctpolyargs$ylow=c(plist$y - errormultiply * sqrt(abs(x[[subiname]][[errorvec[kveci]]][kdimi,kdimi,])))
            ctpolyargs$y=c(plist$y)
            ctpolyargs$yhigh=c(plist$y + errormultiply * sqrt(abs(x[[subiname]][[errorvec[kveci]]][kdimi,kdimi,])))
            ctpolyargs$col=grDevices::adjustcolor(plist$col,alpha.f=polygonalpha)
            ctpolyargs$col =grDevices::adjustcolor(ctpolyargs$col,alpha.f=max(c(.004,polygonalpha/sqrt(ctpolyargs$steps))))
            do.call(ctPoly,ctpolyargs)
            
            #add quantile lines
            plist$y <- ctpolyargs$ylow
            plist$lwd <- 1
            # plist$col <- grDevices::adjustcolor(plist$col,alpha.f=.5)
            do.call(points,plist)
            plist$y <- ctpolyargs$yhigh
            do.call(points,plist)
          }
          
          #if changing lty then legend needs lty types
          if(subjecti == subjects[1]) { #length(unique(ltyvecnew))>1 && 
            legendtext<-c(legendtext,paste0(kalmanvec[kveci],': ',
              colnames(x[[subiname]][[kalmanvec[kveci]]])[kdimi]))
            legendlty <- c(legendlty,ifelse(plist$type=='p',0,ltyvecnew[dimi]))
            legendpch <- c(legendpch,ifelse(plist$type=='l',NA,pchvecnew[dimi]))
            if(length(subjects) == 1) legendcol = c(legendcol,plist$col) else legendcol=c(legendcol,'black')
          }
        }
      }
    }
    
    if(length(subjects) > 1 && length(unique(colvec))>1) { #include subject color in legend if necessary
      legendtext<-c(legendtext,paste('Subject', subjects))
      legendcol <- c(legendcol,colvec)
      legendlty <-c(legendlty,rep(0,length(subjects)))
      legendpch <-c(legendpch,rep(NA,length(subjects)))
    }
    
    if(legend && length(legendtext)>0){
      legendcontrol$legend<-legendtext
      legendcontrol$col<-legendcol
      legendcontrol$text.col <- legendcol
      legendcontrol$pch <- legendpch
      legendcontrol$lty <- legendlty  
      do.call(graphics::legend,legendcontrol)
    }
  }#end base plots
}


#' Plots Kalman filter output from ctKalman.
#'
#' @param x Output from \code{\link{ctKalman}}. In general it is easier to call 
#' \code{\link{ctKalman}} directly with the \code{plot=TRUE} argument, which calls this function.
#' @param subjects vector of integers denoting which subjects (from 1 to N) to plot predictions for. 
#' @param kalmanvec string vector of names of any elements of the output you wish to plot, 
#' the defaults of 'y' and 'ysmooth' plot the original data, 'y', 
#' and the estimates of the 'true' value of y given all data. Replacing 'y' by 'eta' will 
#' plot latent states instead (though 'eta' alone does not exist) and replacing 'smooth' 
#' with 'upd' or 'prior' respectively plots updated (conditional on all data up to current time point)
#' or prior (conditional on all previous data) estimates.
#' @param errorvec vector of names indicating which kalmanvec elements to plot uncertainty bands for. 
#' 'auto' plots all possible.
#' @param elementNames if NA, all relevant object elements are included -- e.g. if yprior is in the kalmanvec
#' argument, all manifest variables are plotted, and likewise for latent states if etasmooth was specified.
#' Alternatively, a character vector specifying the manifest and latent names to plot explicitly can be specified.
#' @param plot if FALSE, plots are not generated and the ggplot object is simply returned invisibly.
#' @param errormultiply Numeric denoting the multiplication factor of the std deviation of errorvec objects. 
#' Defaults to 1.96, for 95\% intervals.
#' @param polygonsteps Number of steps to use for uncertainty band shading. 
#' @param polygonalpha Numeric for the opacity of the uncertainty region.
#' @param ... not used.
#' @return A ggplot2 object. Side effect -- Generates plots.
#' @method plot ctKalmanDF
#' @export
#' @examples
#' \donttest{
#' ### Get output from ctKalman
#' x<-ctKalman(ctstantestfit,subjects=2,timestep=.01)
#' 
#' ### Plot with plot.ctKalman
#' plot.ctKalmanDF(x, subjects=2)
#' 
#' ###Single step procedure:
#' ctKalman(ctstantestfit,subjects=2,
#'   kalmanvec=c('y','yprior'),
#'   elementNames=c('Y1','Y2'), 
#'   plot=TRUE,timestep=.01)
#' }
plot.ctKalmanDF<-function(x, subjects=1, kalmanvec=c('y','ysmooth'),
  errorvec='auto', errormultiply=1.96,plot=TRUE,elementNames=NA,
  polygonsteps=10,polygonalpha=.1,...){
  
  
  if(!'ctKalmanDF' %in% class(x)) stop('not a ctKalmanDF object')
  
  if(1==99) Time <- Value <- Subject <- Row <- Variable <- Element <- NULL
  setnames(x,'Row', "Variable")
  setnames(x,'value', "Value")
  
  if(any(!is.na(elementNames))) x <- subset(x,Variable %in% elementNames)
  
  klines <- kalmanvec[grep('(prior)|(upd)|(smooth)',kalmanvec)]
  # kpoints<- kalmanvec[-grep('(prior)|(upd)|(smooth)',kalmanvec)]
  colvec=ifelse(length(subjects) > 1, 'Subject', 'Variable')
  ltyvec <- setNames( rep(NA,length(kalmanvec)),kalmanvec)
  ltyvec[kalmanvec %in% klines] = setNames(1:length(klines),klines)
  # if(length(kalmanvec) > length(klines)) ltyvec <- 
  #   c(setNames(rep(0,length(kalmanvec)-length(klines)),kalmanvec[!kalmanvec %in% klines]),
  #     ltyvec)
  shapevec<-ltyvec
  shapevec[shapevec>0] <- NA
  shapevec[is.na(ltyvec)] <- 19
  # 
  d<-subset(x,Element %in% kalmanvec)
  
  g <- ggplot(d,
    aes_string(x='Time',y='Value',colour=colvec,linetype='Element',shape='Element')) +
    scale_linetype_manual(breaks=names(ltyvec),values=ltyvec)+
    scale_shape_manual(breaks=names(shapevec),values=shapevec) +
    # labs(linetype='Element',shape='Element',colour='Element',fill='Element')+
    scale_fill_discrete(guide='none')
  
  if(length(subjects) > 1 && length(unique(subset(x,Element %in% kalmanvec)$Variable)) > 1){
    g <- g+ facet_wrap(vars(Variable),scales = 'free') 
  }
  
  polysteps <- seq(errormultiply,0,-errormultiply/(polygonsteps+1))[c(-polygonsteps+1)]

  for(si in polysteps){
    # alphasum <- alphasum + polygonalpha/polygonsteps
    
    d2 <- subset(d,Element %in% klines)
    d2$sd <- d2$sd *si
    
    if(length(subjects) ==1){
      g<- g+ 
        geom_ribbon(data=d2,aes(ymin=(Value-sd),x=Time,
          ymax=(Value+sd),fill=(Variable)),
          alpha=ifelse(si== polysteps[1],.05,polygonalpha/polygonsteps),
          inherit.aes = FALSE,linetype=0)
      if(si== polysteps[1]) g <- g + 
          geom_line(data=d2,aes(y=(Value-sd),colour=Variable),linetype='dotted',alpha=.4) + 
          geom_line(data=d2,aes(y=(Value+sd),colour=Variable),linetype='dotted',alpha=.4)
    } else {
      g <- g+ 
        geom_ribbon(data=d2,aes(ymin=(Value-sd),x=Time,
          ymax=(Value+sd),fill=(Subject)),inherit.aes = FALSE,
          alpha=polygonalpha/polygonsteps,linetype=0)
      if(si== polysteps[1]) g <- g + 
          geom_line(data=d2,aes(y=(Value-sd),colour=Subject),linetype='dotted',alpha=.7) + 
          geom_line(data=d2,aes(y=(Value+sd),colour=Subject),linetype='dotted',alpha=.7)
    }
    
    g <- g + 
      geom_line()+
      geom_point()+
      theme_minimal()
  }
  if(plot) suppressWarnings(print(g))
  return(invisible(g))
  
}


