#' ctKalman 
#'
#' Outputs predicted, updated, and smoothed estimates of manifest indicators and latent states, 
#' with covariances, for specific subjects from data fit with \code{\link{ctStanFit}}, 
#' based on medians of parameter distribution.
#' 
#' @param fit fit object as generated by \code{\link{ctStanFit}}.
#' @param datalong Optional long format data object as used by \code{\link{ctStanFit}}. 
#' If not included, data from fit will used. 
#' @param timerange Either 'asdata' to just use the observed data range, or a numeric vector of length 2 denoting start and end of time range, 
#' allowing for estimates outside the range of observed data.
#' @param timestep Either 'asdata' to just use the observed data 
#' (which also requires 'asdata' for timerange) or a positive numeric value
#' indicating the time step to use for interpolating values. Lower values give a more accurate / smooth representation,
#' but take a little more time to calculate.
#' @param subjects vector of integers denoting which subjects (from 1 to N) to plot predictions for. 
#' @param plot Logical. If TRUE, plots output instead of returning it. 
#' See \code{\link{ctKalmanPlot}} for the possible arguments.
#' @param ... additional arguments to pass to \code{\link{ctKalmanPlot}}.
#' @return Returns a list containing matrix objects etaprior, etaupd, etasmooth, y, yprior, 
#' yupd, ysmooth, prederror, time, loglik,  with values for each time point in each row. 
#' eta refers to latent states and y to manifest indicators - y itself is thus just 
#' the input data. 
#' Covariance matrices etapriorcov, etaupdcov, etasmoothcov, ypriorcov, yupdcov, ysmoothcov,  
#' are returned in a row * column * time array. 
#' If plot=TRUE, nothing is returned but a plot is generated.
#' @examples
#' #Basic
#' ctKalman(ctstantestfit, timerange=c(0,60), timestep=.5, plot=TRUE)
#' 
#' #Multiple subjects, y and yprior, showing plot arguments
#' ctKalman(ctstantestfit, timerange=c(0,60), timestep=.1, plot=TRUE,
#'   subjects=2:3, 
#'   kalmanvec=c('y','yprior'),
#'   errorvec=c(NA,'ypriorcov'), #'auto' would also have achieved this
#'   ltyvec="auto",
#'   colvec='auto', 
#'   lwdvec='auto', 
#'   subsetindices=2, #Only plotting 2nd dimension of y and yprior
#'   pchvec='auto', typevec='auto',grid=TRUE,legend=TRUE,
#'   plotcontrol=list(xlim=c(0,55),main='Observations and priors'),
#'   polygoncontrol=list(steps=5))
#' @export

ctKalman<-function(fit, datalong=NULL, timerange='asdata', timestep='asdata',
  subjects=1, plot=FALSE,...){

  type=NA
  if(class(fit)=='ctStanFit') type='stan' 
  if(class(fit) =='ctsemFit') type ='omx'
  if(is.na(type)) stop('fit object is not from ctFit or ctStanFit!')
  
  if(type=='stan') n.TDpred <-  fit$ctstanmodel$n.TDpred else n.TDpred <- fit$ctmodelobj$n.TDpred
  if(type=='stan') TDpredNames <- fit$ctstanmodel$TDpredNames else TDpredNames <- fit$ctmodelobj$TDpredNames
  if(type=='stan') manifestNames <- fit$ctstanmodel$manifestNames else manifestNames <- fit$ctmodelobj$manifestNames
  if(type=='stan') latentNames <- fit$ctstanmodel$latentNames else latentNames <- fit$ctmodelobj$latentNames
  
  out<-list()
  if(timerange[1] != 'asdata' & timestep[1] == 'asdata') stop('If timerange is not asdata, a timestep must be specified!')
  
  if(!is.null(datalong)) { #adjust ids and colnames as needed
    datalong <- makeNumericIDs(datalong, fit$ctstanmodel$subjectIDname,fit$ctstanmodel$timeName) #ensure id's are appropriate
    colnames(datalong)[colnames(datalong)==fit$ctstanmodel$subjectIDname] <- 'subject'
    colnames(datalong)[colnames(datalong)==fit$ctstanmodel$timeName] <- 'time'
  }
  
  if(is.null(datalong)) { #get relevant data
    
    if(type=='stan') {
      time<-fit$data$time
      datalong<-cbind(fit$data$subject,time,fit$data$Y)
      datalong[,-1:-2][datalong[,-1:-2] == 99999] <- NA #because stan can't handle NA's
      if(n.TDpred > 0) datalong <- cbind(datalong,fit$data$tdpreds)
      colnames(datalong)<-c('subject','time',
        fit$ctstanmodel$manifestNames,
        fit$ctstanmodel$TDpredNames)
    }
    
    
    if(type=='omx'){
      if(is.null(fit$mxobj$expectation$P0)) { #if not fit with kalman filter then data needs rearranging
        datalong=suppressMessages(ctWideToLong(datawide = fit$mxobj$data$observed[subjects,,drop=FALSE],
          Tpoints=fit$ctmodelobj$Tpoints,
          n.manifest=fit$ctmodelobj$n.manifest,manifestNames = manifestNames,
          n.TDpred=fit$ctmodelobj$n.TDpred,TDpredNames = TDpredNames,
          n.TIpred = fit$ctmodelobj$n.TIpred, TIpredNames = fit$ctmodelobj$TIpredNames))
        datalong <- suppressMessages(ctDeintervalise(datalong = datalong,id = 'id',dT = 'dT'))
      } else {
        datalong=fit$mxobj$data$observed
      datalong <- suppressMessages(ctDeintervalise(datalong = datalong,id = 'id',dT = 'dT1'))
      }
      colnames(datalong)[colnames(datalong) == 'id'] <- 'subject'
    }
    
  }
  
  
  
  if(!all(subjects %in% datalong[,'subject'])) stop('Invalid subjects specified!')
  
  if(type=='stan') derrind<-fit$data$derrind else derrind<-c()
  
  for(subjecti in subjects){
    #setup subjects data, interpolating and extending as necessary
    sdat=datalong[datalong[,'subject'] == subjecti,,drop=FALSE]
    if(timestep != 'asdata' || timerange[1] != 'asdata') {
      if(timerange[1]=='asdata') stimerange <- range(sdat[,'time']) else {
        stimerange <- timerange
        if(timerange[1] > min(sdat[,'time']) || timerange[2] < max(sdat[,'time']) ) stop('Specified timerange must contain all subjects time ranges!')
      }
      snewtimes <- seq(stimerange[1],stimerange[2],timestep)
      snewdat <- array(NA,dim=c(length(snewtimes),dim(sdat)[-1]),dimnames=list(c(),dimnames(sdat)[[2]]))
      snewdat[,'time'] <- snewtimes
      snewdat[,TDpredNames] <- 0
      sdat <- rbind(sdat,snewdat)
      sdat<-sdat[!duplicated(sdat[,'time']),,drop=FALSE]
      sdat <- sdat[order(sdat[,'time']),,drop=FALSE]
      sdat[,c(manifestNames,TDpredNames)] [sdat[,c(manifestNames,TDpredNames)]==99999] <- NA
      sdat[,'subject'] <- subjecti
    }
    
    #get parameter matrices
    if(type=='stan') model<-ctStanContinuousPars(fit, subjects=subjecti)
    if(type=='omx') model <- ctModelFromFit(fit)
    
    #get kalman estimates

    out[[paste('subject',subjecti)]]<-Kalman(kpars=model,
      datalong=sdat,
      manifestNames=manifestNames,
      latentNames=latentNames,
      TDpredNames=TDpredNames,
      idcol='subject',
      timecol='time',
      derrind=derrind)
  }
  
  if(plot) {
    ctKalmanPlot(x=out,subjects=subjects,...)
  } else return(out)
}


#' ctKalmanPlot
#' 
#' Plots Kalman filter output from ctKalman.
#'
#' @param x Output from \code{\link{ctKalman}}. In general it is easier to call 
#' \code{\link{ctKalman}} directly with the \code{plot=TRUE} argument, which calls this function.
#' @param subjects vector of integers denoting which subjects (from 1 to N) to plot predictions for. 
#' @param kalmanvec string vector of names of any elements of the output you wish to plot, 
#' the defaults of 'y' and 'yprior' plot the original data, 'y', 
#' and the prior from the Kalman filter for y. Replacing 'y' by 'eta' will 
#' plot latent variables instead (though 'eta' alone does not exist) and replacing 'prior' 
#' with 'upd' or 'smooth' respectively plotting updated (conditional on all data up to current time point)
#' or smoothed (conditional on all data) estimates.
#' @param errorvec vector of names of covariance elements to use for uncertainty indication 
#' around the kalmanvec items. 'auto' uses the latent covariance when plotting
#' latent states, and total covariance when plotting expectations of observed states.
#' @param errormultiply Numeric denoting the multiplication factor of the std deviation of errorvec objects. 
#' Defaults to 1.96, for 95\% intervals.
#' @param ltyvec vector of line types, varying over dimensions of the kalmanvec object.
#' @param colvec color vector, varying either over subject if multiple subjects, or otherwise over 
#' the dimensions of the kalmanvec object.
#' @param lwdvec vector of line widths, varying over the kalmanvec objects. 
#' @param subsetindices Either NULL, or vector of integers to use for subsetting the (columns) of kalmanvec objects.
#' @param pchvec vector of symbol types, varying over the dimensions of the kalmanvec object.
#' @param typevec vector of plot types, varying over the kalmanvec objects. 'auto' plots lines for
#' any  'prior', 'upd', or 'smooth' objects, and points otherwise.
#' @param grid Logical. Plot a grid?
#' @param add Logical. Create a new plot or update existing plot?
#' @param plotcontrol List of graphical arguments (see \code{\link{par}}), 
#' though lty,col,lwd,x,y, will all be ignored.
#' @param legend Logical, whether to include a legend if plotting.
#' @param legendcontrol List of arguments to the \code{\link{legend}} function.
#' @param polygoncontrol List of arguments to the \code{\link{ctPoly}} function for filling the uncertainty region.
#' @param polygonalpha Numeric for the opacity of the uncertainty region.
#' @return Nothing. Generates plots.
#' @export
#' @examples
#' ### Get output from ctKalman
#' x<-ctKalman(ctstantestfit,subjects=2)
#' 
#' ### Plot with ctKalmanPlot
#' ctKalmanPlot(x, subjects=2)
#' 
#' ###Single step procedure:
#' ctKalman(ctstantestfit,subjects=2,plot=TRUE)
ctKalmanPlot<-function(x, subjects, kalmanvec=c('y','yprior'),
  errorvec='auto', errormultiply=1.96,
  ltyvec="auto",colvec='auto', lwdvec='auto', 
  subsetindices=NULL,pchvec='auto', typevec='auto',grid=FALSE,add=FALSE, 
  plotcontrol=list(ylab='Value',xlab='Time',xaxs='i'),
  polygoncontrol=list(steps=20),polygonalpha=.3,
  legend=TRUE, legendcontrol=list(x='topright',bg='white')){
  

  out<-x
  
  if(length(subjects) > 1 & colvec[1] =='auto') colvec = rainbow(length(subjects),v=.9)
  
  if(lwdvec[1] %in% 'auto') lwdvec=rep(2,length(kalmanvec))
  
  if(is.null(plotcontrol$ylab)) plotcontrol$ylab='Value'
  if(is.null(plotcontrol$xlab)) plotcontrol$xlab='Time'
  
  if(typevec[1] %in% 'auto') typevec=c('p','l')[grepl("prior|upd|smooth|eta",kalmanvec)+1]
  
  if(errorvec[1] %in% 'auto') {
    errorvec=rep(NA,length(kalmanvec))
    errorvec[grepl("prior|upd|smooth",kalmanvec)]<-paste0(
      kalmanvec[grepl("prior|upd|smooth",kalmanvec)],'cov')
  }
  
  if(is.null(plotcontrol$xlim)) plotcontrol$xlim <- range(sapply(out,function(x) x$time))
  

  if(is.null(plotcontrol$ylim)) {
    plotcontrol$ylim <- range(unlist(lapply(out,function(x) { #for every subject
      if(!is.null(x)){
        ret<-c()
        
        for(kveci in 1:length(kalmanvec)){
          est<-x[[kalmanvec[kveci]]][,
            if(is.null(subsetindices)) 1:dim(x[[kalmanvec[kveci]]])[2] else subsetindices]
          
          if(!is.na(errorvec[kveci])) err <- sqrt(abs(c(apply(x[[errorvec[kveci]]][
            (if(is.null(subsetindices)) 1:dim(x[[errorvec[kveci]]])[2] else subsetindices),
            (if(is.null(subsetindices)) 1:dim(x[[errorvec[kveci]]])[2] else subsetindices),
            ,drop=FALSE],3,diag))))
          
      if(is.na(errorvec[kveci])) err <- 0
      
          esthigh <- est + err*errormultiply
          estlow <- est - err*errormultiply
            ret <- c(ret,esthigh,estlow)
        }
        return(ret)}})),na.rm=TRUE)
    if(legend) plotcontrol$ylim[2] <- plotcontrol$ylim[2] + sd(plotcontrol$ylim)/4
  }
  
  
  
  
  legendtext<-c()
  legendcol <- c()
  legendlty<-c()
  legendpch<-c()
  
  #when not set to auto, must define 'new' vector as the user specified vector
  colvecnew <- colvec
  ltyvecnew<-ltyvec
  pchvecnew <- pchvec
  

  for(si in 1:length(subjects)){#subjects
    subjecti = subjects[si]
    subiname=paste('subject',subjecti)
    plist<-plotcontrol
    if(length(subjects) > 1) {
      plist$col = colvec[si] #set colour based on subject if multiple subjects
    }
    
    for(kveci in 1:length(kalmanvec)){ #kalman output types
      kvecdims=1:dim(out[[subiname]][[kalmanvec[kveci]]])[-1]
      if(length(subjects) == 1 & colvec[1] =='auto') colvecnew = rainbow(length(kvecdims),v=.9)
      if(any(subsetindices > max(kvecdims))) stop('subsetindices contains a value greater than relevant dimensions of object in kalmanvec!')
      if(!is.null(subsetindices)) kvecdims=kvecdims[subsetindices]
      if(rl(ltyvec[1]=='auto')) ltyvecnew <- 1:length(kvecdims) else ltyvecnew <- ltyvec
      
      if(rl(pchvec[1] =='auto')) pchvecnew = 1:(length(kvecdims)) else pchvecnew <- pchvec
      
      # if((length(unique(pchvec[typevec!='l']))>1) || 
      #     (length(subjects) == 1 && length(unique(colvec)) > 1)) { #if changing pch, then legend needs to show pch elements along with lty and maybe colour
      #   legendtext<-c(legendtext,paste(kalmanvec[kveci],
      #     colnames(out[[subiname]][[kalmanvec[kveci]]])[kdimi]))
      #   legendlty <- c(legendlty,ifelse(plist$type=='p',0,ltyvec[dimi]))
      #   legendpch <- c(legendpch,ifelse(plist$type=='l',NA,plist$pch))
      #   if(length(subjects) == 1) legendcol = c(legendcol,plist$col) else legendcol=c(legendcol,'black')
      # }
      
      for(dimi in 1:length(kvecdims)){ #dimensions of kalman matrix
        kdimi <- kvecdims[dimi]
        plist$x=out[[subiname]]$time
        plist$y=out[[subiname]][[kalmanvec[kveci]]][,kdimi] 
        plist$lwd=lwdvec[kveci]
        plist$lty=ltyvecnew[dimi] 
        plist$pch=pchvecnew[dimi]
        plist$type=typevec[kveci]
        if(length(subjects)==1) plist$col=colvecnew[dimi]
        
        
        if(subjecti == subjects[1] & kveci==1 && dimi == 1 && !add) {
          do.call(graphics::plot.default,plist) 
          if(grid) {
            grid()
            par(new=TRUE)
            do.call(graphics::plot.default,plist) 
            par(new=FALSE)
          }
        } else do.call(graphics::points.default,plist) 
        
        if(!is.na(errorvec[kveci])){
          if(is.null(out[[subiname]][[errorvec[kveci]]])) stop('Invalid errorvec specified!')
          backwardstimesindex=order(plist$x,decreasing=TRUE)
          
          # if(is.null(polygoncontrol$angle)) 
          # polygoncontrol$angle=stats::runif(1,0,359)
          
          # polygonargs<-polygoncontrol
          # polygonargs$x=c(plist$x,plist$x[backwardstimesindex])
          # polygonargs$y=c(plist$y + errormultiply * sqrt(abs(out[[subiname]][[errorvec[kveci]]][kdimi,kdimi,])), 
          #   (plist$y - errormultiply * sqrt(abs(out[[subiname]][[errorvec[kveci]]][kdimi,kdimi,])))[backwardstimesindex])
          # polygonargs$col=grDevices::adjustcolor(plist$col,alpha.f=polygonalpha)
          # do.call(graphics::polygon,polygonargs)
          
          ctpolyargs<-polygoncontrol
          ctpolyargs$x=c(plist$x)
          ctpolyargs$ylow=c(plist$y - errormultiply * sqrt(abs(out[[subiname]][[errorvec[kveci]]][kdimi,kdimi,])))
          ctpolyargs$y=c(plist$y)
          ctpolyargs$yhigh=c(plist$y + errormultiply * sqrt(abs(out[[subiname]][[errorvec[kveci]]][kdimi,kdimi,])))
          ctpolyargs$col=grDevices::adjustcolor(plist$col,alpha.f=polygonalpha)
          ctpolyargs$col =grDevices::adjustcolor(ctpolyargs$col,alpha.f=max(c(.004,polygonalpha/sqrt(ctpolyargs$steps))))
          do.call(ctPoly,ctpolyargs)
          
          #add quantile lines
          plist$y <- ctpolyargs$ylow
          plist$lwd <- 1
          # plist$col <- grDevices::adjustcolor(plist$col,alpha.f=.5)
          do.call(points,plist)
          plist$y <- ctpolyargs$yhigh
          do.call(points,plist)
        }
        
        #if changing lty then legend needs lty types
        if(subjecti == subjects[1]) { #length(unique(ltyvecnew))>1 && 
          legendtext<-c(legendtext,paste0(kalmanvec[kveci],': ',
            colnames(out[[subiname]][[kalmanvec[kveci]]])[kdimi]))
          legendlty <- c(legendlty,ifelse(plist$type=='p',0,ltyvecnew[dimi]))
          legendpch <- c(legendpch,ifelse(plist$type=='l',NA,pchvecnew[dimi]))
          if(length(subjects) == 1) legendcol = c(legendcol,plist$col) else legendcol=c(legendcol,'black')
        }
      }
    }
  }
  
  if(length(subjects) > 1 && length(unique(colvec))>1) { #include subject color in legend if necessary
    legendtext<-c(legendtext,paste('Subject', subjects))
    legendcol <- c(legendcol,colvec)
    legendlty <-c(legendlty,rep(0,length(subjects)))
    legendpch <-c(legendpch,rep(NA,length(subjects)))
  }
  
  if(legend && length(legendtext)>0){
    legendcontrol$legend<-legendtext
    legendcontrol$col<-legendcol
    legendcontrol$text.col <- legendcol
    legendcontrol$pch <- legendpch
    legendcontrol$lty <- legendlty  
    do.call(graphics::legend,legendcontrol)
  }
}



