import com.vividsolutions.jump.io.DriverProperties;
import com.vividsolutions.jump.io.ShapefileReader;
import com.vividsolutions.jump.feature.FeatureCollection;
import com.vividsolutions.jump.feature.Feature;
import com.vividsolutions.jts.geom.Coordinate;
import com.vividsolutions.jts.geom.MultiPolygon;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.simplify.TopologyPreservingSimplifier;
import com.vividsolutions.jts.operation.distance.DistanceOp;

import java.util.*;
import java.io.File;

import Jama.Matrix;

import javax.swing.*;

/**
 * Compute country distances based on CShapes polygons
 * to be used from the cshapes R package
 * @author Nils Weidmann (nils.weidmann@gmail.com)
 */
public class CountryDistancer {

	private int year;
	private int month;
	private int day;
	private int dimension;
	private double[] ctrcodes;
    private double tolerance; // Douglas-Peucker tolerance
	private String ctrcode;
	private String syear, smonth, sday;
	private String eyear, emonth, eday;
    private ArrayList<Feature> flist;

	private String packageHome;

	public CountryDistancer(String packageHome, int year, int month, int day, double tolerance, boolean useGW) {
		this.packageHome = packageHome;
		this.year = year;
		this.month = month;
		this.day = day;
        this.tolerance = tolerance;
		if (!useGW) {
			ctrcode = "COWCODE";
			syear = "COWSYEAR";
			smonth = "COWSMONTH";
			sday = "COWSDAY";
			eyear = "COWEYEAR";
			emonth = "COWEMONTH";
			eday = "COWEDAY";
		}
		else {
			ctrcode = "GWCODE";
			syear = "GWSYEAR";
			smonth = "GWSMONTH";
			sday = "GWSDAY";
			eyear = "GWEYEAR";
			emonth = "GWEMONTH";
			eday = "GWEDAY";
		}

        flist = getRelevantFeatures();
	}

	private ArrayList<Feature> getRelevantFeatures() {

		// read the shapefile and select the relevant features
		FeatureCollection fc = null;
		try {
			File file = new File(packageHome, "shp/cshapes.shp");
			DriverProperties dp = new DriverProperties(file.getAbsolutePath());
			fc = (new ShapefileReader()).read(dp);
		} catch (Exception e) {
			e.printStackTrace();
		}

		ArrayList<Feature> inlist = new ArrayList<Feature>();
		Iterator it = fc.iterator();
		GregorianCalendar atdate = new GregorianCalendar(year, month, day);
		while (it.hasNext()) {
			Feature f = (Feature) it.next();

            // drop features with NA country codes
            int ctrid = ((Number) f.getAttribute(ctrcode)).intValue();
            if (ctrid<=0) continue;

			int startyear = ((Number) f.getAttribute(syear)).intValue();
			int startmonth = ((Number) f.getAttribute(smonth)).intValue();
			int startday = ((Number) f.getAttribute(sday)).intValue();

			int endyear = ((Number) f.getAttribute(eyear)).intValue();
			int endmonth = ((Number) f.getAttribute(emonth)).intValue();
			int endday = ((Number) f.getAttribute(eday)).intValue();

			GregorianCalendar startdate
					= new GregorianCalendar(startyear, startmonth, startday);
			GregorianCalendar enddate
					= new GregorianCalendar(endyear, endmonth, endday);
			if (startdate.compareTo(atdate) <= 0 && enddate.compareTo(atdate) >= 0) {

                // simplify using Douglas-Peucker
                f.setGeometry(TopologyPreservingSimplifier.simplify(f.getGeometry(), tolerance));
				inlist.add(f);
			}
		}

		// sort feature list by ascending country code
		Collections.sort(inlist, new Comparator<Feature>() {
			public int compare(Feature o1, Feature o2) {
				Integer cow1 = (Integer) o1.getAttribute(ctrcode);
				Integer cow2 = (Integer) o2.getAttribute(ctrcode);
				return cow1.compareTo(cow2);
			}
		});

		// store country codes
		this.ctrcodes = new double[inlist.size()];
		for (int i = 0; i < inlist.size(); i++) {
			Feature feature =  inlist.get(i);
			Integer code = (Integer) feature.getAttribute(ctrcode);
			this.ctrcodes[i] = code;
		}

		// set dimension
		this.dimension = inlist.size();

		return inlist;
	}

	public int getDimension() {
		return dimension;
	}

	public double[] getCtrcodes() {
		return ctrcodes;
	}

	public double[] getMinDistMatrix() {

		Matrix result = new Matrix(flist.size(), flist.size());
		for (int i=0; i<flist.size()-1; i++) {
			Feature f1 = flist.get(i);
			for (int j=i+1; j<flist.size(); j++) {
				Feature f2 = flist.get(j);
				double dist = distanceFeatures(f1, f2);
				result.set(i,j,dist);
				result.set(j,i,dist);
			}
		}
		return result.getColumnPackedCopy();
	}

	public double[] getCentroidDistMatrix() {

		Matrix result = new Matrix(flist.size(), flist.size());
		for (int i=0; i<flist.size()-1; i++) {
			Coordinate cent1 = flist.get(i).getGeometry().getCentroid().getCoordinate();
			for (int j=i+1; j<flist.size(); j++) {
				Coordinate cent2 = flist.get(j).getGeometry().getCentroid().getCoordinate();
				double dist = distanceCoordinates(cent1, cent2);
				result.set(i,j,dist);
				result.set(j,i,dist);
			}
		}
		return result.getColumnPackedCopy();
	}

	public double[] getCapDistMatrix() {

		Matrix result = new Matrix(flist.size(), flist.size());
		for (int i=0; i<flist.size()-1; i++) {

			double caplong1 = (Double) flist.get(i).getAttribute("CAPLONG");
			double caplat1 = (Double) flist.get(i).getAttribute("CAPLAT");
			Coordinate cap1 = new Coordinate(caplong1, caplat1);

			for (int j=i+1; j<flist.size(); j++) {

				double caplong2 = (Double) flist.get(j).getAttribute("CAPLONG");
				double caplat2 = (Double) flist.get(j).getAttribute("CAPLAT");
				Coordinate cap2 = new Coordinate(caplong2, caplat2);

				double dist = distanceCoordinates(cap1, cap2);
				result.set(i,j,dist);
				result.set(j,i,dist);
			}
		}
		return result.getColumnPackedCopy();
	}


    private double distanceFeatures(Feature f1, Feature f2) {

        Coordinate[] coords1 = f1.getGeometry().getCoordinates();
        Coordinate[] coords2 = f2.getGeometry().getCoordinates();


        double minDist = Double.MAX_VALUE;

        for (int i = 0; i < coords1.length; i++) {
            Coordinate c1 = coords1[i];
            for (int j = 0; j < coords2.length; j++) {
                Coordinate c2 = coords2[j];
                double dist = distanceCoordinates(c1, c2);
                if (dist <= 0.1) return 0;
                if (dist<minDist) minDist = dist;
            }
        }

        return minDist;
    }

    private double distanceCoordinates(Coordinate c1, Coordinate c2) {


        final double fromLon = Math.toRadians(c1.x);
        final double fromLat = Math.toRadians(c1.y);
        final double toLon = Math.toRadians(c2.x);
        final double toLat = Math.toRadians(c2.y);
        final double d = 2 * Math.asin(Math.sqrt(Math.pow(Math.sin((fromLat - toLat) / 2), 2) +
                Math.cos(fromLat) * Math.cos(toLat) * Math.pow(Math.sin((fromLon - toLon) / 2), 2)));

        return d * 6378.137;
	}

    // for testing only
	public static void main(String[] args) {

		CountryDistancer cd = new CountryDistancer("/Users/nilsw/Projects/CShapes/R-Forge/pkg/inst", 1980, 1, 1, 0.1, false);

		double[] m = cd.getMinDistMatrix();
        double[] c = cd.getCentroidDistMatrix();
        double[] c2 = cd.getCapDistMatrix();
	}

}
