#' Compute all shortest distance between origin and destination nodes.
#' 
#' @param Graph  An object generated by makegraph(), cpp_simplify() or cpp_contract() function.
#' @param from A vector of one or more vertices from which distances are calculated (origin).
#' @param to A vector of one or more vertices (destination).
#' @param algorithm Character. Only for contracted graph, "mch" for Many to many CH, "phast" for PHAST algorithm
#' @param allcores Logical. If TRUE, all cores are used.
#' @return Matrix of shortest distances.
#' @details If graph is not contracted, get_distance_matrix() recursively perform Dijkstra algorithm for each 'from' nodes.
#' If graph is contracted, the user has the choice between : \itemize{
#'   \item many to many contraction hierarchies (mch) : optimal for square matrix.  
#'   \item PHAST (phast) : outperform mch on rectangular matrix
#' }
#'  See details in package website : \url{https://github.com/vlarmet/cppRouting/blob/master/README.md}
#' @examples 
#' #Data describing edges of the graph
#' edges<-data.frame(from_vertex=c(0,0,1,1,2,2,3,4,4), 
#'                   to_vertex=c(1,3,2,4,4,5,1,3,5), 
#'                   cost=c(9,2,11,3,5,12,4,1,6))
#' #Get all nodes
#' nodes<-unique(c(edges$from_vertex,edges$to_vertex))
#'  
#' #Construct directed and undirected graph 
#' directed_graph<-makegraph(edges,directed=TRUE)
#' non_directed<-makegraph(edges,directed=FALSE)
#' 
#' #Get matrix of distance between all nodes in the two graphs
#' dir_dist<-get_distance_matrix(Graph=directed_graph, from=nodes, to=nodes, allcores=FALSE)
#' non_dir_dist<-get_distance_matrix(Graph=non_directed, from=nodes, to=nodes, allcores=FALSE)
#' print(dir_dist)
#' print(non_dir_dist)

get_distance_matrix<-function(Graph,from,to,algorithm="phast",allcores=FALSE){
  if (any(is.na(from))) stop("NAs are not allowed in origin/destination nodes")
  if (any(is.na(to))) stop("NAs are not allowed in origin/destination nodes")
  from<-as.character(from)
  to<-as.character(to)
  allnodes<-c(from,to)
  if (sum(allnodes %in% Graph$dict$ref)<length(allnodes)) stop("Some nodes are not in the graph")
  
  from_id<-Graph$dict$id[match(from,Graph$dict$ref)]
  to_id<-Graph$dict$id[match(to,Graph$dict$ref)]
  
  if (length(Graph)==4){
    if (allcores==TRUE){
      
      
      
      if (length(to)< length(from)){
        res<-Dijkstra_mat_par(to_id,from_id,Graph$data[,2],Graph$data[,1],Graph$data[,3],Graph$nbnode)
        
      }
      else {
        res<-Dijkstra_mat_par(from_id,to_id,Graph$data[,1],Graph$data[,2],Graph$data[,3],Graph$nbnode)
        
      }
      
      
      
    }
    else {
      if (length(to)< length(from))  res<-Dijkstra_mat(Graph$data[,2],Graph$data[,1],Graph$data[,3],Graph$nbnode,to_id,from_id)
      else res<-Dijkstra_mat(Graph$data[,1],Graph$data[,2],Graph$data[,3],Graph$nbnode,from_id,to_id)
    }
  }
  
  if (length(Graph)==5){
    
    if (algorithm=="mch"){
      if (allcores==TRUE){
        
        if (length(to)< length(from)){
          res<-par_Bidir_mat2(to_id,from_id,Graph$data[,2],Graph$data[,1],Graph$data[,3],Graph$nbnode,Graph$rank)
          
        }
        else {
          res<-par_Bidir_mat2(from_id,to_id,Graph$data[,1],Graph$data[,2],Graph$data[,3],Graph$nbnode,Graph$rank)
          
        }
        
        
      }
      else {
        if (length(to)< length(from))  res<-Bidir_mat3(to_id,from_id,Graph$data[,2],Graph$data[,1],Graph$data[,3],Graph$nbnode,Graph$rank)
        else res<-Bidir_mat3(from_id,to_id,Graph$data[,1],Graph$data[,2],Graph$data[,3],Graph$nbnode,Graph$rank)
      }
    }
    else{
      
      invrank<-(Graph$nbnode)-Graph$rank
      
      if (allcores==TRUE){
        
        if (length(to)< length(from)){
          res<-Phast_par(invrank[to_id+1],
                         invrank[from_id+1],
                         invrank[Graph$data$to+1],
                         invrank[Graph$data$from+1],
                         Graph$data[,3],
                         Graph$nbnode)
          
        }
        else {
          res<-Phast_par(invrank[from_id+1],
                         invrank[to_id+1],
                         invrank[Graph$data$from+1],
                         invrank[Graph$data$to+1],
                         Graph$data[,3],
                         Graph$nbnode)
          
        }
        
        
      }
      else {
        if (length(to)< length(from))  res<-Phast3(invrank[to_id+1],invrank[from_id+1],invrank[Graph$data$to+1],invrank[Graph$data$from+1],Graph$data[,3],Graph$nbnode)
        else res<-Phast3(invrank[from_id+1],invrank[to_id+1],invrank[Graph$data$from+1],invrank[Graph$data$to+1],Graph$data[,3],Graph$nbnode)
      }
    }

    
  }
  

  
  
  
  
  
  
  if (length(to)< length(from)) res<-t(res)
  
  rownames(res)<-from
  colnames(res)<-to
  return(res)
}
