% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/STAR_Bayesian.R
\name{genMCMC_star}
\alias{genMCMC_star}
\title{Generalized MCMC Algorithm for STAR}
\usage{
genMCMC_star(
  y,
  sample_params,
  init_params,
  transformation = "np",
  y_max = Inf,
  nsave = 5000,
  nburn = 5000,
  nskip = 0,
  save_y_hat = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{y}{\code{n x 1} vector of observed counts}

\item{sample_params}{a function that inputs data \code{y} and a named list \code{params} containing
\enumerate{
\item \code{mu}: the \code{n x 1} vector of conditional means (fitted values)
\item \code{sigma}: the conditional standard deviation
\item \code{coefficients}: a named list of parameters that determine \code{mu}
}
and outputs an updated list \code{params} of samples from the full conditional posterior
distribution of \code{coefficients} and \code{sigma} (and updates \code{mu})}

\item{init_params}{an initializing function that inputs data \code{y}
and initializes the named list \code{params} of \code{mu}, \code{sigma}, and \code{coefficients}}

\item{transformation}{transformation to use for the latent data; must be one of
\itemize{
\item "identity" (identity transformation)
\item "log" (log transformation)
\item "sqrt" (square root transformation)
\item "np" (nonparametric transformation estimated from empirical CDF)
\item "pois" (transformation for moment-matched marginal Poisson CDF)
\item "neg-bin" (transformation for moment-matched marginal Negative Binomial CDF)
\item "box-cox" (box-cox transformation with learned parameter)
}}

\item{y_max}{a fixed and known upper bound for all observations; default is \code{Inf}}

\item{nsave}{number of MCMC iterations to save}

\item{nburn}{number of MCMC iterations to discard}

\item{nskip}{number of MCMC iterations to skip between saving iterations,
i.e., save every (nskip + 1)th draw}

\item{save_y_hat}{logical; if TRUE, compute and save the posterior draws of
the expected counts, E(y), which may be slow to compute}

\item{verbose}{logical; if TRUE, print time remaining}
}
\value{
a list with at least the following elements:
\itemize{
\item \code{post.pred}: draws from the posterior predictive distribution of \code{y}
\item \code{post.sigma}: draws from the posterior distribution of \code{sigma}
\item \code{post.log.like.point}: draws of the log-likelihood for each of the \code{n} observations
\item \code{WAIC}: Widely-Applicable/Watanabe-Akaike Information Criterion
\item \code{p_waic}: Effective number of parameters based on WAIC
\item \code{post.lambda}: draws from the posterior distribution of \code{lambda}
(NULL unless \code{transformation='box-cox'})
\item \code{fitted.values}: the posterior mean of the conditional expectation of the counts \code{y}
(\code{NULL} if \code{save_y_hat=FALSE})
\item \code{post.fitted.values}: posterior draws of the conditional mean of the counts \code{y}
(\code{NULL} if \code{save_y_hat=FALSE})
}
If the coefficients list from \code{init_params} and \code{sample_params} contains a named element \code{beta},
e.g. for linear regression, then the function output contains
\itemize{
\item \code{coefficients}: the posterior mean of the beta coefficients
\item \code{post.beta}: draws from the posterior distribution of \code{beta}
\item \code{post.othercoefs}: draws from the posterior distribution of any other sampled coefficients, e.g. variance terms
}

If no \code{beta} exists in the parameter coefficients, then the output list just contains
\itemize{
\item \code{coefficients}: the posterior mean of all coefficients
\item \code{post.beta}: draws from the posterior distribution of all coefficients
}

Additionally, if \code{init_params} and \code{sample_params} have output \code{mu_test}, then the sampler will output
\code{post.predtest}, which contains draws from the posterior predictive distribution at test points.
}
\description{
Run the MCMC algorithm for STAR given
\enumerate{
\item a function to initialize model parameters; and
\item a function to sample (i.e., update) model parameters.
}
The transformation can be known (e.g., log or sqrt) or unknown
(Box-Cox or estimated nonparametrically) for greater flexibility.
}
\details{
STAR defines a count-valued probability model by
(1) specifying a Gaussian model for continuous *latent* data and
(2) connecting the latent data to the observed data via a
*transformation and rounding* operation.

Posterior and predictive inference is obtained via a Gibbs sampler
that combines (i) a latent data augmentation step (like in probit regression)
and (ii) an existing sampler for a continuous data model.

There are several options for the transformation. First, the transformation
can belong to the *Box-Cox* family, which includes the known transformations
'identity', 'log', and 'sqrt', as well as a version in which the Box-Cox parameter
is inferred within the MCMC sampler ('box-cox'). Second, the transformation
can be estimated (before model fitting) using the empirical distribution of the
data \code{y}. Options in this case include the empirical cumulative
distribution function (CDF), which is fully nonparametric ('np'), or the parametric
alternatives based on Poisson ('pois') or Negative-Binomial ('neg-bin')
distributions. For the parametric distributions, the parameters of the distribution
are estimated using moments (means and variances) of \code{y}.
}
\examples{
# Simulate data with count-valued response y:
sim_dat = simulate_nb_lm(n = 100, p = 5)
y = sim_dat$y; X = sim_dat$X

# STAR: log-transformation:
fit_log = genMCMC_star(y = y,
                         sample_params = function(y, params) sample_lm_gprior(y, X, params),
                         init_params = function(y) init_lm_gprior(y, X),
                         transformation = 'log')
# Posterior mean of each coefficient:
coef(fit_log)

# WAIC for STAR-log:
fit_log$WAIC

# MCMC diagnostics:
plot(as.ts(fit_log$post.beta[,1:3]))

# Posterior predictive check:
hist(apply(fit_log$post.pred, 1,
           function(x) mean(x==0)), main = 'Proportion of Zeros', xlab='');
abline(v = mean(y==0), lwd=4, col ='blue')

}
