% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/markov.R
\name{markov}
\alias{markov}
\alias{rate_markov}
\alias{rank_markov}
\title{Markov method}
\usage{
rate_markov(cr_data, ..., fill = list(), stoch_modify = teleport(0.15),
  weights = 1, force_nonneg_h2h = TRUE)

rank_markov(cr_data, ..., fill = list(), stoch_modify = teleport(0.15),
  weights = 1, force_nonneg_h2h = TRUE, keep_rating = FALSE,
  ties = c("average", "first", "last", "random", "max", "min"),
  round_digits = 7)
}
\arguments{
\item{cr_data}{Competition results in format ready for
\link[comperes:as_longcr]{as_longcr()}.}

\item{...}{Name-value pairs of Head-to-Head functions (see
\link[comperes:h2h_long]{h2h_long()}).}

\item{fill}{A named list that for each Head-to-Head function supplies a
single value to use instead of NA for missing pairs (see
\link[comperes:h2h_long]{h2h_long()}).}

\item{stoch_modify}{A single function to modify column-stochastic matrix or a
list of them (see \link[=stoch-modifiers]{Stochastic matrix modifiers}).}

\item{weights}{Weights for different stochastic matrices.}

\item{force_nonneg_h2h}{Whether to force nonnegative values in Head-to-Head
matrix.}

\item{keep_rating}{Whether to keep rating column in ranking output.}

\item{ties}{Value for \code{ties} in \code{\link[=round_rank]{round_rank()}}.}

\item{round_digits}{Value for \code{round_digits} in \code{\link[=round_rank]{round_rank()}}.}
}
\value{
\code{rate_markov()} returns a \link[tibble:tibble]{tibble} with columns
\code{player} (player identifier) and \code{rating_markov} (Markov
\link[=rating-ranking]{rating}). The sum of all ratings should be equal to 1.
\strong{Bigger value indicates better player performance}.

\code{rank_markov} returns a \code{tibble} with columns \code{player}, \code{rating_markov} (if
\code{keep_rating = TRUE}) and \code{ranking_markov} (Markov \link[=rating-ranking]{ranking}
computed with \code{\link[=round_rank]{round_rank()}}).
}
\description{
Functions to compute \link[=rating-ranking]{rating} and \link[=rating-ranking]{ranking}
using Markov method.
}
\details{
Markov ratings are based on players 'voting' for other players being
better. Algorithm is as follows:
\enumerate{
\item 'Voting' is done with Head-to-Head values supplied in \code{...} (see
\link[comperes:h2h_mat]{h2h_mat()} for technical details and section \strong{Design
of Head-to-Head values} for design details). Take special care of
Head-to-Head values for self plays (when player1 equals player2). \strong{Note}
that Head-to-Head values should be non-negative. Use \code{force_nonneg_h2h = TRUE} to force that by subtracting minimum Head-to-Head value (in case some
Head-to-Head value is strictly negative).
\item Head-to-Head values are transformed into matrix which is normalized to be
column-stochastic (sum of every column should be equal to 1) Markov matrix
\emph{S}. \strong{Note} that all missing values are converted into 0. To specify other
value use \code{fill} argument.
\item \emph{S} is modified with \code{stoch_modify} to deal with possible problems behind
\emph{S}, such as reducibility and rows with all 0.
\item Stationary vector is computed based on \emph{S} as probability transition
matrix of Markov chain process (transition probabilities from state \strong{i} are
elements from column \strong{i}). The result is declared as Markov ratings.
}

Considering common values and structure of stochastic matrices one can
naturally combine different 'votings' in one stochastic matrix:
\enumerate{
\item \link[comperes:h2h_long]{Long format} of Head-to-Head values is computed
using \code{...} (which in this case should be several expressions for
Head-to-Head functions).
\item Each set of Head-to-Head values is transformed into matrix which is
normalized to column-stochastic.
\item Each stochastic matrix is modified with respective modifier which is
stored in \code{stoch_modify} (which can be a list of functions).
\item The resulting stochastic matrix is computed as weighted average of
modified stochastic matrices.
}

For Head-to-Head functions in \code{...} (considered as list) and argument
\code{stoch_modify} general R recycling rule is applied. If \code{stoch_modify} is a
function it is transformed to list with one function.

\code{weights} is recycled to the maximum length of two mentioned recycled
elements and then is normalized to sum to 1.

Ratings are computed based only on games between players of interest (see
Players).
}
\section{Design of Head-to-Head values}{


Head-to-Head values in these functions are assumed to follow the property
which can be \emph{equivalently} described in two ways:
\itemize{
\item In terms of \link[comperes:h2h_mat]{matrix format}: \strong{the more Head-to-Head
value in row \emph{i} and column \emph{j} the better player from row \emph{i} performed than
player from column \emph{j}}.
\item In terms of \link[comperes:h2h_long]{long format}: \strong{the more Head-to-Head
value the better player1 performed than player2}.
}

This design is chosen because in most competitions the goal is to score
\strong{more points} and not less. Also it allows for more smooth use of
\link[comperes:h2h_funs]{h2h_funs} from \code{comperes} package.
}

\section{Players}{


\code{comperank} offers a possibility to handle certain set of players. It is done
by having \code{player} column (in \link[comperes:as_longcr]{longcr} format) as factor
with levels specifying all players of interest. In case of factor the result
is returned only for players from its levels. Otherwise - for all present
players.
}

\examples{
rate_markov(
  cr_data = ncaa2005,
  # player2 "votes" for player1 if player1 won
  comperes::num_wins(score1, score2, half_for_draw = FALSE),
  stoch_modify = vote_equal
)

rank_markov(
  cr_data = ncaa2005,
  comperes::num_wins(score1, score2, half_for_draw = FALSE),
  stoch_modify = vote_equal
)

rank_markov(
  cr_data = ncaa2005,
  comperes::num_wins(score1, score2, half_for_draw = FALSE),
  stoch_modify = vote_equal,
  keep_rating = TRUE
)

# Combine multiple stochastic matrices and
# use inappropriate `fill` which misrepresents reality
rate_markov(
  cr_data = ncaa2005[-(1:2), ],
  win = comperes::num_wins(score1, score2, half_for_draw = FALSE),
  # player2 "votes" for player1 proportionally to the amount player1 scored
  # more in direct confrontations
  score_diff = max(mean(score1 - score2), 0),
  fill = list(win = 0.5, score_diff = 10),
  stoch_modify = list(vote_equal, teleport(0.15)),
  weights = c(0.8, 0.2)
)

}
\references{
\href{https://en.wikipedia.org/wiki/Markov_chain}{Wikipedia
page} for Markov chain.
}
