% Generated by roxygen2 (4.0.1): do not edit by hand
\name{species-response}
\alias{Beta}
\alias{Gaussian}
\alias{species-response}
\title{Species response models for coenocline simulation}
\usage{
Gaussian(x, y = NULL, px, py = NULL, corr = 0)

Beta(x, y = NULL, px, py = NULL)
}
\arguments{
\item{x}{numeric; locations of observations on the primary gradient.}

\item{y}{numeric; locations of observations on the secondary gradient. Can be missing is only a single gradient is required.}

\item{px}{a list of named elements, each of which is a vector of numeric parameter values for the species response on the primary gradient \code{x}. See Details for further information on the required parameters.}

\item{py}{a list of named elements, each of which is a vector of numeric parameter values for the species response on the secondary gradient \code{y}. See Details for further information on the required parameters.}

\item{corr}{numeric; the correlation between gradients \code{x} and \code{y}. Only applies to \code{Gaussian()}.}
}
\value{
A numeric vector of species "abundances" of length equal to \code{length(x)}.
}
\description{
Parameterise species response curves along one or two gradients according to a Gaussian or generalised beta response model.
}
\details{
\code{Gaussian()} and \code{Beta()} return values from appropriately parameterised Gaussian or generalised beta response models respectively. Parameters for the primary (\code{x}) and secondary (\code{y}) gradients are supplied as lists via arguments \code{px} and \code{py}. Parameters are supplied in the form of vectors, one per parameter. These vectors must be supplied to named components in the respective lists. The names of the components must match the parameters of the required response model.

For \code{Gaussian()} the following named components must be supplied:
\describe{
  \item{opt}{the species optima}
  \item{tol}{the species tolerances}
  \item{h}{the heights of the response curves at the optima. This parameter should only be supplied to \code{px}; in the case of simulations along two gradients, the height of the response curve applies to both gradients and is the hieght of a bivariate Guassian distribution at the bivariate optima.}
}

For \code{Beta()} the following named components must be supplied:
\describe{
  \item{A0}{The heights of the species response curves at their modes. Like the parameter \code{h} for the Gaussian response, this parameter should only be passed via \code{px}; in the case of simulations along two gradients, the height of the response curve applies to both gradients and is the height of a bivariate generalised beta distribution at the bivariate mode.}
  \item{m}{the locations on the gradient of the modal abundance (the species optima)}
  \item{r}{the ranges of occurrence of species on the gradient}
  \item{alpha}{a shape parameter. With \code{gamma}, \code{alpha} informs the shape of the response curve and control the skewness and kurtosis of the curve. Only positive values are allowed, which lead to unimodal response curves. If \code{alpha} is equal to \code{gamma}, the species response curve is symmetric, otherwise an asymmetric curve is generated.}
  \item{gamma}{a shape parameter. With \code{alpha}, \code{gamma} informs the shape of the response curve and control the skewness and kurtosis of the curve. Only positive values are allowed, which lead to unimodal response curves. If \code{gamma} is equal to \code{alpha}, the species response curve is symmetric, otherwise an asymmetric curve is generated.}
}

See the examples here and in \code{\link{coenocline}} for details on how to set up calls to these species response functions.
}
\examples{
# A simple example with a single species
x <- seq(from = 4, to = 6, length = 100)
px <- list(opt = 4.5, tol = 0.25, h = 20)
G <- Gaussian(x, px = px)
head(G)
length(G)

# A more complex example with 6 species, which needs the parameters
# repeating for each gradient location:

# Recreate Fig. 2 of Minchin (1987)
# Parameters for each of 6 six species
A0 <- c(5,4,7,5,9,8) * 10
m <- c(25,85,10,60,45,60)
r <- c(3,3,4,4,6,5) * 10
alpha <- c(0.1,1,2,4,1.5,1)
gamma <- c(0.1,1,2,4,0.5,4)
# Gradient locations
x <- 1:100

# expand parameter set
pars <- expand(x, m = m, A0 = A0, r = r, alpha = alpha,
gamma = gamma)
head(pars)

xvec <- pars[, "x"]
px <- as.list(data.frame(pars[, -1]))
spprc <- Beta(xvec, px = px)
matplot(matrix(spprc, ncol = 6), ## 6 species
        type = "l", lty = "solid")

# Bivariate beta, single species
xx <- 1:100
yy <- 1:100
xy <- expand.grid(x = xx, y = yy)
parx <- expand(xy[, "x"], A0 = 50, m = 60, r = 40, alpha = 4, gamma = 4)
pary <- expand(xy[, "y"], m = 60, r = 40, alpha = 4, gamma = 4)

x <- parx[,1]
px <- as.list(as.list(data.frame(parx[, -1])))
y <- pary[,1]
py <- as.list(as.list(data.frame(pary[, -1])))

spprc <- Beta(x, y, px = px, py = py)
persp(xx, yy, matrix(spprc, ncol = length(xx)))
}
\author{
Gavin L. Simpson
}
\keyword{datagen}

