% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/remap.R
\name{remap}
\alias{remap}
\title{Grid interpolation.}
\usage{
remap(
  var,
  infile1,
  infile2,
  outfile,
  method = "nearest",
  dxy_factor = 1,
  nc34 = 4,
  overwrite = FALSE,
  verbose = FALSE,
  nc1 = NULL,
  nc2 = NULL
)
}
\arguments{
\item{var}{Name of NetCDF variable (character).}

\item{infile1}{Filename of first input NetCDF file. This may include the
directory (character). The data of infile1 are interpolated.}

\item{infile2}{Filename of second input file. This may include the directory
(character). The grid information of infile2 are the target grid for the
interpolation. This File may also be an ASCII-File containing the grid
information.}

\item{outfile}{Filename of output NetCDF file. This may include the directory
(character).}

\item{method}{Method used for remapping (character).
Options are "bilinear" for bilinear interpolation,
"conservative" for conservative remapping (only for regular grids, respectively)
and "nearest" for nearest-neighbor interpolation.
Default is "nearest".}

\item{dxy_factor}{In case of nearest neighbor all grid points with distance > (dxy * dxy_factor)
are set to NA (numeric). Default is 1.}

\item{nc34}{NetCDF version of output file. If \code{nc34 = 3} the output file will be
in NetCDFv3 format (numeric). Default output is NetCDFv4.}

\item{overwrite}{logical; should existing output file be overwritten?}

\item{verbose}{logical; if TRUE, progress messages are shown}

\item{nc1}{Alternatively to \code{infile1} you can specify the input as an
object of class \code{ncdf4} (as returned from \code{ncdf4::nc_open}).}

\item{nc2}{Alternatively to \code{infile2} you can specify the input as an
object of class \code{ncdf4} (as returned from \code{ncdf4::nc_open}).}
}
\value{
A NetCDF file including the interpolated data of infile1 on the grid of
infile2 is written.
}
\description{
The function interpolates the data of infile1 to the grid of infile2. From
infile2 only the grid information is used.
By default, a nearest neighbor interpolation provided by
\code{\link[FNN:get.knn]{get.knnx}} is used. For interpolation between
regular grids a simple bilinear interpolation as provided by
\code{\link[fields:interp.surface]{interp.surface.grid}} as well as a conservative
remapping as provided by \code{\link[rainfarmr:remapcon]{remapcon}} can be chosen.
}
\examples{
## Create an example NetCDF file with a similar structure as used by CM
## SAF. The file is created with the ncdf4 package.  Alternatively
## example data can be freely downloaded here: <https://wui.cmsaf.eu/>

library(ncdf4)

## create some (non-realistic) example data

lon <- seq(5, 15, 0.5)
lat <- seq(45, 55, 0.5)
lon2 <- seq(5, 15, 1)
lat2 <- seq(45, 55, 1)
time <- c(as.Date("2000-01-01"), as.Date("2001-02-01"))
origin <- as.Date("1983-01-01 00:00:00")
time <- as.numeric(difftime(time, origin, units = "hour"))
data1 <- array(250:350, dim = c(21, 21, 1))
data2 <- array(230:320, dim = c(21, 21, 1))

## create two example NetCDF files

x <- ncdim_def(name = "lon", units = "degrees_east", vals = lon)
y <- ncdim_def(name = "lat", units = "degrees_north", vals = lat)
t <- ncdim_def(name = "time", units = "hours since 1983-01-01 00:00:00",
 vals = time[1], unlim = TRUE)
var1 <- ncvar_def("SIS", "W m-2", list(x, y, t), -1, prec = "short")
vars <- list(var1)
ncnew <- nc_create(file.path(tempdir(),"CMSAF_example_file_1.nc"), vars)
ncvar_put(ncnew, var1, data1)
ncatt_put(ncnew, "lon", "standard_name", "longitude", prec = "text")
ncatt_put(ncnew, "lat", "standard_name", "latitude", prec = "text")
nc_close(ncnew)

x <- ncdim_def(name = "lon", units = "degrees_east", vals = lon2)
y <- ncdim_def(name = "lat", units = "degrees_north", vals = lat2)
t <- ncdim_def(name = "time", units = "hours since 1983-01-01 00:00:00",
 vals = time[1], unlim = TRUE)
ncnew <- nc_create(file.path(tempdir(),"CMSAF_example_file_2.nc"), vars)
ncvar_put(ncnew, var1, data2)
ncatt_put(ncnew, "lon", "standard_name", "longitude", prec = "text")
ncatt_put(ncnew, "lat", "standard_name", "latitude", prec = "text")
nc_close(ncnew)

## Interpolate the fields of both example CM SAF NetCDF file 1 to the
## coarser grid of file 2 and write the result into one output file.
remap(var = "SIS", infile1 = file.path(tempdir(),"CMSAF_example_file_1.nc"), 
 infile2 = file.path(tempdir(),"CMSAF_example_file_2.nc"),
 outfile = file.path(tempdir(),"CMSAF_example_file_remap.nc"))

unlink(c(file.path(tempdir(),"CMSAF_example_file_1.nc"), 
 file.path(tempdir(),"CMSAF_example_file_2.nc"),
 file.path(tempdir(),"CMSAF_example_file_remap.nc")))
}
\seealso{
Other data manipulation functions: 
\code{\link{acsaf_box_mergetime}()},
\code{\link{add_grid_info}()},
\code{\link{box_mergetime}()},
\code{\link{cmsaf.transform.coordinate.system}()},
\code{\link{levbox_mergetime}()},
\code{\link{map_regular}()}
}
\concept{data manipulation functions}
