#' Generate the GRN panel of the shiny app
#' @description These are the UI and server components of the GRN panel of the 
#' shiny app. It is generated by including 'GRN' in the panels.default argument
#' of \code{\link{generateShinyApp}}.
#' @inheritParams DEpanel
#' @param seed the random seed to be set when running GRN inference, to ensure
#' reproducibility of GENIE3 outputs
#' @return The UI and Server components of the shiny module, that can be used
#' within the UI and Server definitions of a shiny app.
#' @name GRNpanel
NULL

#' @rdname GRNpanel
#' @export
GRNpanelUI <- function(id){
  ns <- NS(id)
  
  tabPanel(
    'GRN inference',
    sidebarLayout(
      
      sidebarPanel(
        selectInput(ns("targetGenes"), "Target genes:", multiple = TRUE, choices = character(0)),
        actionButton(ns('goGRN'), label = 'Start GRN inference'),
        
        numericInput(ns("plotConnections"), "Connections to plot:", 5, 0, 100),
        textInput(ns('plotFileName'), 'File name for plot download', value ='GRNplot.html'),
        downloadButton(ns('download'), 'Download Plot'),
      ),
      
      mainPanel(
        visNetwork::visNetworkOutput(ns('plot')),
      )
    )
  )
}

#' @rdname GRNpanel
#' @export
GRNpanelServer <- function(id, expression.matrix, anno, seed = 13){
  
  stopifnot({
    is.reactive(expression.matrix)
    !is.reactive(anno)
  })
  
  moduleServer(id, function(input, output, session){
    
    updateSelectizeInput(session, "targetGenes", choices = anno$NAME, server = TRUE)
    
    GRNresults <- reactive({
      target.genes <- anno$ENSEMBL[match(input[["targetGenes"]], anno$NAME)]
      set.seed(seed)
      GENIE3::GENIE3(expression.matrix(), targets = target.genes)
    }) %>%
      bindEvent(input[["goGRN"]])
    
    GRNplot <- reactive({
      weightMat <- GRNresults()
      edges <- GENIE3::getLinkList(weightMat, input[["plotConnections"]]) %>%
        dplyr::rename(from = .data$regulatoryGene, to = .data$targetGene, value = .data$weight) %>%
        dplyr::mutate(from = as.character(.data$from), to = as.character(.data$to))
      nodes <- tibble::tibble(
        id = c(colnames(weightMat), edges$from),
        label = anno$NAME[match(id, anno$ENSEMBL)],
        group = c(rep("target", ncol(weightMat)), rep("regulator", nrow(edges)))
      ) %>%
        dplyr::distinct(id, .keep_all = TRUE)
      
      visNetwork::visNetwork(nodes, edges)
    })
    
    output[['plot']] <- visNetwork::renderVisNetwork(GRNplot())
    
    output[['download']] <- downloadHandler(
      filename = function() {input[['plotFileName']]},
      content = function(file) {
        GRNplot() %>% visNetwork::visSave(file)
      }
    )
    
  })
}