% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gbRecord-class.R
\name{gbRecord}
\alias{gbRecord}
\title{Read a GenBank/GenPept or Embl format file.}
\usage{
gbRecord(rcd, progress = FALSE)
}
\arguments{
\item{rcd}{A vector of paths to GenBank/Embl format records,
an \code{\link[reutils]{efetch}} object containing GenBank record(s), or
a \code{textConnection} to a character vector that can be parsed as
a Genbank or Embl record.}

\item{progress}{Print a nice progress bar if parsing multiple Genbank records.
(This will not work if you process the records in parallel.)}
}
\value{
An instance of the \code{\linkS4class{gbRecord}} or
\code{\linkS4class{gbRecordList}} classes.
}
\description{
Import data from GenBank/GenPept, Embl, or IMGT/HLA flat files into R,
represented as an instance of the \code{\linkS4class{gbRecord}} or
\code{\linkS4class{gbRecordList}} classes.
}
\details{
For a sample GenBank record see
\url{https://www.ncbi.nlm.nih.gov/Sitemap/samplerecord.html},
for a detailed description of the GenBank feature table format see
\url{https://www.ncbi.nlm.nih.gov/collab/FT/}.

For a description of the EMBL flat file format see
\url{ftp://ftp.ebi.ac.uk/pub/databases/embl/doc/usrman.txt}.

For a description of the format and conventions of IMGT/HLA flat files
see \url{https://www.ebi.ac.uk/ipd/imgt/hla/docs/manual.html}.
}
\note{
The \code{\linkS4class{gbRecord}} class is modelled after the Genbank flat
file format. Both Embl and IMGT/HLA files do not fit this model perfectly,
so some pretty arbitrary choices were made to make the data from these files
fitr the model.
}
\examples{
\dontrun{
### import from file
gbk_file <- system.file("extdata", "marine_metagenome.gb", package = "biofiles")
x <- gbRecord(gbk_file)
}

load(system.file("extdata", "marine_metagenome.rda", package = "biofiles"))
getHeader(x)
getFeatures(x)

### quickly extract features as GRanges
ranges(x["CDS"], include = c("product", "note", "protein_id"))

## Directly subset features
x[[1]]

### import directly from NCBI
\dontrun{
x <- gbRecord(reutils::efetch("139189709", "protein", rettype = "gp", retmode = "text"))
x
}

## import a file containing multiple GenBank records as a
## gbRecordList. With many short records it pays of to
## run the parsing in parallel
\dontrun{
gss_file <- system.file("extdata", "gss.gb", package = "biofiles")
library(doParallel)
registerDoParallel(cores = 4)
gss <- gbRecord(gss_file)
gss
}
}
\seealso{
\code{\link{genomeRecordFromNCBI}}
}
