% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/allocate.R
\name{allocate}
\alias{allocate}
\title{Allocate soil properties within various classification systems.}
\usage{
allocate(
  ...,
  to = c("FAO Salt Severity", "FAO Black Soil", "ST Diagnostic Features"),
  droplevels = FALSE
)
}
\arguments{
\item{...}{arguments to specific allocation functions, see details and examples}

\item{to}{character specifying the classification scheme: FAO Salt Severity, FAO Black Soil (see details for the required \code{...})}

\item{droplevels}{logical indicating whether to drop unused levels in factors. This is useful when the results have a large number of unused classes, which can waste space in tables and figures.}
}
\value{
A vector or \code{data.frame} object.
}
\description{
Generic function to allocate soil properties to different classification schemes.
}
\details{
This function is intended to allocate a set of soil properties to an established soil classification scheme, such as Salt Severity or Black Soil. Allocation is semantically different from classification. While classification is the 'act' of developing a grouping scheme, allocation is the assignment or identification of measurements to a established class (Powell, 2008).
\subsection{Usage Details}{

Each classification scheme (\code{to} argument) uses a different set of arguments.
\itemize{
\item \verb{FAO Salt Severity}
\itemize{
\item \strong{EC:} electrical conductivity column name, dS/m
\item \strong{pH:} pH column name, saturated paste extract
\item \strong{ESP:} exchangeable sodium percentage column name, percent
}
\item \verb{FAO Black Soils}
\itemize{
\item \strong{object:} a \code{data.frame} or \code{SoilProfileCollection}
\item \strong{pedonid:} pedon ID column name, required when \code{object} is a \code{data.frame}
\item \strong{hztop:} horizon top depth column name, required when \code{object} is a \code{data.frame}
\item \strong{hzbot:} horizon bottom depth column name, required when \code{object} is a \code{data.frame}
\item \strong{OC}: organic carbon column name, percent
\item \strong{m_chroma:} moist Munsell chroma column name
\item \strong{m_value:} moist Munsell value column name
\item \strong{d_value:} dry Munsell value column name
\item \strong{CEC:} cation exchange capacity column name (NH4OAc at pH 7), units of cmol(+)/kg soil
\item \strong{BS:} base saturation column name (NH4OAc at pH 7), percent
\item \strong{tropical:} logical, data are associated with "tropical soils"
}
\item \verb{ST Diagnostic Features}
\itemize{
\item \strong{object:} a \code{data.frame} or \code{SoilProfileCollection}
\item \strong{pedonid:} pedon ID column name, required when \code{object} is a \code{data.frame}
\item \strong{hzname:} horizon name column, required when \code{object} is a \code{data.frame}
\item \strong{hztop:} horizon top depth column name, required when \code{object} is a \code{data.frame}
\item \strong{hzbot:} horizon bottom depth column name, required when \code{object} is a \code{data.frame}
\item \strong{texcl:} soil texture class (USDA) column name
\item \strong{rupresblkcem:} rupture resistance column name
\item \strong{m_value:} moist Munsell value column name
\item \strong{m_chroma:} moist Munsell chroma column name
\item \strong{d_value:} dry Munsell value column name
\item \strong{BS:} base saturation column name (method ??), percent
\item \strong{OC}: organic carbon column name, percent
\item \strong{n_value:} ??
\item \strong{featkind:} ??
}
}
}
}
\note{
The results returned by \code{allocate(to = "ST Diagnostic Features")} currently return a limited set of diagnostic features that are easily defined. Also, the logic implemented for some features does not include all the criteria defined in the Keys to Soil Taxonomy.
}
\examples{

# Salt Severity
test <- expand.grid(
  EC  = sort(sapply(c(0, 0.75, 2, 4, 8, 15, 30), function(x) x + c(0, -0.05, 0.05))),
  pH  = c(8.1, 8.2, 8.3, 8.4, 8.5, 8.6),
  ESP = sort(sapply(c(0, 15, 30, 50, 70, 100), function(x) x + c(0, 0.1, -0.1)))
)
test$ss      <- with(test, allocate(EC = EC, pH = pH, ESP = ESP, to = "FAO Salt Severity"))
table(test$ss)

# Black Soil Category 1 (BS1)
test <- expand.grid(
  dept = seq(0, 50, 10),
  OC   = sort(sapply(c(0, 0.6, 1.2, 20, 40), function(x) x + c(0, -0.05, 0.05))),
  chroma_moist  = 2:4,
  value_moist   = 2:4,
  value_dry     = 4:6,
  thickness     = 24:26,
  CEC           = 24:26,
  BS            = 49:51,
  tropical      = c(TRUE, FALSE)
)
test$pedon_id <- rep(1:21870, each = 6)
test$depb     <- test$dept + 10

bs1 <- allocate(test, pedonid = "pedon_id", hztop = "dept", hzbot = "depb", 
                OC = "OC", m_chroma = "chroma_moist", m_value = "value_moist", 
                d_value = "value_dry", CEC = "CEC", BS = "BS", 
                to = "FAO Black Soil"
)

table(BS1 = bs1$BS1, BS2 = bs1$BS2)


# SoilProfileCollection interface

data(sp3)
depths(sp3) <- id ~ top + bottom
hzdesgnname(sp3) <- 'name'

# fake base saturation
horizons(sp3)$bs <- 75

plotSPC(sp3)

allocate(
  sp3, 
  to = 'FAO Black Soil', 
  OC = 'tc', 
  m_chroma = 'chroma', 
  m_value = 'value', 
  d_value = 'value',
  CEC = 'cec',
  BS = 'bs'
)

# make a copy and edit horizon values
x <- sp3
x$value <- 2
x$chroma <- 2
x$cec <- 26
x$tc <- 2

x$soil_color <- munsell2rgb(x$hue, x$value, x$chroma)

plotSPC(x)

allocate(
  x, 
  to = 'FAO Black Soil', 
  OC = 'tc', 
  m_chroma = 'chroma', 
  m_value = 'value', 
  d_value = 'value',
  CEC = 'cec',
  BS = 'bs'
)


# Soil Taxonomy Diagnostic Features
data(sp1)
sp1$texcl = gsub("gr|grv|cbv", "", sp1$texture)
df <- allocate(object = sp1, pedonid = "id", hzname = "name", 
               hzdept = "top", hzdepb = "bottom", texcl = "texcl", 
               to = "ST Diagnostic Features"
)
aggregate(featdept ~ id, data = df, summary)

}
\references{
Abrol, I., Yadav, J. & Massoud, F. 1988. \href{https://www.fao.org/3/x5871e/x5871e00.htm}{Salt-affected soils and their management}. No. Bulletin 39. Rome, FAO Soils.

FAO. 2006. \href{https://www.fao.org/publications/card/en/c/903943c7-f56a-521a-8d32-459e7e0cdae9/}{Guidelines for soil description}. Rome, Food and Agriculture Organization of the United Nations.

FAO. 2020. DEFINITION | What is a black soil? (online). (Cited 28 December 2020). http://www.fao.org/global-soil-partnership/intergovernmental-technical-panel-soils/gsoc17-implementation/internationalnetworkblacksoils/more-on-black-soils/definition-what-is-a-black-soil/es/

Powell, B., 2008. Classifying soil and land, in: McKenzie, N.J., Grundy, M.J., Webster, R., Ringrose-Voase, A.J. (Eds.), Guidelines for Survey Soil and Land Resources, Australian Soil and Land Survey Handbook Series. CSIRO, Melbourne, p. 572.

Richards, L.A. 1954. \href{https://www.ars.usda.gov/ARSUserFiles/20360500/hb60_pdf/hb60complete.pdf}{Diagnosis and Improvement of Saline and Alkali Soils}. U. S. Government Printing Office. 166 pp.

Soil Survey Staff, 2014. Keys to Soil Taxonomy, 12th ed. USDA-Natural Resources Conservation Service, Washington, D.C.
}
