#' Mesh topology and visualization for spatial mesh structures.
#'
#' The  anglr package gives direct access to generic 3D tools and provides a
#' full suite of mesh-creation and 3D plotting functions. By extending the rgl
#' package conversion and visualization functions for the `mesh3d` class a wide
#' variety of complex spatial data can be brought into 3D scenes. These tools
#' allow for spatial raster, polygons, and lines that are common in GIS
#' contexts to be converted into mesh forms with high flexibility and the
#' ability to integrate disparate data types. Vector and raster data can be
#' seamlessly combined as meshes, and surfaces can be set to have material
#' properties based on data values or with image textures. Textures and other
#' data combinations use projection transformations to map between coordinate
#' systems, and objects can be easily visualized in an interactive scene at any
#' stage.
#'
#' The 'anglr' package show-cases extended features for *geo*-spatial data by
#' extending and supporting the data models of the silicate package. Any kind
#' of spatial data is intended to be supported, not just the geographic ones:
#' \itemize{
#'  \item coordinates beyond X and Y, or longitude and latitude
#'  \item storing attributes on vertices, primitives, branches (parts), or objects
#'  \item topology and geometry are distinguishable and not conflated
#'  \item spatial data can be represented as a graph of spatial primitives
#'  \item polygons as true surfaces, not just path structures with a 2D-only region-filling rule
#'  \item TBD higher dimensional primitives are possible
#'  \item TBD n-dimensional rasters with curvilinear coordinates, and the discrete-continuous distinction
#' }
#'
#' @section Licensing:
#'
#' The anglr package is released with license CC BY-NC-SA 4.0 to match the one
#' dependency RTriangle. Please note and respect the license of the RTriangle
#' package used by the [DEL()] or [DEL0()] functions in anglr, and invoked
#' within 3D plot methods. These return high-quality constrained Delaunay
#' triangulations of polygonal regions, with the ability to control mesh
#' characteristics including maximum triangle area, minimum internal angle, and
#' conformance to the Delaunay criterion. If you are interested in
#' a less restrictive license for high-quality meshing in R please
#' get involved with [the laridae package](https://github.com/hypertidy/laridae)
#' which aims to provide access to [CGAL](https://www.cgal.org/).
#'
#' @section Creation:
#' \tabular{ll}{
#'  \code{\link[rgl]{as.mesh3d}} \tab coercion function to convert most spatial data to mesh forms \cr
#'   \code{\link[silicate]{SC}}  \tab and other silicate models are all supported, including the structural forms SC0, TRI0, PATH0 \cr
#'   \code{\link[sp]{Spatial}} \tab most spatial types can be used directly including raster and sf \cr
#'   \code{\link{DEL}} \tab create a mostly-Delaunay shape-preserving constrained triangulation  \cr
#'  }
#'
#' @section III. Plotting:
#'
#' As much as possible plotting will represent the true nature of the data given.
#'
#' \tabular{ll}{
#' \code{\link{mesh_plot}} \tab plot in 2D, including curvilinear reprojections with rasters \cr
#' \code{\link[rgl]{plot3d}} \tab and related 3D plot functions in rgl can be used directly on most input types \cr
#'  \code{\link{globe}} \tab convert X,Y planar or angular to 3D on the surface of a globe, based on the data in longitude-latitude form \cr
#'  \code{\link{plot3d.SC}} \tab plot 1D topology in 3D geometry space \cr
#'  \code{\link{plot3d.TRI}} \tab plot 2D topology in 3D geometry space (DEL or TRI) \cr
#' }
#'
#' @section Options and technicalities:
#'
#' There is an option set for the maximum number of triangles that can be
#' generated by the [DEL()] or [DEL0()] models when using the `max_area`
#' argument. Inspect the limit with `getOption("anglr.max.triangles")` or set a
#' new limit with `options(anglr.max.triangles = <new limit>)`.
#'
#' In terms of the [RTriangle::triangulate()] function the `max_area` argument
#' controls and masks the `a` argument for RTriangle. It's possible to pass in
#' values for `q`, `Y`, `j`, `D`, `S`, `V`, and `Q` - but we don't recommend
#' experimenting with these unless you know what they are for. There is a
#' coarse check for a limit on the number of triangles that can be created but
#' general caution is advised when experimenting.
#' @name anglr-package
#' @docType package
NULL

#' Coordinate reprojection
#'
#' The [reproj()] function is imported from the [reproj::reproj()] package
#' and re-exported.

#' @importFrom reproj reproj
#' @export reproj
#' @name reproj
NULL

#' silicate models
#'
#' The anglr functions [DEL()], [DEL0()] and [QUAD()] extend the models
#' of the silicate package. In particular `DEL()` and `DEL0()` are high
#' quality Delaunay-constrained triangulation meshers analogous to the
#' ear-cutting algorithms used by [silicate::TRI()] and [silicate::TRI0()].
#'
#' All models in silicate are imported by anglr and re-exported.
#' @name silicate-models
#' @param x any data format understood by the model
#' @param ... unused
#' @aliases PATH PATH0 SC SC0  TRI0 ARC
#' @seealso [DEL()] [DEL0()] [QUAD()] [silicate::PATH()] [silicate::TRI()] [silicate::ARC()] [silicate::SC()] [silicate::PATH0()] [silicate::TRI0()] [silicate::SC0()]
#' @importFrom silicate PATH
#' @export PATH
#' @importFrom silicate PATH0
#' @export PATH0
#' @importFrom silicate SC
#' @export SC
#' @importFrom silicate SC0
#' @export SC0
#' @importFrom silicate TRI
#' @export TRI
#' @importFrom silicate TRI0
#' @export TRI0
#' @importFrom silicate ARC
#' @export ARC
NULL


# @export
# @name reproj
# @examples
# plot3d(reproj(QUAD(raster::crop(gebco, raster::extent(100, 120, -50, 0))), "+proj=geocent +datum=WGS84"))
# rgl::aspect3d(1, 1, 1)

# reproj.QUAD <- function(x, target = NULL, ..., source = NULL) {
#   vv <- get_vertex(x)
#   if (!"z_" %in% names(vv)) {
#     vv$z_ <-
#   }
#   v <- reproj::reproj(vv, source = get_proj(x), target = target)
#   x$vertex <- tibble::tibble(x_ = v[,1L], y_ = v[,2L], z_ = v[,3])
#   meta <- x$meta
#   row <- x$meta[1, ]
#   row$proj <- target
#   row$ctime <- Sys.time()
#   x
# }



#' sf data frame zoo.
#'
#' Each kind of geometry in an sf data frame, in a list.
#' @name sf_data_zoo
#' @docType data
NULL

#' Cadastre and Contour
#'
#' Coincident polygon cadastre layer and line contour layer.
#'
#' These two sf layers are [cad_tas] a sf polygons layer of a small region
#' of cadastral parcels, and [cont_tas] a sf lines layer of the same
#' region with elevation contours of the underlying topography.
#'
#' These layers are fused together in an [in-progress example](https://github.com/hypertidy/anglr/issues/16).
#'
#' `cont_tas` has an elevation value for each line in `cont_tas[["ELEVATION"]]`.
#'
#' These data sets are derived from the LIST Cadastral Parcels and LIST
#' Contours 5m from [theLIST](https://www.thelist.tas.gov.au/app/content/home)
#' Copyright State of Tasmania. These data are distributed under the
#' [Creative Commons Attribution 3.0 Australia License](http://creativecommons.org/licenses/by/3.0/au/).
#' @name cad_tas
#' @aliases cont_tas
#' @docType data
#' @examples
#' plot3d(cont_tas)
#' \donttest{
#' plot3d(copy_down(silicate::SC0(cont_tas), "ELEVATION"))
#' auto_3d()
#' }
NULL

#' simple world
#'
#' A simple polygon map of world sovereign countries, a modified copy of
#' the rnaturalearth counties110 (see data-raw/simpleworld.R for details).
#' @name simpleworld
#' @docType data
#' @examples
#' DEL(simpleworld[1:10, ])
NULL


#' world elevation raster
#'
#' A simple raster map of world topography, elevation relative to sea level in
#' metres. Source data is Gebco 2014, converted to a much reduced 1 degree
#' resolution global map.
#'
#'  Data downloaded from GEBCO 2014 (0.0083 degrees = 30sec arcmin resolution)
#'  and set at resolution 1 degrees.
#'  [GEBCO 2014](https://www.gebco.net/data_and_products/gridded_bathymetry_data/).
#' @name gebco
#' @docType data
#' @examples
#' data("gebco", package = "anglr")
#' library(silicate)
#' laea <- "+proj=laea +lon_0=147 +lat_0=-42"
#' longlat <- "+proj=longlat +datum=WGS84"
#' x <- SC(simpleworld) %>% copy_down(gebco + 500)
#' plot3d(x); rgl::aspect3d(1, 1, 0.07)
NULL
