
# WeibullR.plotly

<!-- badges: start -->

[![CRAN
status](https://www.r-pkg.org/badges/version/WeibullR.plotly)](https://CRAN.R-project.org/package=WeibullR.plotly)
![](http://cranlogs.r-pkg.org/badges/grand-total/WeibullR.plotly)
![](http://cranlogs.r-pkg.org/badges/WeibullR.plotly) [![Lifecycle:
experimental](https://img.shields.io/badge/lifecycle-experimental-orange.svg)](https://lifecycle.r-lib.org/articles/stages.html#experimental)
[![DOI](https://zenodo.org/badge/639144870.svg)](https://zenodo.org/doi/10.5281/zenodo.8003549)
[![Codecov test
coverage](https://codecov.io/gh/paulgovan/WeibullR.plotly/branch/main/graph/badge.svg)](https://app.codecov.io/gh/paulgovan/WeibullR.plotly?branch=main)
<!-- badges: end -->

Build interactive Weibull Probability Plots with WeibullR, an R package
for Weibull analysis, and plotly, an interactive web-based graphing
library.

# Getting Started

To install WeibullR.plotly in R:

``` r
install.packages('WeibullR.plotly')
```

Or install the development version:

``` r
devtools::install_github('paulgovan/weibullr.plotly')
```

# Basic Examples

To build a probability plot, first fit a `wblr` object using the
`WeibullR` package and then use `plotly_wblr` to build the plot.

``` r
library(WeibullR)
library(WeibullR.plotly)
failures<-c(30, 49, 82, 90, 96)
obj<-wblr.conf(wblr.fit(wblr(failures)))
plotly_wblr(obj)
```

<div class="plotly html-widget html-fill-item" id="htmlwidget-3f2bcf7c0888e22c1b8a" style="width:672px;height:480px;"></div>
<script type="application/json" data-for="htmlwidget-3f2bcf7c0888e22c1b8a">{"x":{"data":[{"x":[30,49,82,90,96],"y":[0.13862943611198913,0.37660097008368176,0.69314718055994529,1.1589670285784068,2.0444649242511774],"mode":"markers","marker":{"color":"black","line":{"color":"rgba(31,119,180,1)"}},"showlegend":false,"error_x":{"color":"black"},"text":["Probability: (30, 0.129)","Probability: (49, 0.314)","Probability: (82, 0.5)","Probability: (90, 0.686)","Probability: (96, 0.871)"],"hoverinfo":["text","text","text","text","text"],"type":"scatter","error_y":{"color":"rgba(31,119,180,1)"},"line":{"color":"rgba(31,119,180,1)"},"xaxis":"x","yaxis":"y","frame":null},{"x":[3.3476444889053316,4.1386544904659992,5.1230632058306984,6.3084335988170928,7.7706633137958763,9.5057136132010456,9.5909763094405847,11.89790714859539,14.727791534966785,18.274004762807259,20.127075262226978,22.532390887214973,27.858909708064054,27.919097398048848,34.447600144466271,42.730204671575827,52.752171716890352,65.916558067095607,66.758385942808019,79.791423254195919,82.672903015180779,103.47021564326833,128.38426320141841,158.03205715622124,194.60888383680705],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789821359],"mode":"markers+lines","marker":{"color":"transparent","line":{"color":"rgba(255,127,14,1)"}},"showlegend":false,"error_x":{"color":"black"},"text":["Fit: 3.348, 0.001)","Fit: 4.139, 0.002)","Fit: 5.123, 0.003)","Fit: 6.308, 0.004)","Fit: 7.771, 0.006)","Fit: 9.506, 0.01)","Fit: 9.591, 0.01)","Fit: 11.898, 0.016)","Fit: 14.728, 0.026)","Fit: 18.274, 0.041)","Fit: 20.127, 0.05)","Fit: 22.532, 0.064)","Fit: 27.859, 0.1)","Fit: 27.919, 0.1)","Fit: 34.448, 0.154)","Fit: 42.73, 0.234)","Fit: 52.752, 0.345)","Fit: 65.917, 0.491)","Fit: 66.758, 0.5)","Fit: 79.791, 0.632)","Fit: 82.673, 0.658)","Fit: 103.47, 0.819)","Fit: 128.384, 0.934)","Fit: 158.032, 0.987)","Fit: 194.609, 0.999)"],"hoverinfo":["text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text"],"type":"scatter","line":{"color":"black"},"error_y":{"color":"rgba(255,127,14,1)"},"xaxis":"x","yaxis":"y","frame":null},{"x":[0.13639852423108145,0.20701332051156007,0.3160198815679453,0.48255937224054235,0.73507557762428255,1.1021515930153192,1.1204242217002591,1.7108671130489261,2.6173390117712332,3.9325589084545247,4.780060246789092,5.9774480930822769,8.9748104229371659,9.0075871913720373,13.470330501030441,20.070939328909116,29.256493925468291,41.634653047234409,42.4561983890774,52.897079506434096,54.99919883660791,67.745703227766924,79.966597456923722,91.568289087883855,103.61398734803257],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789821359],"mode":"markers+lines","marker":{"color":"transparent","line":{"color":"rgba(44,160,44,1)"}},"showlegend":false,"error_x":{"color":"black"},"text":["Upper: 0.136, 0.001)","Upper: 0.207, 0.002)","Upper: 0.316, 0.003)","Upper: 0.483, 0.004)","Upper: 0.735, 0.006)","Upper: 1.102, 0.01)","Upper: 1.12, 0.01)","Upper: 1.711, 0.016)","Upper: 2.617, 0.026)","Upper: 3.933, 0.041)","Upper: 4.78, 0.05)","Upper: 5.977, 0.064)","Upper: 8.975, 0.1)","Upper: 9.008, 0.1)","Upper: 13.47, 0.154)","Upper: 20.071, 0.234)","Upper: 29.256, 0.345)","Upper: 41.635, 0.491)","Upper: 42.456, 0.5)","Upper: 52.897, 0.632)","Upper: 54.999, 0.658)","Upper: 67.746, 0.819)","Upper: 79.967, 0.934)","Upper: 91.568, 0.987)","Upper: 103.614, 0.999)"],"hoverinfo":["text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text"],"type":"scatter","line":{"color":"transparent"},"error_y":{"color":"rgba(44,160,44,1)"},"xaxis":"x","yaxis":"y","frame":null},{"fillcolor":"rgba(0,0,0,0.2)","x":[21.559811517518412,23.598559689929029,25.954900083700188,28.560496862156214,31.425282768948406,34.497949430082706,34.648573099255074,38.217671665986018,42.068891151093574,46.509280846183849,48.817611685729609,51.701231080788617,57.341243695706815,57.382181538013725,63.641193281889549,71.483185017234106,80.969930832952187,93.961938162854281,94.810657432110546,109.87749079962673,113.61724779216236,147.72445178455152,207.83173079474457,307.37251983605194,459.92541124351732],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789821359],"mode":"markers+lines","marker":{"color":"transparent","line":{"color":"rgba(214,39,40,1)"}},"showlegend":false,"error_x":{"color":"black"},"text":["Lower: 21.56, 0.001)","Lower: 23.599, 0.002)","Lower: 25.955, 0.003)","Lower: 28.56, 0.004)","Lower: 31.425, 0.006)","Lower: 34.498, 0.01)","Lower: 34.649, 0.01)","Lower: 38.218, 0.016)","Lower: 42.069, 0.026)","Lower: 46.509, 0.041)","Lower: 48.818, 0.05)","Lower: 51.701, 0.064)","Lower: 57.341, 0.1)","Lower: 57.382, 0.1)","Lower: 63.641, 0.154)","Lower: 71.483, 0.234)","Lower: 80.97, 0.345)","Lower: 93.962, 0.491)","Lower: 94.811, 0.5)","Lower: 109.877, 0.632)","Lower: 113.617, 0.658)","Lower: 147.724, 0.819)","Lower: 207.832, 0.934)","Lower: 307.373, 0.987)","Lower: 459.925, 0.999)"],"hoverinfo":["text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text"],"type":"scatter","fill":"tonexty","line":{"color":"transparent"},"error_y":{"color":"rgba(214,39,40,1)"},"xaxis":"x","yaxis":"y","frame":null},{"mode":"markers","marker":{"color":"black","line":{"color":"rgba(148,103,189,1)"}},"showlegend":false,"text":"Suspension: ","hoverinfo":"text","type":"scatter","error_y":{"color":"rgba(148,103,189,1)"},"error_x":{"color":"rgba(148,103,189,1)"},"line":{"color":"rgba(148,103,189,1)"},"xaxis":"x2","yaxis":"y2","frame":null},{"domain":{"x":[0.77500000000000002,1]},"header":{"values":["Parameter","Value"],"align":["center","center"],"line":{"width":1,"color":"black"},"fill":{"color":["grey","grey"]},"font":{"family":"Arial","color":"white"}},"cells":{"values":[["Ranks","n","Failures","Intervals","Suspensions","Distribution","Method","Beta","Eta","R^2","CI","Type"],["median","5","5","0","0","weibull","rr-xony","2.168","79.791","0.931","0.9","pivotal-rr"]],"align":["center","center"],"line":{"color":"black","width":1},"font":{"family":"Arial","color":"black"}},"type":"table","frame":null}],"layout":{"xaxis":{"domain":[0,0.75],"automargin":true,"type":"log","title":"Time to Failure","showline":true,"mirror":"ticks","showgrid":true,"gridcolor":"lightgray","range":[0.52473933083118884,2.2891626617968717],"anchor":"y"},"xaxis2":{"domain":[0,0.75],"automargin":true,"type":"log","title":"","zeroline":false,"showline":true,"mirror":"ticks","showticklabels":false,"showgrid":false,"range":[0.52473933083118884,2.2891626617968717],"anchor":"y2"},"yaxis2":{"domain":[0.90000000000000002,1],"automargin":true,"title":"","zeroline":false,"showline":true,"mirror":"ticks","showticklabels":false,"showgrid":false,"anchor":"x2"},"yaxis":{"domain":[0,0.875],"automargin":true,"type":"log","title":"Unreliability","showline":true,"mirror":"ticks","size":["function (x, ...) ","UseMethod(\"text\")"],"showgrid":true,"gridcolor":"lightgray","range":[-3,1.0611856930352348],"tickvals":[1.0000000111269894e-08,1.0000000505039327e-07,1.0000005000059672e-06,1.000005000029529e-05,0.00010000500033327544,0.00050012504168215146,0.001000500333583622,0.0020020026706729687,0.0050125418235441935,0.010050335853501506,0.020202707317519469,0.051293294387550481,0.10536051565782635,0.22314355131420976,0.69314718055994529,2.3025850929940459,4.6051701859880909,11.512925464963677],"ticktext":[9.9999999999999995e-07,1.0000000000000001e-05,0.0001,0.001,0.01,0.050000000000000003,0.10000000000000001,0.20000000000000001,0.5,1,2,5,10,20,50,90,99,99.998999999999995],"anchor":"x"},"annotations":[],"shapes":[],"images":[],"margin":{"b":40,"l":60,"t":25,"r":10},"title":"Probability Plot","hovermode":"closest","showlegend":false,"xaxis3":{"domain":[0.77500000000000002,1]},"yaxis3":{"domain":[0,0.84999999999999998]}},"attrs":{"e659863a034":{"x":[30,49,82,90,96],"y":[0.13862943611198913,0.37660097008368176,0.69314718055994529,1.1589670285784068,2.0444649242511774],"mode":"markers","marker":{"color":"black"},"showlegend":false,"error_x":{"array":{},"color":"black"},"text":{},"hoverinfo":"text","name":"","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter"},"e659863a034.1":{"x":[3.3476444889053316,4.1386544904659992,5.1230632058306984,6.3084335988170928,7.7706633137958763,9.5057136132010456,9.5909763094405847,11.89790714859539,14.727791534966785,18.274004762807259,20.127075262226978,22.532390887214973,27.858909708064054,27.919097398048848,34.447600144466271,42.730204671575827,52.752171716890352,65.916558067095607,66.758385942808019,79.791423254195919,82.672903015180779,103.47021564326833,128.38426320141841,158.03205715622124,194.60888383680705],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789821359],"mode":"markers+lines","marker":{"color":"transparent"},"showlegend":false,"error_x":{"color":"black"},"text":{},"hoverinfo":"text","name":"","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter","line":{"color":"black"},"inherit":true},"e659863a034.2":{"x":[0.13639852423108145,0.20701332051156007,0.3160198815679453,0.48255937224054235,0.73507557762428255,1.1021515930153192,1.1204242217002591,1.7108671130489261,2.6173390117712332,3.9325589084545247,4.780060246789092,5.9774480930822769,8.9748104229371659,9.0075871913720373,13.470330501030441,20.070939328909116,29.256493925468291,41.634653047234409,42.4561983890774,52.897079506434096,54.99919883660791,67.745703227766924,79.966597456923722,91.568289087883855,103.61398734803257],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789821359],"mode":"markers+lines","marker":{"color":"transparent"},"showlegend":false,"error_x":{"color":"black"},"text":{},"hoverinfo":"text","name":"","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter","line":{"color":"transparent"},"inherit":true},"e659863a034.3":{"x":[21.559811517518412,23.598559689929029,25.954900083700188,28.560496862156214,31.425282768948406,34.497949430082706,34.648573099255074,38.217671665986018,42.068891151093574,46.509280846183849,48.817611685729609,51.701231080788617,57.341243695706815,57.382181538013725,63.641193281889549,71.483185017234106,80.969930832952187,93.961938162854281,94.810657432110546,109.87749079962673,113.61724779216236,147.72445178455152,207.83173079474457,307.37251983605194,459.92541124351732],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789821359],"mode":"markers+lines","marker":{"color":"transparent"},"showlegend":false,"error_x":{"color":"black"},"text":{},"hoverinfo":"text","name":"","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter","fill":"tonexty","fillcolor":"rgba(0,0,0,0.2)","line":{"color":"transparent"},"inherit":true},"e65944aaba39":{"x":null,"y":null,"mode":"markers","marker":{"color":"black"},"showlegend":false,"text":{},"hoverinfo":"text","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter"},"e65924a81767":{"domain":{"x":[0.77500000000000002,1]},"header":{"values":["Parameter","Value"],"align":["center","center"],"line":{"width":1,"color":"black"},"fill":{"color":["grey","grey"]},"font":{"family":"Arial","color":"white"}},"cells":{"values":[["Ranks","n","Failures","Intervals","Suspensions","Distribution","Method","Beta","Eta","R^2","CI","Type"],["median","5","5","0","0","weibull","rr-xony","2.168","79.791","0.931","0.9","pivotal-rr"]],"align":["center","center"],"line":{"color":"black","width":1},"font":{"family":"Arial","color":"black"}},"alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"table"}},"source":"A","config":{"modeBarButtonsToAdd":["hoverclosest","hovercompare"],"showSendToCloud":false},"highlight":{"on":"plotly_click","persistent":false,"dynamic":false,"selectize":false,"opacityDim":0.20000000000000001,"selected":{"opacity":1},"debounce":0},"subplot":true,"shinyEvents":["plotly_hover","plotly_click","plotly_selected","plotly_relayout","plotly_brushed","plotly_brushing","plotly_clickannotation","plotly_doubleclick","plotly_deselect","plotly_afterplot","plotly_sunburstclick"],"base_url":"https://plot.ly"},"evals":[],"jsHooks":[]}</script>

To build a contour plot, use the `plotly_contour` function. Note that
contour plots are only available where `method.fit='mle'` and
`method.conf='lrb'`.

``` r
obj<-wblr.conf(wblr.fit(wblr(failures), method.fit = 'mle'), method.conf = 'lrb')
plotly_contour(obj)
```

<div class="plotly html-widget html-fill-item" id="htmlwidget-04ae17d4b5e908ac2b8f" style="width:672px;height:480px;"></div>
<script type="application/json" data-for="htmlwidget-04ae17d4b5e908ac2b8f">{"x":{"visdat":{"e6591c77bacc":["function () ","plotlyVisDat"]},"cur_data":"e6591c77bacc","attrs":{"e6591c77bacc":{"x":[77.811999999999998,75.719999999999999,73.602000000000004,71.465000000000003,69.316000000000003,67.174000000000007,65.066999999999993,63.043999999999997,61.183,59.603000000000002,58.445,57.820999999999998,57.710999999999999,57.960000000000001,58.389000000000003,58.878,59.362000000000002,59.82,60.241,60.627000000000002,60.979999999999997,61.305,61.603999999999999,61.881999999999998,62.142000000000003,62.384999999999998,62.616,62.835999999999999,63.045999999999999,63.249000000000002,63.444000000000003,63.634999999999998,63.822000000000003,64.006,64.186999999999998,64.369,64.549999999999997,64.731999999999999,64.917000000000002,65.103999999999999,65.296000000000006,65.492999999999995,65.697999999999993,65.909999999999997,66.134,66.369,66.619,66.887,67.176000000000002,67.491,67.837999999999994,68.224000000000004,68.658000000000001,69.155000000000001,69.731999999999999,70.414000000000001,71.239000000000004,72.266999999999996,73.590999999999994,75.364999999999995,77.811999999999998,80.988,84.192999999999998,86.730000000000004,88.640000000000001,90.122,91.313999999999993,92.302000000000007,93.141000000000005,93.867999999999995,94.506,95.075000000000003,95.587999999999994,96.055999999999997,96.484999999999999,96.884,97.256,97.605999999999995,97.936000000000007,98.251000000000005,98.552999999999997,98.841999999999999,99.122,99.394000000000005,99.659000000000006,99.918999999999997,100.173,100.42400000000001,100.67100000000001,100.91500000000001,101.157,101.39700000000001,101.634,101.866,102.09399999999999,102.31399999999999,102.523,102.71599999999999,102.887,103.023,103.11,103.131,103.06100000000001,102.871,102.532,102.014,101.301,100.383,99.269000000000005,97.974999999999994,96.527000000000001,94.951999999999998,93.272999999999996,91.510999999999996,89.683000000000007,87.801000000000002,85.873999999999995,83.908000000000001,81.908000000000001,79.875,77.811999999999998],"y":[1.569,1.5589999999999999,1.554,1.5529999999999999,1.5569999999999999,1.5680000000000001,1.587,1.6180000000000001,1.665,1.7310000000000001,1.821,1.9350000000000001,2.0630000000000002,2.1930000000000001,2.3140000000000001,2.4220000000000002,2.5179999999999998,2.6019999999999999,2.6760000000000002,2.742,2.8010000000000002,2.855,2.9039999999999999,2.9500000000000002,2.992,3.0310000000000001,3.0680000000000001,3.1040000000000001,3.137,3.1699999999999999,3.2010000000000001,3.2320000000000002,3.262,3.2909999999999999,3.3199999999999998,3.3490000000000002,3.3780000000000001,3.4079999999999999,3.4369999999999998,3.468,3.4980000000000002,3.5299999999999998,3.5630000000000002,3.5979999999999999,3.6339999999999999,3.6720000000000002,3.7130000000000001,3.7559999999999998,3.8029999999999999,3.855,3.9119999999999999,3.9750000000000001,4.0469999999999997,4.1289999999999996,4.2240000000000002,4.3369999999999997,4.4729999999999999,4.641,4.8529999999999998,5.1219999999999999,5.4400000000000004,5.6950000000000003,5.6989999999999998,5.5179999999999998,5.2969999999999997,5.0910000000000002,4.9109999999999996,4.7539999999999996,4.6180000000000003,4.4980000000000002,4.391,4.2949999999999999,4.2080000000000002,4.1280000000000001,4.0540000000000003,3.9860000000000002,3.9209999999999998,3.8610000000000002,3.8029999999999999,3.7469999999999999,3.694,3.6419999999999999,3.5920000000000001,3.5419999999999998,3.4929999999999999,3.4449999999999998,3.3969999999999998,3.3490000000000002,3.2999999999999998,3.2509999999999999,3.2010000000000001,3.1499999999999999,3.0979999999999999,3.044,2.9889999999999999,2.931,2.871,2.8079999999999998,2.742,2.673,2.6000000000000001,2.5249999999999999,2.4460000000000002,2.3660000000000001,2.2850000000000001,2.2050000000000001,2.1280000000000001,2.0539999999999998,1.986,1.9239999999999999,1.8680000000000001,1.8169999999999999,1.7729999999999999,1.7330000000000001,1.698,1.667,1.641,1.6180000000000001,1.5980000000000001,1.581,1.569],"mode":"markers+lines","showlegend":false,"fill":"tonexty","fillcolor":"rgba(0,0,0,0.2)","marker":{"color":"transparent"},"line":{"color":"transparent"},"text":{},"hoverinfo":"text","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter"},"e6591c77bacc.1":{"x":77.811999999999998,"y":3.2010000000000001,"mode":"markers+lines","showlegend":false,"fill":"tonexty","fillcolor":"rgba(0,0,0,0.2)","marker":{"color":"black","size":20},"line":{"color":"transparent"},"text":{},"hoverinfo":"text","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter","inherit":true}},"layout":{"margin":{"b":40,"l":60,"t":25,"r":10},"title":"Contour Plot","xaxis":{"domain":[0,1],"automargin":true,"title":"Eta","showline":true,"mirror":"ticks","showgrid":true,"gridcolor":"lightgray"},"yaxis":{"domain":[0,1],"automargin":true,"title":"Beta","showline":true,"mirror":"ticks","showgrid":true,"gridcolor":"lightgray"},"hovermode":"closest","showlegend":false},"source":"A","config":{"modeBarButtonsToAdd":["hoverclosest","hovercompare"],"showSendToCloud":false},"data":[{"fillcolor":"rgba(0,0,0,0.2)","x":[77.811999999999998,75.719999999999999,73.602000000000004,71.465000000000003,69.316000000000003,67.174000000000007,65.066999999999993,63.043999999999997,61.183,59.603000000000002,58.445,57.820999999999998,57.710999999999999,57.960000000000001,58.389000000000003,58.878,59.362000000000002,59.82,60.241,60.627000000000002,60.979999999999997,61.305,61.603999999999999,61.881999999999998,62.142000000000003,62.384999999999998,62.616,62.835999999999999,63.045999999999999,63.249000000000002,63.444000000000003,63.634999999999998,63.822000000000003,64.006,64.186999999999998,64.369,64.549999999999997,64.731999999999999,64.917000000000002,65.103999999999999,65.296000000000006,65.492999999999995,65.697999999999993,65.909999999999997,66.134,66.369,66.619,66.887,67.176000000000002,67.491,67.837999999999994,68.224000000000004,68.658000000000001,69.155000000000001,69.731999999999999,70.414000000000001,71.239000000000004,72.266999999999996,73.590999999999994,75.364999999999995,77.811999999999998,80.988,84.192999999999998,86.730000000000004,88.640000000000001,90.122,91.313999999999993,92.302000000000007,93.141000000000005,93.867999999999995,94.506,95.075000000000003,95.587999999999994,96.055999999999997,96.484999999999999,96.884,97.256,97.605999999999995,97.936000000000007,98.251000000000005,98.552999999999997,98.841999999999999,99.122,99.394000000000005,99.659000000000006,99.918999999999997,100.173,100.42400000000001,100.67100000000001,100.91500000000001,101.157,101.39700000000001,101.634,101.866,102.09399999999999,102.31399999999999,102.523,102.71599999999999,102.887,103.023,103.11,103.131,103.06100000000001,102.871,102.532,102.014,101.301,100.383,99.269000000000005,97.974999999999994,96.527000000000001,94.951999999999998,93.272999999999996,91.510999999999996,89.683000000000007,87.801000000000002,85.873999999999995,83.908000000000001,81.908000000000001,79.875,77.811999999999998],"y":[1.569,1.5589999999999999,1.554,1.5529999999999999,1.5569999999999999,1.5680000000000001,1.587,1.6180000000000001,1.665,1.7310000000000001,1.821,1.9350000000000001,2.0630000000000002,2.1930000000000001,2.3140000000000001,2.4220000000000002,2.5179999999999998,2.6019999999999999,2.6760000000000002,2.742,2.8010000000000002,2.855,2.9039999999999999,2.9500000000000002,2.992,3.0310000000000001,3.0680000000000001,3.1040000000000001,3.137,3.1699999999999999,3.2010000000000001,3.2320000000000002,3.262,3.2909999999999999,3.3199999999999998,3.3490000000000002,3.3780000000000001,3.4079999999999999,3.4369999999999998,3.468,3.4980000000000002,3.5299999999999998,3.5630000000000002,3.5979999999999999,3.6339999999999999,3.6720000000000002,3.7130000000000001,3.7559999999999998,3.8029999999999999,3.855,3.9119999999999999,3.9750000000000001,4.0469999999999997,4.1289999999999996,4.2240000000000002,4.3369999999999997,4.4729999999999999,4.641,4.8529999999999998,5.1219999999999999,5.4400000000000004,5.6950000000000003,5.6989999999999998,5.5179999999999998,5.2969999999999997,5.0910000000000002,4.9109999999999996,4.7539999999999996,4.6180000000000003,4.4980000000000002,4.391,4.2949999999999999,4.2080000000000002,4.1280000000000001,4.0540000000000003,3.9860000000000002,3.9209999999999998,3.8610000000000002,3.8029999999999999,3.7469999999999999,3.694,3.6419999999999999,3.5920000000000001,3.5419999999999998,3.4929999999999999,3.4449999999999998,3.3969999999999998,3.3490000000000002,3.2999999999999998,3.2509999999999999,3.2010000000000001,3.1499999999999999,3.0979999999999999,3.044,2.9889999999999999,2.931,2.871,2.8079999999999998,2.742,2.673,2.6000000000000001,2.5249999999999999,2.4460000000000002,2.3660000000000001,2.2850000000000001,2.2050000000000001,2.1280000000000001,2.0539999999999998,1.986,1.9239999999999999,1.8680000000000001,1.8169999999999999,1.7729999999999999,1.7330000000000001,1.698,1.667,1.641,1.6180000000000001,1.5980000000000001,1.581,1.569],"mode":"markers+lines","showlegend":false,"fill":"tonexty","marker":{"color":"transparent","line":{"color":"rgba(31,119,180,1)"}},"line":{"color":"transparent"},"text":["Contour: (77.812, 1.569)","Contour: (75.72, 1.559)","Contour: (73.602, 1.554)","Contour: (71.465, 1.553)","Contour: (69.316, 1.557)","Contour: (67.174, 1.568)","Contour: (65.067, 1.587)","Contour: (63.044, 1.618)","Contour: (61.183, 1.665)","Contour: (59.603, 1.731)","Contour: (58.445, 1.821)","Contour: (57.821, 1.935)","Contour: (57.711, 2.063)","Contour: (57.96, 2.193)","Contour: (58.389, 2.314)","Contour: (58.878, 2.422)","Contour: (59.362, 2.518)","Contour: (59.82, 2.602)","Contour: (60.241, 2.676)","Contour: (60.627, 2.742)","Contour: (60.98, 2.801)","Contour: (61.305, 2.855)","Contour: (61.604, 2.904)","Contour: (61.882, 2.95)","Contour: (62.142, 2.992)","Contour: (62.385, 3.031)","Contour: (62.616, 3.068)","Contour: (62.836, 3.104)","Contour: (63.046, 3.137)","Contour: (63.249, 3.17)","Contour: (63.444, 3.201)","Contour: (63.635, 3.232)","Contour: (63.822, 3.262)","Contour: (64.006, 3.291)","Contour: (64.187, 3.32)","Contour: (64.369, 3.349)","Contour: (64.55, 3.378)","Contour: (64.732, 3.408)","Contour: (64.917, 3.437)","Contour: (65.104, 3.468)","Contour: (65.296, 3.498)","Contour: (65.493, 3.53)","Contour: (65.698, 3.563)","Contour: (65.91, 3.598)","Contour: (66.134, 3.634)","Contour: (66.369, 3.672)","Contour: (66.619, 3.713)","Contour: (66.887, 3.756)","Contour: (67.176, 3.803)","Contour: (67.491, 3.855)","Contour: (67.838, 3.912)","Contour: (68.224, 3.975)","Contour: (68.658, 4.047)","Contour: (69.155, 4.129)","Contour: (69.732, 4.224)","Contour: (70.414, 4.337)","Contour: (71.239, 4.473)","Contour: (72.267, 4.641)","Contour: (73.591, 4.853)","Contour: (75.365, 5.122)","Contour: (77.812, 5.44)","Contour: (80.988, 5.695)","Contour: (84.193, 5.699)","Contour: (86.73, 5.518)","Contour: (88.64, 5.297)","Contour: (90.122, 5.091)","Contour: (91.314, 4.911)","Contour: (92.302, 4.754)","Contour: (93.141, 4.618)","Contour: (93.868, 4.498)","Contour: (94.506, 4.391)","Contour: (95.075, 4.295)","Contour: (95.588, 4.208)","Contour: (96.056, 4.128)","Contour: (96.485, 4.054)","Contour: (96.884, 3.986)","Contour: (97.256, 3.921)","Contour: (97.606, 3.861)","Contour: (97.936, 3.803)","Contour: (98.251, 3.747)","Contour: (98.553, 3.694)","Contour: (98.842, 3.642)","Contour: (99.122, 3.592)","Contour: (99.394, 3.542)","Contour: (99.659, 3.493)","Contour: (99.919, 3.445)","Contour: (100.173, 3.397)","Contour: (100.424, 3.349)","Contour: (100.671, 3.3)","Contour: (100.915, 3.251)","Contour: (101.157, 3.201)","Contour: (101.397, 3.15)","Contour: (101.634, 3.098)","Contour: (101.866, 3.044)","Contour: (102.094, 2.989)","Contour: (102.314, 2.931)","Contour: (102.523, 2.871)","Contour: (102.716, 2.808)","Contour: (102.887, 2.742)","Contour: (103.023, 2.673)","Contour: (103.11, 2.6)","Contour: (103.131, 2.525)","Contour: (103.061, 2.446)","Contour: (102.871, 2.366)","Contour: (102.532, 2.285)","Contour: (102.014, 2.205)","Contour: (101.301, 2.128)","Contour: (100.383, 2.054)","Contour: (99.269, 1.986)","Contour: (97.975, 1.924)","Contour: (96.527, 1.868)","Contour: (94.952, 1.817)","Contour: (93.273, 1.773)","Contour: (91.511, 1.733)","Contour: (89.683, 1.698)","Contour: (87.801, 1.667)","Contour: (85.874, 1.641)","Contour: (83.908, 1.618)","Contour: (81.908, 1.598)","Contour: (79.875, 1.581)","Contour: (77.812, 1.569)"],"hoverinfo":["text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text"],"type":"scatter","name":"rgba(0,0,0,0.2)","error_y":{"color":"rgba(31,119,180,1)"},"error_x":{"color":"rgba(31,119,180,1)"},"xaxis":"x","yaxis":"y","frame":null},{"fillcolor":"rgba(0,0,0,0.2)","x":[77.811999999999998],"y":[3.2010000000000001],"mode":"markers+lines","showlegend":false,"fill":"tonexty","marker":{"color":"black","size":20,"line":{"color":"rgba(255,127,14,1)"}},"line":{"color":"transparent"},"text":"Estimates: (77.812, 3.201)","hoverinfo":"text","type":"scatter","name":"rgba(0,0,0,0.2)","error_y":{"color":"rgba(255,127,14,1)"},"error_x":{"color":"rgba(255,127,14,1)"},"xaxis":"x","yaxis":"y","frame":null}],"highlight":{"on":"plotly_click","persistent":false,"dynamic":false,"selectize":false,"opacityDim":0.20000000000000001,"selected":{"opacity":1},"debounce":0},"shinyEvents":["plotly_hover","plotly_click","plotly_selected","plotly_relayout","plotly_brushed","plotly_brushing","plotly_clickannotation","plotly_doubleclick","plotly_deselect","plotly_afterplot","plotly_sunburstclick"],"base_url":"https://plot.ly"},"evals":[],"jsHooks":[]}</script>

# Customization

WeibullR.plotly has several customization options.

``` r
plotly_wblr(obj, main='Weibull Probability Plot', xlab='Years', ylab='Failure Probability', confCol='blue', signif=4, grid=FALSE)
```

<div class="plotly html-widget html-fill-item" id="htmlwidget-d390a4cb6fd77613ef95" style="width:672px;height:480px;"></div>
<script type="application/json" data-for="htmlwidget-d390a4cb6fd77613ef95">{"x":{"data":[{"x":[30,49,82,90,96],"y":[0.13862943611198913,0.37660097008368176,0.69314718055994529,1.1589670285784068,2.0444649242511774],"mode":"markers","marker":{"color":"black","line":{"color":"rgba(31,119,180,1)"}},"showlegend":false,"error_x":{"color":"black"},"text":["Probability: (30, 0.1294)","Probability: (49, 0.3138)","Probability: (82, 0.5)","Probability: (90, 0.6862)","Probability: (96, 0.8706)"],"hoverinfo":["text","text","text","text","text"],"type":"scatter","error_y":{"color":"rgba(31,119,180,1)"},"line":{"color":"rgba(31,119,180,1)"},"xaxis":"x","yaxis":"y","frame":null},{"x":[8.9940129561755136,10.400966644723374,12.028027898906155,13.909609986149906,16.085526030678242,18.490692936502519,18.601836174706392,21.511761408882425,24.876931598109238,28.768487759733627,30.767006793678103,33.268827270277939,38.473177004540283,38.52499805425434,44.491660228855075,51.451611681041292,59.500348528807677,68.808167683390806,69.393815371807364,77.811696376384603,79.572029974216292,92.019689982633665,106.41457442665546,123.06144076514173,142.31219975563769],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789822469],"mode":"markers+lines","marker":{"color":"transparent","line":{"color":"rgba(255,127,14,1)"}},"showlegend":false,"error_x":{"color":"black"},"text":["Fit: 8.994, 0.001)","Fit: 10.401, 0.0016)","Fit: 12.028, 0.0025)","Fit: 13.9096, 0.004)","Fit: 16.0855, 0.0064)","Fit: 18.4907, 0.01)","Fit: 18.6018, 0.0102)","Fit: 21.5118, 0.0162)","Fit: 24.8769, 0.0256)","Fit: 28.7685, 0.0405)","Fit: 30.767, 0.05)","Fit: 33.2688, 0.0638)","Fit: 38.4732, 0.0996)","Fit: 38.525, 0.1)","Fit: 44.4917, 0.1538)","Fit: 51.4516, 0.2336)","Fit: 59.5003, 0.3453)","Fit: 68.8082, 0.4906)","Fit: 69.3938, 0.5)","Fit: 77.8117, 0.6321)","Fit: 79.572, 0.6584)","Fit: 92.0197, 0.8193)","Fit: 106.4146, 0.9344)","Fit: 123.0614, 0.9869)","Fit: 142.3122, 0.999)"],"hoverinfo":["text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text"],"type":"scatter","line":{"color":"black"},"error_y":{"color":"rgba(255,127,14,1)"},"xaxis":"x","yaxis":"y","frame":null},{"x":[0.8201963735581661,1.103549730426316,1.4847968997299186,1.997753006249692,2.6879185372122518,3.5725394675824265,3.6165201293583964,4.8659189878683149,6.5262566863898046,8.7488200073916129,10.017716990259581,11.728306066723682,15.653475679011592,15.695206914467454,20.8670184801943,27.616927437788672,36.288288509349371,47.084613329645315,47.791296840060795,57.710548613466834,59.749078343532751,73.471266079626773,86.765600064172261,98.665716186659765,109.58765548015663],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789822469],"mode":"markers+lines","marker":{"color":"transparent","line":{"color":"rgba(44,160,44,1)"}},"showlegend":false,"error_x":{"color":"black"},"text":["Upper: 0.8202, 0.001)","Upper: 1.1035, 0.0016)","Upper: 1.4848, 0.0025)","Upper: 1.9978, 0.004)","Upper: 2.6879, 0.0064)","Upper: 3.5725, 0.01)","Upper: 3.6165, 0.0102)","Upper: 4.8659, 0.0162)","Upper: 6.5263, 0.0256)","Upper: 8.7488, 0.0405)","Upper: 10.0177, 0.05)","Upper: 11.7283, 0.0638)","Upper: 15.6535, 0.0996)","Upper: 15.6952, 0.1)","Upper: 20.867, 0.1538)","Upper: 27.6169, 0.2336)","Upper: 36.2883, 0.3453)","Upper: 47.0846, 0.4906)","Upper: 47.7913, 0.5)","Upper: 57.7105, 0.6321)","Upper: 59.7491, 0.6584)","Upper: 73.4713, 0.8193)","Upper: 86.7656, 0.9344)","Upper: 98.6657, 0.9869)","Upper: 109.5877, 0.999)"],"hoverinfo":["text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text"],"type":"scatter","line":{"color":"transparent"},"error_y":{"color":"rgba(44,160,44,1)"},"xaxis":"x","yaxis":"y","frame":null},{"fillcolor":"rgba(0,0,255,0.2)","x":[25.055100948849507,27.186405316122627,29.499028585989727,32.008367993420443,34.751900447057899,37.678121784566301,37.8093501365015,41.135770821802282,44.754886761018767,48.692371499144286,50.627146742790124,53.042849877078361,57.912564198949092,57.959692613155056,63.429125252595874,69.927996986882221,78.136355352994755,89.510575257434311,90.311994763706082,103.13132317653373,106.12222744555835,130.37137508471946,165.07900433123183,213.4434840203173,279.89645425080414],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789822469],"mode":"markers+lines","marker":{"color":"transparent","line":{"color":"rgba(214,39,40,1)"}},"showlegend":false,"error_x":{"color":"black"},"text":["Lower: 25.0551, 0.001)","Lower: 27.1864, 0.0016)","Lower: 29.499, 0.0025)","Lower: 32.0084, 0.004)","Lower: 34.7519, 0.0064)","Lower: 37.6781, 0.01)","Lower: 37.8094, 0.0102)","Lower: 41.1358, 0.0162)","Lower: 44.7549, 0.0256)","Lower: 48.6924, 0.0405)","Lower: 50.6271, 0.05)","Lower: 53.0428, 0.0638)","Lower: 57.9126, 0.0996)","Lower: 57.9597, 0.1)","Lower: 63.4291, 0.1538)","Lower: 69.928, 0.2336)","Lower: 78.1364, 0.3453)","Lower: 89.5106, 0.4906)","Lower: 90.312, 0.5)","Lower: 103.1313, 0.6321)","Lower: 106.1222, 0.6584)","Lower: 130.3714, 0.8193)","Lower: 165.079, 0.9344)","Lower: 213.4435, 0.9869)","Lower: 279.8965, 0.999)"],"hoverinfo":["text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text"],"type":"scatter","fill":"tonexty","line":{"color":"transparent"},"error_y":{"color":"rgba(214,39,40,1)"},"xaxis":"x","yaxis":"y","frame":null},{"mode":"markers","marker":{"color":"black","line":{"color":"rgba(148,103,189,1)"}},"showlegend":false,"text":"Suspension: ","hoverinfo":"text","type":"scatter","error_y":{"color":"rgba(148,103,189,1)"},"error_x":{"color":"rgba(148,103,189,1)"},"line":{"color":"rgba(148,103,189,1)"},"xaxis":"x2","yaxis":"y2","frame":null},{"domain":{"x":[0.77500000000000002,1]},"header":{"values":["Parameter","Value"],"align":["center","center"],"line":{"width":1,"color":"black"},"fill":{"color":["grey","grey"]},"font":{"family":"Arial","color":"white"}},"cells":{"values":[["Ranks","n","Failures","Intervals","Suspensions","Distribution","Method","Beta","Eta","Loglikelihood","CI","Type"],["median","5","5","0","0","weibull","rr-xony","3.2012","77.8117","-23.1816","0.9","pivotal-rr"]],"align":["center","center"],"line":{"color":"black","width":1},"font":{"family":"Arial","color":"black"}},"type":"table","frame":null}],"layout":{"xaxis":{"domain":[0,0.75],"automargin":true,"type":"log","title":"Years","showline":true,"mirror":"ticks","showgrid":true,"gridcolor":false,"range":[0.95395350884805374,2.1532421317037427],"anchor":"y"},"xaxis2":{"domain":[0,0.75],"automargin":true,"type":"log","title":"","zeroline":false,"showline":true,"mirror":"ticks","showticklabels":false,"showgrid":false,"range":[0.95395350884805374,2.1532421317037427],"anchor":"y2"},"yaxis2":{"domain":[0.90000000000000002,1],"automargin":true,"title":"","zeroline":false,"showline":true,"mirror":"ticks","showticklabels":false,"showgrid":false,"anchor":"x2"},"yaxis":{"domain":[0,0.875],"automargin":true,"type":"log","title":"Failure Probability","showline":true,"mirror":"ticks","size":["function (x, ...) ","UseMethod(\"text\")"],"showgrid":true,"gridcolor":false,"range":[-3,1.0611856930352348],"tickvals":[1.0000000111269894e-08,1.0000000505039327e-07,1.0000005000059672e-06,1.000005000029529e-05,0.00010000500033327544,0.00050012504168215146,0.001000500333583622,0.0020020026706729687,0.0050125418235441935,0.010050335853501506,0.020202707317519469,0.051293294387550481,0.10536051565782635,0.22314355131420976,0.69314718055994529,2.3025850929940459,4.6051701859880909,11.512925464963677],"ticktext":[9.9999999999999995e-07,1.0000000000000001e-05,0.0001,0.001,0.01,0.050000000000000003,0.10000000000000001,0.20000000000000001,0.5,1,2,5,10,20,50,90,99,99.998999999999995],"anchor":"x"},"annotations":[],"shapes":[],"images":[],"margin":{"b":40,"l":60,"t":25,"r":10},"title":"Weibull Probability Plot","hovermode":"closest","showlegend":false,"xaxis3":{"domain":[0.77500000000000002,1]},"yaxis3":{"domain":[0,0.84999999999999998]}},"attrs":{"e659574faead":{"x":[30,49,82,90,96],"y":[0.13862943611198913,0.37660097008368176,0.69314718055994529,1.1589670285784068,2.0444649242511774],"mode":"markers","marker":{"color":"black"},"showlegend":false,"error_x":{"array":{},"color":"black"},"text":{},"hoverinfo":"text","name":"","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter"},"e659574faead.1":{"x":[8.9940129561755136,10.400966644723374,12.028027898906155,13.909609986149906,16.085526030678242,18.490692936502519,18.601836174706392,21.511761408882425,24.876931598109238,28.768487759733627,30.767006793678103,33.268827270277939,38.473177004540283,38.52499805425434,44.491660228855075,51.451611681041292,59.500348528807677,68.808167683390806,69.393815371807364,77.811696376384603,79.572029974216292,92.019689982633665,106.41457442665546,123.06144076514173,142.31219975563769],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789822469],"mode":"markers+lines","marker":{"color":"transparent"},"showlegend":false,"error_x":{"color":"black"},"text":{},"hoverinfo":"text","name":"","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter","line":{"color":"black"},"inherit":true},"e659574faead.2":{"x":[0.8201963735581661,1.103549730426316,1.4847968997299186,1.997753006249692,2.6879185372122518,3.5725394675824265,3.6165201293583964,4.8659189878683149,6.5262566863898046,8.7488200073916129,10.017716990259581,11.728306066723682,15.653475679011592,15.695206914467454,20.8670184801943,27.616927437788672,36.288288509349371,47.084613329645315,47.791296840060795,57.710548613466834,59.749078343532751,73.471266079626773,86.765600064172261,98.665716186659765,109.58765548015663],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789822469],"mode":"markers+lines","marker":{"color":"transparent"},"showlegend":false,"error_x":{"color":"black"},"text":{},"hoverinfo":"text","name":"","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter","line":{"color":"transparent"},"inherit":true},"e659574faead.3":{"x":[25.055100948849507,27.186405316122627,29.499028585989727,32.008367993420443,34.751900447057899,37.678121784566301,37.8093501365015,41.135770821802282,44.754886761018767,48.692371499144286,50.627146742790124,53.042849877078361,57.912564198949092,57.959692613155056,63.429125252595874,69.927996986882221,78.136355352994755,89.510575257434311,90.311994763706082,103.13132317653373,106.12222744555835,130.37137508471946,165.07900433123183,213.4434840203173,279.89645425080414],"y":[0.001000500333583622,0.0015932184988378003,0.0025370856818168483,0.0040401203062774621,0.0064335811726203736,0.010050335853501506,0.010245001264229723,0.016314358386566991,0.025979462726955616,0.041370269795678635,0.051293294387550481,0.065879045906211298,0.10490750717055593,0.10536051565782635,0.1670574483051869,0.26602626140105107,0.4236270420780317,0.67459435354671915,0.69314718055994529,1.0000011992290341,1.0742407032111909,1.7106459504620248,2.7240728814078419,4.3378949233139759,6.9077552789822469],"mode":"markers+lines","marker":{"color":"transparent"},"showlegend":false,"error_x":{"color":"black"},"text":{},"hoverinfo":"text","name":"","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter","fill":"tonexty","fillcolor":"rgba(0,0,255,0.2)","line":{"color":"transparent"},"inherit":true},"e65930550ca3":{"x":null,"y":null,"mode":"markers","marker":{"color":"black"},"showlegend":false,"text":{},"hoverinfo":"text","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter"},"e6591fb0ba1f":{"domain":{"x":[0.77500000000000002,1]},"header":{"values":["Parameter","Value"],"align":["center","center"],"line":{"width":1,"color":"black"},"fill":{"color":["grey","grey"]},"font":{"family":"Arial","color":"white"}},"cells":{"values":[["Ranks","n","Failures","Intervals","Suspensions","Distribution","Method","Beta","Eta","Loglikelihood","CI","Type"],["median","5","5","0","0","weibull","rr-xony","3.2012","77.8117","-23.1816","0.9","pivotal-rr"]],"align":["center","center"],"line":{"color":"black","width":1},"font":{"family":"Arial","color":"black"}},"alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"table"}},"source":"A","config":{"modeBarButtonsToAdd":["hoverclosest","hovercompare"],"showSendToCloud":false},"highlight":{"on":"plotly_click","persistent":false,"dynamic":false,"selectize":false,"opacityDim":0.20000000000000001,"selected":{"opacity":1},"debounce":0},"subplot":true,"shinyEvents":["plotly_hover","plotly_click","plotly_selected","plotly_relayout","plotly_brushed","plotly_brushing","plotly_clickannotation","plotly_doubleclick","plotly_deselect","plotly_afterplot","plotly_sunburstclick"],"base_url":"https://plot.ly"},"evals":[],"jsHooks":[]}</script>

``` r
plotly_contour(obj, main='Weibull Contour Plot', col='red', signif=4, grid=FALSE)
```

<div class="plotly html-widget html-fill-item" id="htmlwidget-ed6141caaf905eb34715" style="width:672px;height:480px;"></div>
<script type="application/json" data-for="htmlwidget-ed6141caaf905eb34715">{"x":{"visdat":{"e659699f5e81":["function () ","plotlyVisDat"]},"cur_data":"e659699f5e81","attrs":{"e659699f5e81":{"x":[77.811700000000002,75.719899999999996,73.602500000000006,71.464799999999997,69.316199999999995,67.1738,65.066599999999994,63.043599999999998,61.183300000000003,59.602699999999999,58.445399999999999,57.821199999999997,57.710500000000003,57.959600000000002,58.389099999999999,58.877499999999998,59.362400000000001,59.819600000000001,60.241100000000003,60.627099999999999,60.9803,61.304600000000001,61.604300000000002,61.881799999999998,62.141599999999997,62.385300000000001,62.616399999999999,62.835599999999999,63.0458,63.248800000000003,63.444099999999999,63.634599999999999,63.821800000000003,64.005600000000001,64.187200000000004,64.368700000000004,64.549899999999994,64.732100000000003,64.916499999999999,65.103999999999999,65.296199999999999,65.493399999999994,65.697900000000004,65.910399999999996,66.133700000000005,66.368600000000001,66.618700000000004,66.886499999999998,67.175799999999995,67.490899999999996,67.837900000000005,68.223600000000005,68.658500000000004,69.155199999999994,69.7316,70.413600000000002,71.239099999999993,72.266800000000003,73.590599999999995,75.364800000000002,77.811700000000002,80.988100000000003,84.192899999999995,86.729799999999997,88.640199999999993,90.122200000000007,91.314300000000003,92.302300000000002,93.141499999999994,93.867599999999996,94.505899999999997,95.075299999999999,95.588200000000001,96.055700000000002,96.485100000000003,96.884200000000007,97.255799999999994,97.605599999999995,97.936199999999999,98.251300000000001,98.552899999999994,98.842399999999998,99.122399999999999,99.394199999999998,99.659400000000005,99.918899999999994,100.17319999999999,100.4239,100.6712,100.9153,101.1571,101.39700000000001,101.6335,101.86579999999999,102.09439999999999,102.31440000000001,102.5234,102.71639999999999,102.88679999999999,103.0226,103.1105,103.1313,103.0608,102.871,102.53149999999999,102.01439999999999,101.3005,100.38290000000001,99.268600000000006,97.974699999999999,96.527299999999997,94.951700000000002,93.2727,91.511099999999999,89.683000000000007,87.801000000000002,85.874300000000005,83.9084,81.907799999999995,79.875,77.811700000000002],"y":[1.5685,1.5590999999999999,1.5536000000000001,1.5526,1.5569,1.5679000000000001,1.5873999999999999,1.6184000000000001,1.6647000000000001,1.7309000000000001,1.8211999999999999,1.9348000000000001,2.0630000000000002,2.1926000000000001,2.3136999999999999,2.4222000000000001,2.5177999999999998,2.6017999999999999,2.6760000000000002,2.7421000000000002,2.8014000000000001,2.8551000000000002,2.9043000000000001,2.9496000000000002,2.9916999999999998,3.0310999999999999,3.0682999999999998,3.1036000000000001,3.1373000000000002,3.1698,3.2012,3.2317,3.2616999999999998,3.2911000000000001,3.3203,3.3494000000000002,3.3784000000000001,3.4077000000000002,3.4373999999999998,3.4674999999999998,3.4984000000000002,3.5303,3.5632000000000001,3.5975999999999999,3.6337000000000002,3.6718999999999999,3.7126000000000001,3.7562000000000002,3.8033999999999999,3.855,3.9119000000000002,3.9752999999999998,4.0468999999999999,4.1288999999999998,4.2241999999999997,4.3369999999999997,4.4733000000000001,4.6414,4.8533999999999997,5.1219999999999999,5.4396000000000004,5.6946000000000003,5.6989000000000001,5.5175999999999998,5.2969999999999997,5.0913000000000004,4.9108000000000001,4.7542,4.6177000000000001,4.4974999999999996,4.3906999999999998,4.2948000000000004,4.2077,4.1280000000000001,4.0544000000000002,3.9857999999999998,3.9214000000000002,3.8605999999999998,3.8027000000000002,3.7471999999999999,3.6938,3.6419999999999999,3.5914999999999999,3.5419999999999998,3.4931999999999999,3.4449000000000001,3.3967000000000001,3.3485,3.2999999999999998,3.2509999999999999,3.2012,3.1503000000000001,3.0981999999999998,3.0444,2.9887999999999999,2.9310999999999998,2.8708,2.8079000000000001,2.7418999999999998,2.6726999999999999,2.6002999999999998,2.5247000000000002,2.4464999999999999,2.3662999999999998,2.2854999999999999,2.2054999999999998,2.1278999999999999,2.0545,1.9862,1.9238,1.8675999999999999,1.8171999999999999,1.7725,1.7330000000000001,1.6980999999999999,1.6674,1.6407,1.6174999999999999,1.5978000000000001,1.5814999999999999,1.5685],"mode":"markers+lines","showlegend":false,"fill":"tonexty","fillcolor":"rgba(255,0,0,0.2)","marker":{"color":"transparent"},"line":{"color":"transparent"},"text":{},"hoverinfo":"text","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter"},"e659699f5e81.1":{"x":77.811700000000002,"y":3.2012,"mode":"markers+lines","showlegend":false,"fill":"tonexty","fillcolor":"rgba(255,0,0,0.2)","marker":{"color":"black","size":20},"line":{"color":"transparent"},"text":{},"hoverinfo":"text","alpha_stroke":1,"sizes":[10,100],"spans":[1,20],"type":"scatter","inherit":true}},"layout":{"margin":{"b":40,"l":60,"t":25,"r":10},"title":"Weibull Contour Plot","xaxis":{"domain":[0,1],"automargin":true,"title":"Eta","showline":true,"mirror":"ticks","showgrid":true,"gridcolor":false},"yaxis":{"domain":[0,1],"automargin":true,"title":"Beta","showline":true,"mirror":"ticks","showgrid":true,"gridcolor":false},"hovermode":"closest","showlegend":false},"source":"A","config":{"modeBarButtonsToAdd":["hoverclosest","hovercompare"],"showSendToCloud":false},"data":[{"fillcolor":"rgba(255,0,0,0.2)","x":[77.811700000000002,75.719899999999996,73.602500000000006,71.464799999999997,69.316199999999995,67.1738,65.066599999999994,63.043599999999998,61.183300000000003,59.602699999999999,58.445399999999999,57.821199999999997,57.710500000000003,57.959600000000002,58.389099999999999,58.877499999999998,59.362400000000001,59.819600000000001,60.241100000000003,60.627099999999999,60.9803,61.304600000000001,61.604300000000002,61.881799999999998,62.141599999999997,62.385300000000001,62.616399999999999,62.835599999999999,63.0458,63.248800000000003,63.444099999999999,63.634599999999999,63.821800000000003,64.005600000000001,64.187200000000004,64.368700000000004,64.549899999999994,64.732100000000003,64.916499999999999,65.103999999999999,65.296199999999999,65.493399999999994,65.697900000000004,65.910399999999996,66.133700000000005,66.368600000000001,66.618700000000004,66.886499999999998,67.175799999999995,67.490899999999996,67.837900000000005,68.223600000000005,68.658500000000004,69.155199999999994,69.7316,70.413600000000002,71.239099999999993,72.266800000000003,73.590599999999995,75.364800000000002,77.811700000000002,80.988100000000003,84.192899999999995,86.729799999999997,88.640199999999993,90.122200000000007,91.314300000000003,92.302300000000002,93.141499999999994,93.867599999999996,94.505899999999997,95.075299999999999,95.588200000000001,96.055700000000002,96.485100000000003,96.884200000000007,97.255799999999994,97.605599999999995,97.936199999999999,98.251300000000001,98.552899999999994,98.842399999999998,99.122399999999999,99.394199999999998,99.659400000000005,99.918899999999994,100.17319999999999,100.4239,100.6712,100.9153,101.1571,101.39700000000001,101.6335,101.86579999999999,102.09439999999999,102.31440000000001,102.5234,102.71639999999999,102.88679999999999,103.0226,103.1105,103.1313,103.0608,102.871,102.53149999999999,102.01439999999999,101.3005,100.38290000000001,99.268600000000006,97.974699999999999,96.527299999999997,94.951700000000002,93.2727,91.511099999999999,89.683000000000007,87.801000000000002,85.874300000000005,83.9084,81.907799999999995,79.875,77.811700000000002],"y":[1.5685,1.5590999999999999,1.5536000000000001,1.5526,1.5569,1.5679000000000001,1.5873999999999999,1.6184000000000001,1.6647000000000001,1.7309000000000001,1.8211999999999999,1.9348000000000001,2.0630000000000002,2.1926000000000001,2.3136999999999999,2.4222000000000001,2.5177999999999998,2.6017999999999999,2.6760000000000002,2.7421000000000002,2.8014000000000001,2.8551000000000002,2.9043000000000001,2.9496000000000002,2.9916999999999998,3.0310999999999999,3.0682999999999998,3.1036000000000001,3.1373000000000002,3.1698,3.2012,3.2317,3.2616999999999998,3.2911000000000001,3.3203,3.3494000000000002,3.3784000000000001,3.4077000000000002,3.4373999999999998,3.4674999999999998,3.4984000000000002,3.5303,3.5632000000000001,3.5975999999999999,3.6337000000000002,3.6718999999999999,3.7126000000000001,3.7562000000000002,3.8033999999999999,3.855,3.9119000000000002,3.9752999999999998,4.0468999999999999,4.1288999999999998,4.2241999999999997,4.3369999999999997,4.4733000000000001,4.6414,4.8533999999999997,5.1219999999999999,5.4396000000000004,5.6946000000000003,5.6989000000000001,5.5175999999999998,5.2969999999999997,5.0913000000000004,4.9108000000000001,4.7542,4.6177000000000001,4.4974999999999996,4.3906999999999998,4.2948000000000004,4.2077,4.1280000000000001,4.0544000000000002,3.9857999999999998,3.9214000000000002,3.8605999999999998,3.8027000000000002,3.7471999999999999,3.6938,3.6419999999999999,3.5914999999999999,3.5419999999999998,3.4931999999999999,3.4449000000000001,3.3967000000000001,3.3485,3.2999999999999998,3.2509999999999999,3.2012,3.1503000000000001,3.0981999999999998,3.0444,2.9887999999999999,2.9310999999999998,2.8708,2.8079000000000001,2.7418999999999998,2.6726999999999999,2.6002999999999998,2.5247000000000002,2.4464999999999999,2.3662999999999998,2.2854999999999999,2.2054999999999998,2.1278999999999999,2.0545,1.9862,1.9238,1.8675999999999999,1.8171999999999999,1.7725,1.7330000000000001,1.6980999999999999,1.6674,1.6407,1.6174999999999999,1.5978000000000001,1.5814999999999999,1.5685],"mode":"markers+lines","showlegend":false,"fill":"tonexty","marker":{"color":"transparent","line":{"color":"rgba(31,119,180,1)"}},"line":{"color":"transparent"},"text":["Contour: (77.8117, 1.5685)","Contour: (75.7199, 1.5591)","Contour: (73.6025, 1.5536)","Contour: (71.4648, 1.5526)","Contour: (69.3162, 1.5569)","Contour: (67.1738, 1.5679)","Contour: (65.0666, 1.5874)","Contour: (63.0436, 1.6184)","Contour: (61.1833, 1.6647)","Contour: (59.6027, 1.7309)","Contour: (58.4454, 1.8212)","Contour: (57.8212, 1.9348)","Contour: (57.7105, 2.063)","Contour: (57.9596, 2.1926)","Contour: (58.3891, 2.3137)","Contour: (58.8775, 2.4222)","Contour: (59.3624, 2.5178)","Contour: (59.8196, 2.6018)","Contour: (60.2411, 2.676)","Contour: (60.6271, 2.7421)","Contour: (60.9803, 2.8014)","Contour: (61.3046, 2.8551)","Contour: (61.6043, 2.9043)","Contour: (61.8818, 2.9496)","Contour: (62.1416, 2.9917)","Contour: (62.3853, 3.0311)","Contour: (62.6164, 3.0683)","Contour: (62.8356, 3.1036)","Contour: (63.0458, 3.1373)","Contour: (63.2488, 3.1698)","Contour: (63.4441, 3.2012)","Contour: (63.6346, 3.2317)","Contour: (63.8218, 3.2617)","Contour: (64.0056, 3.2911)","Contour: (64.1872, 3.3203)","Contour: (64.3687, 3.3494)","Contour: (64.5499, 3.3784)","Contour: (64.7321, 3.4077)","Contour: (64.9165, 3.4374)","Contour: (65.104, 3.4675)","Contour: (65.2962, 3.4984)","Contour: (65.4934, 3.5303)","Contour: (65.6979, 3.5632)","Contour: (65.9104, 3.5976)","Contour: (66.1337, 3.6337)","Contour: (66.3686, 3.6719)","Contour: (66.6187, 3.7126)","Contour: (66.8865, 3.7562)","Contour: (67.1758, 3.8034)","Contour: (67.4909, 3.855)","Contour: (67.8379, 3.9119)","Contour: (68.2236, 3.9753)","Contour: (68.6585, 4.0469)","Contour: (69.1552, 4.1289)","Contour: (69.7316, 4.2242)","Contour: (70.4136, 4.337)","Contour: (71.2391, 4.4733)","Contour: (72.2668, 4.6414)","Contour: (73.5906, 4.8534)","Contour: (75.3648, 5.122)","Contour: (77.8117, 5.4396)","Contour: (80.9881, 5.6946)","Contour: (84.1929, 5.6989)","Contour: (86.7298, 5.5176)","Contour: (88.6402, 5.297)","Contour: (90.1222, 5.0913)","Contour: (91.3143, 4.9108)","Contour: (92.3023, 4.7542)","Contour: (93.1415, 4.6177)","Contour: (93.8676, 4.4975)","Contour: (94.5059, 4.3907)","Contour: (95.0753, 4.2948)","Contour: (95.5882, 4.2077)","Contour: (96.0557, 4.128)","Contour: (96.4851, 4.0544)","Contour: (96.8842, 3.9858)","Contour: (97.2558, 3.9214)","Contour: (97.6056, 3.8606)","Contour: (97.9362, 3.8027)","Contour: (98.2513, 3.7472)","Contour: (98.5529, 3.6938)","Contour: (98.8424, 3.642)","Contour: (99.1224, 3.5915)","Contour: (99.3942, 3.542)","Contour: (99.6594, 3.4932)","Contour: (99.9189, 3.4449)","Contour: (100.1732, 3.3967)","Contour: (100.4239, 3.3485)","Contour: (100.6712, 3.3)","Contour: (100.9153, 3.251)","Contour: (101.1571, 3.2012)","Contour: (101.397, 3.1503)","Contour: (101.6335, 3.0982)","Contour: (101.8658, 3.0444)","Contour: (102.0944, 2.9888)","Contour: (102.3144, 2.9311)","Contour: (102.5234, 2.8708)","Contour: (102.7164, 2.8079)","Contour: (102.8868, 2.7419)","Contour: (103.0226, 2.6727)","Contour: (103.1105, 2.6003)","Contour: (103.1313, 2.5247)","Contour: (103.0608, 2.4465)","Contour: (102.871, 2.3663)","Contour: (102.5315, 2.2855)","Contour: (102.0144, 2.2055)","Contour: (101.3005, 2.1279)","Contour: (100.3829, 2.0545)","Contour: (99.2686, 1.9862)","Contour: (97.9747, 1.9238)","Contour: (96.5273, 1.8676)","Contour: (94.9517, 1.8172)","Contour: (93.2727, 1.7725)","Contour: (91.5111, 1.733)","Contour: (89.683, 1.6981)","Contour: (87.801, 1.6674)","Contour: (85.8743, 1.6407)","Contour: (83.9084, 1.6175)","Contour: (81.9078, 1.5978)","Contour: (79.875, 1.5815)","Contour: (77.8117, 1.5685)"],"hoverinfo":["text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text","text"],"type":"scatter","name":"rgba(255,0,0,0.2)","error_y":{"color":"rgba(31,119,180,1)"},"error_x":{"color":"rgba(31,119,180,1)"},"xaxis":"x","yaxis":"y","frame":null},{"fillcolor":"rgba(255,0,0,0.2)","x":[77.811700000000002],"y":[3.2012],"mode":"markers+lines","showlegend":false,"fill":"tonexty","marker":{"color":"black","size":20,"line":{"color":"rgba(255,127,14,1)"}},"line":{"color":"transparent"},"text":"Estimates: (77.8117, 3.2012)","hoverinfo":"text","type":"scatter","name":"rgba(255,0,0,0.2)","error_y":{"color":"rgba(255,127,14,1)"},"error_x":{"color":"rgba(255,127,14,1)"},"xaxis":"x","yaxis":"y","frame":null}],"highlight":{"on":"plotly_click","persistent":false,"dynamic":false,"selectize":false,"opacityDim":0.20000000000000001,"selected":{"opacity":1},"debounce":0},"shinyEvents":["plotly_hover","plotly_click","plotly_selected","plotly_relayout","plotly_brushed","plotly_brushing","plotly_clickannotation","plotly_doubleclick","plotly_deselect","plotly_afterplot","plotly_sunburstclick"],"base_url":"https://plot.ly"},"evals":[],"jsHooks":[]}</script>

## Code of Conduct

Please note that the WeibullR.plotly project is released with a
[Contributor Code of
Conduct](https://github.com/paulgovan/WeibullR.plotly/blob/f919aeb72a1d4dd3a64e55221eb1ae214b3480f5/CODE_OF_CONDUCT.md).
By contributing to this project, you agree to abide by its terms.

# More Resources

For an interactive introduction to Life Data Analysis, check out
[WeibullR.learnr](https://paulgovan.github.io/WeibullR.learnr/)

To try WeibullR.plotly in a Shiny app, check out
[WeibullR.shiny](https://paulgovan.github.io/WeibullR.shiny/)
