% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hierarchical_mutual_information.R
\name{HierarchicalMutualInfo}
\alias{HierarchicalMutualInfo}
\alias{HMI}
\alias{SelfHMI}
\alias{EHMI}
\alias{AHMI}
\title{Hierarchical Mutual Information}
\usage{
HierarchicalMutualInfo(tree1, tree2 = NULL, normalize = FALSE)

HMI(tree1, tree2 = NULL, normalize = FALSE)

SelfHMI(tree)

EHMI(tree1, tree2, precision = 0.01, minResample = 36)

AHMI(tree1, tree2, Mean = max, precision = 0.01, minResample = 36)
}
\arguments{
\item{normalize}{If \code{FALSE}, return the raw \acronym{HMI}, in bits.
If \code{TRUE}, normalize to range [0,1] by dividing by
\code{max(SelfHMI(tree1), SelfHMI(tree2))}.
If a function, divide by \code{normalize(SelfHMI(tree1), SelfHMI(tree2))}.}

\item{tree, tree1, tree2}{An object that can be coerced to an \code{\link{HPart}}
object.}

\item{precision}{Numeric; Monte Carlo sampling will terminate once the
relative standard error falls below this value.}

\item{minResample}{Integer specifying minimum number of Monte Carlo samples
to conduct.  Avoids early termination when sample size is too small to
reliably estimate the standard error of the mean.}

\item{Mean}{Function by which to combine the self-information of the
two input hierarchies, in order to normalize the \acronym{HMI}.}
}
\value{
\code{HierarchicalMutualInfo()} returns a numeric value representing the
hierarchical mutual information between the input trees, in bits,
normalized as specified.
Higher values indicate more shared hierarchical structure.

\code{SelfHMI()} returns the hierarchical mutual information of a tree
compared with itself, i.e. its hierarchical entropy (\acronym{HH}).

\code{EHMI()} returns the expected \acronym{HMI} against a uniform
shuffling of element labels, estimated by performing Monte Carlo resampling
on the same hierarchical structure until the relative standard error of the
estimate falls below \code{precision}.
The attributes of the returned object list the variance (\code{var}),
standard deviation (\code{sd}), standard error of the mean (\code{sem}) and
relative error (\code{relativeError}) of the estimate, and the number of Monte
Carlo samples used to obtain it (\code{samples}).

\code{AHMI()} returns the adjusted \acronym{HMI}, normalized such that
zero corresponds to the expected \acronym{HMI} given a random shuffling
of elements on the same hierarchical structure.  The attribute \code{sem} gives
the standard error of the estimate.
}
\description{
Calculate the Hierarchical Mutual Information (\acronym{HMI})
between two trees, following the recursive algorithm of
\insertCite{Perotti2020;textual}{TreeDist}.

This function was written during a code sprint: its documentation and test
cases have not yet been carefully scrutinized, and its implementation may
change without notice.
Please alert the maintainer to any issues you encounter.
}
\details{
\code{HierarchicalMutualInfo()} computes the hierarchical mutual content of trees
\insertCite{Perotti2015,Perotti2020}{TreeDist}, which accounts for the
non-independence of information represented by nested splits.

\code{tree} is converted to a set of hierarchical partitions, and the mutual
information (in bits) is computed recursively; the contribution of a node is
given by:

\deqn{I(t,s) = \log_2(n_{ts}) - \dfrac{H_{us} + H_{tv} - H_{uv}}{n_{ts}} +
\text{mean}(I_{uv})}

Where:
\itemize{
\item \eqn{n_{ts}} is the number of common elements between partitions
\item \eqn{H_{us}, H_{tv}, H_{uv}} are entropy terms from child comparisons
\item \eqn{I_{uv}} is the recursive \acronym{HMI} for child pairs
}

\code{AHMI()} calculates the adjusted hierarchical mutual information:
\deqn{\text{AHMI}(t, s) = \dfrac{I(t, s) - \hat{I}(t, s)}{
 \text{mean}(H(t), H(s)) - \hat{I}(t, s)}}
Where:
\itemize{
\item \eqn{I(t, s)} is the hierarchical mutual information between \code{tree1} and
\code{tree2}
\item \eqn{\hat{I}(t, s)} is the expected \acronym{HMI} between \code{tree1} and
\code{tree2}, estimated by Monte Carlo sampling
\item \eqn{H(t), H(s)} is the entropy (self-mutual information) of each tree
}
}
\examples{
library("TreeTools", quietly = TRUE)

tree1 <- BalancedTree(8)
tree2 <- PectinateTree(8)

# Calculate HMI between two trees
HierarchicalMutualInfo(tree1, tree2)

# HMI normalized against the mean information content of tree1 and tree2
HierarchicalMutualInfo(tree1, tree2, normalize = mean)

# Normalized HMI above is equivalent to:
HMI(tree1, tree2) / mean(SelfHMI(tree1), SelfHMI(tree2))
# Expected mutual info for this pair of hierarchies
EHMI(tree1, tree2, precision = 0.1)
# The adjusted HMI normalizes against this expectation
AHMI(tree1, tree2, precision = 0.1)
}
\references{
\insertAllCited{}
}
\seealso{
Other tree distances: 
\code{\link{JaccardRobinsonFoulds}()},
\code{\link{KendallColijn}()},
\code{\link{MASTSize}()},
\code{\link{MatchingSplitDistance}()},
\code{\link{NNIDist}()},
\code{\link{NyeSimilarity}()},
\code{\link{PathDist}()},
\code{\link{Robinson-Foulds}},
\code{\link{SPRDist}()},
\code{\link{TreeDistance}()}
}
\concept{tree distances}
