% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computeVPB}
\alias{computeVPB}
\title{A Vector Summary of the Persistence Block}

\description{
For a given persistence diagram \eqn{D=\{(b_i,p_i)\}_{i=1}^N}, \code{computeVPB()} vectorizes the persistence block \deqn{f(x,y)=\sum_{i=1}^N \bold 1_{E(b_i,p_i)}(x,y),} where \eqn{E(b_i,p_i)=[b_i-\frac{\lambda_i}{2},b_i+\frac{\lambda_i}{2}]\times [p_i-\frac{\lambda_i}{2},p_i+\frac{\lambda_i}{2}]} and \eqn{\lambda_i=2\tau p_i} with \eqn{\tau\in (0,1]}. Points of \eqn{D} with infinite persistence value are ignored
}

\usage{
computeVPB(D, homDim, xSeq, ySeq, tau)
}
\arguments{
\item{D}{matrix with three columns containing the dimension, birth and persistence values respectively}

\item{homDim}{homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.)}

\item{xSeq}{numeric vector of increasing x (birth) values used for vectorization}

\item{ySeq}{numeric vector of increasing y (persistence) values used for vectorization}

\item{tau}{parameter (between 0 and 1) controlling block size. By default, \code{tau=0.3}}
}

\value{
A numeric vector whose elements are the weighted averages of the persistence block computed over each cell of the two-dimensional grid constructred from \code{xSeq}=\eqn{\{x_1,x_2,\ldots,x_n\}} and \code{ySeq}=\eqn{\{y_1,y_2,\ldots,y_m\}}:

\deqn{\Big(\frac{1}{\Delta x_1\Delta y_1}\int_{[x_1,x_2]\times [y_1,y_2]}f(x,y)wdA,\ldots,\frac{1}{\Delta x_{n-1}\Delta y_{m-1}}\int_{[x_{n-1},x_n]\times [y_{m-1},y_m]}f(x,y)wdA\Big),}
where \eqn{wdA=(x+y)dxdy}, \eqn{\Delta x_k=x_{k+1}-x_k} and \eqn{\Delta y_j=y_{j+1}-y_j}
}

\author{Umar Islambekov, Aleksei Luchinsky}

\references{
1. Chan, K. C., Islambekov, U., Luchinsky, A., & Sanders, R. (2022). A computationally efficient framework for vector representation of persistence diagrams. Journal of Machine Learning Research 23, 1-33.
}

\examples{
N <- 100 
set.seed(123)
# sample N points uniformly from unit circle and add Gaussian noise
X <- TDA::circleUnif(N,r=1) + rnorm(2*N,mean = 0,sd = 0.2)

# compute a persistence diagram using the Rips filtration built on top of X
D <- TDA::ripsDiag(X,maxdimension = 1,maxscale = 2)$diagram 

# switch from the birth-death to the birth-persistence coordinates
D[,3] <- D[,3] - D[,2] 
colnames(D)[3] <- "Persistence"

# construct one-dimensional grid of scale values
ySeqH0 <- unique(quantile(D[D[,1]==0,3],probs = seq(0,1,by=0.2))) 
tau <- 0.3 # parameter in [0,1] which controls the size of blocks around each point of the diagram 
# compute VPB for homological dimension H_0
computeVPB(D,homDim = 0,xSeq=NA,ySeqH0,tau)

xSeqH1 <- unique(quantile(D[D[,1]==1,2],probs = seq(0,1,by=0.2)))
ySeqH1 <- unique(quantile(D[D[,1]==1,3],probs = seq(0,1,by=0.2)))
# compute VPB for homological dimension H_1
computeVPB(D,homDim = 1,xSeqH1,ySeqH1,tau) 
}
