% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotABC.R
\name{plot.ABCSMC}
\alias{plot.ABCSMC}
\title{Plots \code{ABCSMC} objects}
\usage{
\method{plot}{ABCSMC}(
  x,
  type = c("post", "output"),
  gen = NA,
  joint = FALSE,
  transfunc = NA,
  ...
)
}
\arguments{
\item{x}{An \code{ABCSMC} object.}

\item{type}{Takes the value \code{"post"} if you want to plot posterior distributions.
Takes the value \code{"output"} if you want to plot the simulated outputs.}

\item{gen}{A vector of generations to plot. If left missing then defaults to all generations.}

\item{joint}{A logical describing whether joint or marginal distributions are wanted.}

\item{transfunc}{Is a \code{function} object where the arguments to the function must
match all or a subset of the parameters in the model. This function needs 
to return a \code{data.frame} object with columns containing the transformed
parameters.}

\item{...}{Not used here.}
}
\value{
A plot of the ABC posterior distributions for different generations, or the distributions
        of the simulated summary measures for different generations.
}
\description{
Plot method for \code{ABCSMC} objects.
}
\examples{
\donttest{

## set up SIR simulation model
transitions <- c(
    "S -> beta * S * I -> I", 
    "I -> gamma * I -> R"
)
compartments <- c("S", "I", "R")
pars <- c("beta", "gamma")
model <- mparseRcpp(
    transitions = transitions, 
    compartments = compartments,
    pars = pars
)
model <- compileRcpp(model)

## generate function to run simulators
## and return summary statistics
simSIR <- function(pars, data, tols, u, model) {

    ## run model
    sims <- model(pars, 0, data[2] + tols[2], u)
    
    ## this returns a vector of the form:
    ## completed (1/0), t, S, I, R (here)
    if(sims[1] == 0) {
        ## if simulation rejected
        return(NA)
    } else {
        ## extract finaltime and finalsize
        finaltime <- sims[2]
        finalsize <- sims[5]
    }
    
    ## return vector if match, else return NA
    if(all(abs(c(finalsize, finaltime) - data) <= tols)){
        return(c(finalsize, finaltime))
    } else {
        return(NA)
    }
}

## set priors
priors <- data.frame(
    parnames = c("beta", "gamma"), 
    dist = rep("gamma", 2), 
    stringsAsFactors = FALSE
)
priors$p1 <- c(10, 10)
priors$p2 <- c(10^4, 10^2)

## define the targeted summary statistics
data <- c(
    finalsize = 30, 
    finaltime = 76
)

## set initial states (1 initial infection 
## in population of 120)
iniStates <- c(S = 119, I = 1, R = 0)

## set initial tolerances
tols <- c(
    finalsize = 50,
    finaltime = 50
)

## run 2 generations of ABC-SMC
## setting tolerance to be 50th
## percentile of the accepted 
## tolerances at each generation
post <- ABCSMC(
    x = data, 
    priors = priors, 
    func = simSIR, 
    u = iniStates, 
    tols = tols, 
    ptol = 0.2, 
    ngen = 2, 
    npart = 50,
    model = model
)
post

## run one further generation
post <- ABCSMC(post, ptols = 0.5, ngen = 1)
post
summary(post)

## plot posteriors
plot(post)

## plot outputs
plot(post, "output")
}

}
\seealso{
\code{\link{ABCSMC}}, \code{\link{print.ABCSMC}}, \code{\link{summary.ABCSMC}}
}
