% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SVEMnet.R
\name{SVEMnet}
\alias{SVEMnet}
\title{Fit an SVEMnet model (Self-Validated Ensemble Elastic Net)}
\usage{
SVEMnet(
  formula,
  data,
  nBoot = 200,
  glmnet_alpha = c(0.5, 1),
  weight_scheme = c("SVEM", "FRW_plain", "Identity"),
  objective = c("auto", "wAIC", "wBIC", "wSSE"),
  relaxed = "auto",
  response = NULL,
  unseen = c("warn_na", "error"),
  family = c("gaussian", "binomial"),
  ...
)
}
\arguments{
\item{formula}{A formula specifying the model to be fitted, or a
\code{bigexp_spec} created by \code{bigexp_terms()}.}

\item{data}{A data frame containing the variables in the model.}

\item{nBoot}{Integer. Number of bootstrap replicates (default \code{200}).
Each replicate draws FRW weights and fits a glmnet path.}

\item{glmnet_alpha}{Numeric vector of elastic net mixing parameters
\code{alpha} in \code{[0, 1]}. \code{alpha = 1} is lasso, \code{alpha = 0}
is ridge. Defaults to \code{c(0.5, 1)}. When \code{relaxed = TRUE},
\code{alpha = 0} is automatically dropped (ridge + relaxed is not used).}

\item{weight_scheme}{Character. Weighting scheme for train/validation
copies. One of:
\itemize{
\item \code{"SVEM"} (default): Self-Validated Ensemble Model weights.
For each replicate and row, a shared uniform draw \code{U_i ~ Unif(0, 1)}
is converted to anti-correlated FRW weights
\code{w_train_i = -log(U_i)} and \code{w_valid_i = -log(1 - U_i)},
each rescaled to have mean 1.
\item \code{"FRW_plain"}: Fractional random-weight regression without a
separate validation copy. A single FRW vector
\code{w_i = -log(U_i)} (rescaled to mean 1) is used for both training
and validation. This reproduces the FRW bootstrap regression of
Xu et al. (2020) and related work.
\item \code{"Identity"}: Uses unit weights for both training and
validation (no resampling). In combination with \code{nBoot = 1} this
wraps a single \code{glmnet} fit and selects the penalty by the chosen
information criterion, while still using SVEMnet's expansion and
diagnostics.
}}

\item{objective}{Character. One of \code{"auto"}, \code{"wAIC"},
\code{"wBIC"}, or \code{"wSSE"}.
\itemize{
\item For \code{family = "gaussian"}, \code{objective = "auto"} resolves to \code{"wAIC"}.
\item For \code{family = "binomial"}, \code{objective = "auto"} resolves to \code{"wBIC"}.
}
See Details for the exact criteria in the Gaussian and binomial cases.}

\item{relaxed}{Logical or character. Default \code{"auto"}.
If \code{TRUE}, use glmnet's relaxed elastic-net path and select both
the penalty \code{lambda} and the relaxed refit parameter \code{gamma}
on each bootstrap. If \code{FALSE}, fit the standard glmnet path without
the relaxed step. If \code{"auto"} (default), SVEMnet uses
\code{relaxed = TRUE} for \code{family = "gaussian"} and
\code{relaxed = FALSE} for \code{family = "binomial"}.}

\item{response}{Optional character. When \code{formula} is a \code{bigexp_spec},
this names the response column to use on the left-hand side. Defaults to
the response stored in the spec.}

\item{unseen}{How to treat factor levels not seen in the original
\code{bigexp_spec} when \code{formula} is a \code{bigexp_spec}. One of
\code{"warn_na"} (default; convert unseen levels to \code{NA} with a
warning) or \code{"error"} (stop with an error).}

\item{family}{Character. One of \code{"gaussian"} (default) or
\code{"binomial"}. For Gaussian models SVEMnet uses the identity link; for
binomial it uses the canonical logit link. The binomial response must be
numeric 0/1, logical, or a factor with exactly two levels (the second
level is treated as 1).}

\item{...}{Additional arguments passed to \code{glmnet()}, such as
\code{penalty.factor}, \code{lower.limits}, \code{upper.limits},
\code{offset}, or \code{standardize.response}. Any user-supplied
\code{weights} are ignored (SVEMnet supplies its own bootstrap weights).
Any user-supplied \code{standardize} is ignored; SVEMnet always calls
\code{glmnet} with \code{standardize = TRUE}.}
}
\value{
An object of class \code{"svem_model"} (and \code{"svem_binomial"}
when \code{family = "binomial"}) with components:
\itemize{
\item \code{parms}: Vector of ensemble-averaged coefficients, including
the intercept.
\item \code{parms_debiased}: Vector of coefficients after optional
debiasing (see Details; Gaussian only).
\item \code{debias_fit}: If debiasing was performed, the calibration
model \code{lm(y ~ y_pred)}; otherwise \code{NULL}.
\item \code{coef_matrix}: Matrix of per-bootstrap coefficients
(rows = bootstraps, columns = intercept and predictors).
\item \code{nBoot}: Number of bootstrap replicates actually used.
\item \code{glmnet_alpha}: Vector of alpha values considered.
\item \code{best_alphas}: Per-bootstrap alpha selected by the criterion.
\item \code{best_lambdas}: Per-bootstrap lambda selected by the criterion.
\item \code{best_relax_gammas}: Per-bootstrap relaxed gamma selected when
\code{relaxed = TRUE}; \code{NA} otherwise.
\item \code{weight_scheme}: The weighting scheme that was used.
\item \code{relaxed}: Logical flag indicating whether relaxed paths were
used.
\item \code{relaxed_input}: The user-supplied value for \code{relaxed}
(one of \code{TRUE}, \code{FALSE}, or \code{"auto"}). The resolved flag
actually used is reported in \code{relaxed}.
\item \code{dropped_alpha0_for_relaxed}: Logical; \code{TRUE} if
\code{alpha = 0} was dropped because \code{relaxed = TRUE}.
\item \code{objective_input}: The objective requested by the user.
\item \code{objective_used}: The objective actually used after applying
the "auto" rule (for example \code{"wAIC"} or \code{"wBIC"}).
\item \code{objective}: Same as \code{objective_used} (for convenience).
\item \code{auto_used}: Logical; \code{TRUE} if \code{objective = "auto"}.
\item \code{auto_decision}: The objective selected by the auto rule
(wAIC or wBIC) when \code{auto_used = TRUE}.
\item \code{diagnostics}: List with summary information, including:
\itemize{
\item \code{k_summary}: Median and IQR of selected model size
(number of nonzero coefficients including intercept).
\item \code{fallback_rate}: Proportion of bootstraps that fell back
to an intercept-only fit.
\item \code{n_eff_summary}: Summary of raw Kish effective validation
sizes \eqn{n_eff = (\sum_i w_i^{valid})^2 / \sum_i (w_i^{valid})^2}
across bootstraps (before truncation to form \eqn{n_eff_adm}).
\item \code{alpha_freq}: Relative frequency of selected alpha values
(if any).
\item \code{relax_gamma_freq}: Relative frequency of selected relaxed
gamma values (if \code{relaxed = TRUE} and any were selected).
}
\item \code{actual_y}: Numeric response vector used in fitting (0/1 for
binomial).
\item \code{training_X}: Numeric model matrix without the intercept
column used for training.
\item \code{y_pred}: Fitted values from the ensemble on the training
data. For Gaussian this is on the response scale; for binomial it is
on the probability scale.
\item \code{y_pred_debiased}: Debiased fitted values on the training
data (Gaussian only); \code{NULL} otherwise.
\item \code{nobs}: Number of observations used in fitting.
\item \code{nparm}: Number of parameters in the full expansion
(intercept plus predictors).
\item \code{formula}: The formula used for fitting (possibly derived
from a \code{bigexp_spec}).
\item \code{terms}: \code{terms} object used for building the design
matrix, with environment set to \code{baseenv()} for safety.
\item \code{xlevels}: Factor levels recorded at training time.
\item \code{contrasts}: Contrasts used for building the design matrix.
\item \code{schema}: Compact description for safe prediction, including
\code{feature_names}, \code{terms_str}, \code{xlevels},
\code{contrasts}, \code{contrasts_options}, and a simple hash.
\item \code{sampling_schema}: Schema used to generate random candidate
tables, including predictor names, variable classes, numeric ranges,
and factor levels.
\item \code{used_bigexp_spec}: Logical flag indicating whether a
\code{bigexp_spec} was used.
\item \code{family}: The fitted family ("gaussian" or "binomial").
}
}
\description{
Fit a Self-Validated Ensemble Model (SVEM) with elastic net or relaxed
elastic net base learners using \code{glmnet}. Fractional random-weight
(FRW) train/validation weights are drawn on each bootstrap replicate,
a validation-weighted information criterion (wAIC, wBIC, or wSSE) is
minimized to select the penalty, and predictions are ensembled across
replicates. Gaussian and binomial responses are supported.
}
\details{
You can pass either:
\itemize{
\item a standard model formula, e.g.,
\code{y ~ X1 + X2 + X3 + I(X1^2) + (X1 + X2 + X3)^2}, or
\item a \code{bigexp_spec} created by \code{bigexp_terms()}, in which
case SVEMnet will build the design matrix deterministically (locked
types, levels, and contrasts) and, if requested, swap the response
to fit multiple independent responses over the same expansion.
}

SVEMnet implements Self-Validated Ensemble Models using elastic net and
relaxed elastic net base learners from glmnet. Each bootstrap replicate
draws fractional random weights, builds a train and validation copy, fits
a path over lambda (and optionally over alpha and relaxed gamma), and
selects a path point by minimizing a validation-weighted criterion. Final
predictions are obtained by averaging replicate predictions on the chosen
scale.

By default, \code{relaxed = "auto"} resolves to \code{TRUE} for Gaussian
fits and \code{FALSE} for binomial fits.

The function is typically used in small-n design-of-experiments (DOE)
workflows where classical train/validation splits and cross-validation
can be unstable. A common pattern is: (1) build a deterministic expansion
with \code{bigexp_terms()}, (2) fit SVEM models via \code{SVEMnet()},
(3) perform whole-model significance testing, and (4) call
\code{svem_score_random()} for constrained multi-response optimization.

Weighting schemes:
\itemize{
\item With \code{weight_scheme = "SVEM"}, SVEMnet uses a pair of
anti-correlated FRW vectors for train and validation. All rows appear
in every replicate, but train and validation contributions are
separated through the shared uniform draws.
\item With \code{weight_scheme = "FRW_plain"}, a single FRW vector is
used for both train and validation, which reproduces FRW regression
without a self-validation split. This is mainly provided for method
comparison and teaching.
\item With \code{weight_scheme = "Identity"}, both train and validation
weights are 1. Setting \code{nBoot = 1} in this mode yields a single
glmnet fit whose penalty is chosen by the selected information
criterion, without any bootstrap variation.
}

Selection criteria (Gaussian):
For \code{family = "gaussian"}, the validation loss is a weighted sum of
squared errors on the validation copy. The criteria are:
\itemize{
\item \code{"wSSE"}: weighted sum of squared errors (loss-only selector),
\item \code{"wAIC"}: Gaussian AIC analog based on the weighted SSE,
\item \code{"wBIC"}: Gaussian BIC analog based on the weighted SSE.
}
The FRW validation weights are rescaled to have mean one so that their
sum defines a likelihood scale \eqn{n_like = \sum_i w_i^{valid}} (equal to
\code{n} when all validation weights are 1), and the Gaussian loss term is
expressed as \eqn{n_like} times the log of the weighted mean squared error.

The effective validation size is computed from the FRW weights using Kish's
effective sample size \eqn{n_eff = (\sum_i w_i^{valid})^2 / \sum_i (w_i^{valid})^2}
and then truncated to lie between 2 and \code{n} to form \eqn{n_eff_adm}.
The AIC-style selector uses a \eqn{2 k} penalty; the BIC-style selector uses
a \eqn{log(n_eff_adm) k} penalty, so that the loss term is scaled by total
validation weight while the complexity penalty reflects the effective amount
of information under unequal weights. This follows the survey-weighted
information-criterion literature, where the pseudo-likelihood uses total
weight and the BIC penalty uses an effective sample size.

For diagnostics, SVEMnet reports the raw Kish effective sizes across
bootstraps (see \code{diagnostics$n_eff_summary}), while \eqn{n_eff_adm}
is used internally in the penalty and model-size guardrail. Near-interpolating
path points are screened out via a simple model size guardrail before
minimization. When \code{objective = "auto"}, SVEMnet uses \code{"wAIC"}.

This structure (pseudo-likelihood using total weight and BIC penalty using a
Kish-type effective sample size) parallels survey-weighted information
criteria as in Lumley and Scott (2015) and Kish (1965).

Selection criteria (binomial):
For \code{family = "binomial"}, the validation loss is the weighted
negative log-likelihood on the FRW validation copy (equivalently,
proportional to the binomial deviance up to a constant). The same labels
are used:
\itemize{
\item \code{"wSSE"}: loss-only selector based on the weighted negative
log-likelihood (the name is retained for backward compatibility),
\item \code{"wAIC"}: deviance plus a \eqn{2 k} penalty,
\item \code{"wBIC"}: deviance plus a \eqn{log(n_eff_adm) k} penalty.
}
The effective validation size \eqn{n_eff_adm} and the model size guardrail
are handled as in the Gaussian case: we compute a Kish effective size from
the FRW validation weights, truncate it to lie between 2 and \code{n}, and
require the number of nonzero coefficients (excluding the intercept) to be
less than this effective size before evaluating the criterion. For
diagnostics, \code{diagnostics$n_eff_summary} reports the raw Kish effective
sizes prior to truncation. When \code{objective = "auto"} and
\code{family = "binomial"}, SVEMnet always uses \code{"wBIC"} for stability
in small-n logistic fits.

Auto rule:
When \code{objective = "auto"}, SVEMnet selects the criterion by family:
\itemize{
\item \code{family = "gaussian"} -> \code{"wAIC"}
\item \code{family = "binomial"} -> \code{"wBIC"}
}

Relaxed elastic net:
When \code{relaxed = TRUE}, SVEMnet calls glmnet with
\code{relax = TRUE} and traverses a small grid of relaxed refit values
(gamma). For each alpha and gamma, SVEMnet evaluates all lambda path
points on the validation copy and records the combination that minimizes
the selected criterion. Model size is always defined as the number of
nonzero coefficients including the intercept, so standard and relaxed
paths are scored on the same scale.

Gaussian debiasing:
For Gaussian models, SVEMnet optionally performs a simple linear
calibration of ensemble predictions on the training data. When there is
sufficient variation in the fitted values and \code{nBoot} is at least
10, the function fits \code{lm(y ~ y_pred)} and uses the coefficients to
construct debiased coefficients and debiased fitted values. Binomial
fits do not use debiasing; predictions are ensembled on the probability
or link scale directly.

Implementation notes:
\itemize{
\item Predictors are always standardized internally via
\code{glmnet(..., standardize = TRUE)}.
\item The terms object is stored with its environment set to
\code{baseenv()} so that prediction does not accidentally capture
objects from the calling environment.
\item A compact schema (feature names, factor levels, contrasts, and
a simple hash) is stored to allow \code{predict()} and companion
functions to rebuild model matrices deterministically, even when the
original data frame is not available.
\item A separate sampling schema stores raw predictor ranges and
factor levels for use in random candidate generation for optimization.
}
}
\section{Acknowledgments}{

OpenAI's GPT models (o1-preview through GPT-5 Pro) were
used to assist with coding and roxygen documentation; all
content was reviewed and finalized by the author.
}

\examples{
set.seed(42)

n  <- 30
X1 <- rnorm(n)
X2 <- rnorm(n)
X3 <- rnorm(n)
eps <- rnorm(n, sd = 0.5)
y <- 1 + 2 * X1 - 1.5 * X2 + 0.5 * X3 + 1.2 * (X1 * X2) +
  0.8 * (X1^2) + eps
dat <- data.frame(y, X1, X2, X3)

# Minimal hand-written expansion
mod_relax <- SVEMnet(
  y ~ (X1 + X2 + X3)^2 + I(X1^2) + I(X2^2),
  data          = dat,
  glmnet_alpha  = c(1, 0.5),
  nBoot         = 75,
  objective     = "auto",
  weight_scheme = "SVEM",
  relaxed       = FALSE
)

pred_in_raw <- predict(mod_relax, dat, debias = FALSE)
pred_in_db  <- predict(mod_relax, dat, debias = TRUE)

\donttest{
# ---------------------------------------------------------------------------
# Big expansion (full factorial + polynomial surface + partial-cubic crosses)
# Build once, reuse for one or more responses
# ---------------------------------------------------------------------------
spec <- bigexp_terms(
  y ~ X1 + X2 + X3,
  data             = dat,
  factorial_order  = 3,
  polynomial_order = 3,
  include_pc_3way  = FALSE
)

# Fit using the spec (auto-prepares data)
fit_y <- SVEMnet(
  spec, dat,
  glmnet_alpha  = c(1, 0.5),
  nBoot         = 50,
  objective     = "auto",
  weight_scheme = "SVEM"
)

# A second, independent response over the same expansion
set.seed(99)
dat$y2 <- 0.5 + 1.4 * X1 - 0.6 * X2 + 0.2 * X3 + rnorm(n, 0, 0.4)
fit_y2 <- SVEMnet(
  spec, dat, response = "y2",
  glmnet_alpha  = c(1, 0.5),
  nBoot         = 50,
  objective     = "auto",
  weight_scheme = "SVEM"
)

svem_nonzero(fit_y2)

p1 <- predict(fit_y,  dat)
p2 <- predict(fit_y2, dat, debias = TRUE)

# Show that a new batch expands identically under the same spec
newdat <- data.frame(
  y  = y,
  X1 = X1 + rnorm(n, 0, 0.05),
  X2 = X2 + rnorm(n, 0, 0.05),
  X3 = X3 + rnorm(n, 0, 0.05)
)
prep_new <- bigexp_prepare(spec, newdat)
stopifnot(identical(
  colnames(model.matrix(spec$formula, bigexp_prepare(spec, dat)$data)),
  colnames(model.matrix(spec$formula, prep_new$data))
))
preds_new <- predict(fit_y, prep_new$data)

## Binomial example
set.seed(2)
n  <- 120
X1 <- rnorm(n); X2 <- rnorm(n); X3 <- rnorm(n)
eta <- -0.3 + 1.1 * X1 - 0.8 * X2 + 0.5 * X1 * X3
p   <- plogis(eta)
yb  <- rbinom(n, 1, p)
db  <- data.frame(yb = yb, X1 = X1, X2 = X2, X3 = X3)

fit_b <- SVEMnet(
  yb ~ (X1 + X2 + X3)^2, db,
  nBoot        = 50,
  glmnet_alpha = c(1, 0.5),
  family       = "binomial"
)

## Probabilities, link, and classes
p_resp <- predict(fit_b, db, type = "response")
p_link <- predict(fit_b, db, type = "link")
y_hat  <- predict(fit_b, db, type = "class")  # 0/1 labels

## Mean aggregation with uncertainty on probability scale
out_b <- predict(
  fit_b, db,
  type     = "response",
  se.fit   = TRUE,
  interval = TRUE,
  level    = 0.9
)
str(out_b)
}
}
\references{
Gotwalt, C., & Ramsey, P. (2018). Model Validation Strategies for Designed Experiments Using Bootstrapping Techniques With Applications to Biopharmaceuticals. \emph{JMP Discovery Conference}. \url{https://community.jmp.com/t5/Abstracts/Model-Validation-Strategies-for-Designed-Experiments-Using/ev-p/849873/redirect_from_archived_page/true}

Karl, A. T. (2024). A randomized permutation whole-model test heuristic for Self-Validated Ensemble Models (SVEM). \emph{Chemometrics and Intelligent Laboratory Systems}, \emph{249}, 105122. \doi{10.1016/j.chemolab.2024.105122}

Karl, A., Wisnowski, J., & Rushing, H. (2022). JMP Pro 17 Remedies for Practical Struggles with Mixture Experiments. JMP Discovery Conference. \doi{10.13140/RG.2.2.34598.40003/1}

Lemkus, T., Gotwalt, C., Ramsey, P., & Weese, M. L. (2021). Self-Validated Ensemble Models for Design of Experiments. \emph{Chemometrics and Intelligent Laboratory Systems}, 219, 104439. \doi{10.1016/j.chemolab.2021.104439}

Xu, L., Gotwalt, C., Hong, Y., King, C. B., & Meeker, W. Q. (2020). Applications of the Fractional-Random-Weight Bootstrap. \emph{The American Statistician}, 74(4), 345–358. \doi{10.1080/00031305.2020.1731599}

Ramsey, P., Gaudard, M., & Levin, W. (2021). Accelerating Innovation with Space Filling Mixture Designs, Neural Networks and SVEM. \emph{JMP Discovery Conference}. \url{https://community.jmp.com/t5/Abstracts/Accelerating-Innovation-with-Space-Filling-Mixture-Designs/ev-p/756841}

Ramsey, P., & Gotwalt, C. (2018). Model Validation Strategies for Designed Experiments Using Bootstrapping Techniques With Applications to Biopharmaceuticals. \emph{JMP Discovery Conference - Europe}. \url{https://community.jmp.com/t5/Abstracts/Model-Validation-Strategies-for-Designed-Experiments-Using/ev-p/849647/redirect_from_archived_page/true}

Ramsey, P., Levin, W., Lemkus, T., & Gotwalt, C. (2021). SVEM: A Paradigm Shift in Design and Analysis of Experiments. \emph{JMP Discovery Conference - Europe}. \url{https://community.jmp.com/t5/Abstracts/SVEM-A-Paradigm-Shift-in-Design-and-Analysis-of-Experiments-2021/ev-p/756634}

Ramsey, P., & McNeill, P. (2023). CMC, SVEM, Neural Networks, DOE, and Complexity: It's All About Prediction. \emph{JMP Discovery Conference}.

Friedman, J. H., Hastie, T., and Tibshirani, R. (2010).
Regularization Paths for Generalized Linear Models via Coordinate Descent.
Journal of Statistical Software, 33(1), 1-22.

Meinshausen, N. (2007).
Relaxed Lasso. Computational Statistics & Data Analysis, 52(1), 374-393.

Kish, L. (1965). \emph{Survey Sampling}. Wiley.

Lumley, T. (2004). Analysis of complex survey samples.
\emph{Journal of Statistical Software}, 9(1), 1–19.

Lumley, T. and Scott, A. (2015). AIC and BIC for modelling with complex survey data.
\emph{Journal of Survey Statistics and Methodology}, 3(1), 1–18.
}
