###################################################################################################
#' Algorithm Interface to RGP + KTFG 
#'
#' Interface for mixed model tuning of RGP (Genetic Programming for R) doing symbolic regression on
#' test functions generated by the Kriging test function generator (KTFG).
#' TODO update docs
#' For each repeat of RGP parameter setting, a different random seed is used. At the same time,
#' this seed is used to create different Gaussian Landscapes with the Gaussian Landscape Generator GLG, see \code{\link{spotGlgCreate}}.
#'
#' @param spotConfig Contains the list of spot configurations, results of the algorithm can be passed to this list instead of the .res file.
#'		  spotConfig defaults to "NA", and will only be passed to the Algorithm if spotConfig$spot.fileMode=FALSE. See also: \code{\link{spotGetOptions}}
#'			Items used are: \cr \cr
#'			alg.currentDesign: data frame holding the design points that will be evaluated \cr
#'			io.apdFileName: name of the apd file \cr
#'			io.desFileName: name of the des file \cr
#'			io.resFileName: name of the res file, for logging results (if spotConfig$spot.fileMode==TRUE)\cr
#'			spot.fileMode: boolean, if selected with true the results will also be written to the res file, otherwise it will only be saved in the spotConfig returned by this function\cr
#' @return this function returns the \code{spotConfig} list with the results in spotConfig$alg.currentResult
#' @seealso  \code{\link{SPOT}} \code{\link{spot}} \code{\link{demo}} \code{\link{optim}}
#' \code{\link{spotFuncStartBranin}} \code{\link{spotAlgStartSannVar}}
#' @export
###################################################################################################
spotAlgStartKtfgRgp <- function(spotConfig) {
  #require("parallel")
  require("rgp")

  #spotWriteLines(spotConfig$io.verbosity, 2, "Entering spotAlgStartKtfgRgp")

  # additional test functions...
  spotSalutowiczFunction <- function(x) exp(-1*x)*x*x*x*sin(x)*cos(x)*(sin(x)*sin(x)*cos(x)-1)

	# SPOT default settings (can be changed in the .apd file)
  spotControl <- list(
    numberOfCpuCores = 4) # number of CPU cores to use for parallel evaluation of SPO repeats

	# KTFG default settings (can be changed in the .apd file)
  ktfgControl <- list(
    testFunction = spotSalutowiczFunction, #spotBraninFunction,
	  numberOfProblemInstances = 4, # number of random problem instances
    dimensions = 1, #2, # test function dimension
    lowerBounds = c(0), #c(-5, 0),
    upperBounds = c(10), #c(10, 15),
    deltaThetaSd = 0.5, #0.1,
    trainingSamples = 64,
    trainingSeed = 1,
    runSeed = 1)
	
  # RGP default settings (can be changed in the .apd file)
  rgpControl <- list(
    targetFunction = ktfgControl$testFunction,
    dimensions = ktfgControl$dimensions,
    lowerBounds = ktfgControl$lowerBounds,
    upperBounds = ktfgControl$upperBounds,
    numberOfFitnessCases = 64 ^ ktfgControl$dimensions,
    fitnessCases = NA,
    functionSetIndex = 4,
    populationSizeIndex = 3,
    lambdaIndex = 1,
    crossoverProbabilityIndex = 3,
    newIndividualsPerGenerationIndex = 2,
    enableComplexityCriterionIndex = 1,
    enableAgeCriterionIndex = 2, 
    subSamplingShare = 1.0,
    numberOfFitnessEvaluations = 50000,
    seed = 1)

  # save state of the random number generator...
  if (exists(as.character(substitute(.Random.seed)))) {
    SAVESEED <- .Random.seed
  }else{
    SAVESEED <- NULL
  }
  
  # read problem design file...
	source(spotConfig$io.apdFileName, local = TRUE)

  if (spotConfig$spot.fileMode) { 
    des <- read.table(spotConfig$io.desFileName, sep=" ", header = TRUE)
  } else {
    des <- spotConfig$alg.currentDesign
  }

  # init kriging test function generator...
  ktfg <- makeKrigingTestFunctionGenerator(testFunction = ktfgControl$testFunction, dimensions = ktfgControl$dimensions,
                                           lowerBounds = ktfgControl$lowerBounds, upperBounds = ktfgControl$upperBounds,
                                           deltaThetaSd = ktfgControl$deltaThetaSd,
                                           samples = ktfgControl$trainingSamples, trainingSeed = ktfgControl$trainingSeed)

  # init counters and help variables...
  parameterNames <- names(des)
  numberOfConfigs <- nrow(des)

  # loop over problem instances...
  for (j in 1:ktfgControl$numberOfProblemInstances) {
     # generate a symbolic regression problem instance via the KTFG...
     problemInstanceId <- ktfgControl$runSeed + j
     problemInstance <- ktfg(problemInstanceId)

     # plot current problem instance (for debugging)...
     if (ktfgControl$dimensions == 1) {
       plot(ktfgControl$testFunction(seq(from = ktfgControl$lowerBounds[1], to = ktfgControl$upperBounds[1],
                                         length.out = rgpControl$numberOfFitnessCases)),
                                     type = "l", col = "gray", xlab = "Fitness Case #", ylab = "Function Value")
       lines(problemInstance(seq(from = ktfgControl$lowerBounds[1], to = ktfgControl$upperBounds[1],
                                 length.out = rgpControl$numberOfFitnessCases)), col = "black")
       legend("bottomright", c("Base Problem", "Problem Instance"),
              lty = c(1,1), col = c("gray","black"))
     } else if (ktfgControl$dimensions == 2) {
       spotSurfContour(problemInstance, ktfgControl$lowerBounds, ktfgControl$upperBounds)
     }

     # create fitness cases from problem instance
     if (is.na(rgpControl$fitnessCases)) {
       x <- latinHypercubeDesign(rgpControl$dimensions, rgpControl$numberOfFitnessCases,
                                 rgpControl$lowerBounds, rgpControl$upperBounds)
       y <- as.matrix(apply(x, 1, rgpControl$targetFunction))
       rgpControl$fitnessCases <- data.frame(cbind(x, y))
       colnames(rgpControl$fitnessCases) <- c(paste("x", 1:rgpControl$dimensions, sep = ""), "y")
    }

    # loop over configs...
    for (k in 1:numberOfConfigs) {
      writeLines(paste("-Running config number", k, "of", numberOfConfigs))
      numberOfRepeats <- des$REPEATS[k]

			for(i in 1:numberOfRepeats){
      # parallel loop over repeats... does not work on windows
      #mclapply(1:numberOfRepeats, FUN = function(i) {
        writeLines(paste("--Running repeat number", i, "of", numberOfRepeats))

        # read RGP parameter settings from des...
        if (is.element("FUNCTIONSETINDEX", parameterNames)) {
          rgpControl$functionSetIndex <- des$FUNCTIONSETINDEX[k]
        }
        if (is.element("POPULATIONSIZEINDEX", parameterNames)) {
          rgpControl$populationSizeIndex <- des$POPULATIONSIZEINDEX[k]
        }
        if (is.element("LAMBDAINDEX", parameterNames)) {
          rgpControl$lambdaIndex <- des$LAMBDAINDEX[k]
        }
        if (is.element("CROSSOVERPROBABILITYINDEX", parameterNames)) {
          rgpControl$crossoverProbabilityIndex <- des$CROSSOVERPROBABILITYINDEX[k]
        }
        if (is.element("NEWINDIVIDUALSPERGENERATIONINDEX", parameterNames)) {
          rgpControl$newIndividualsPerGenerationIndex <- des$NEWINDIVIDUALSPERGENERATIONINDEX[k]
        }
        if (is.element("ENABLECOMPLEXITYCRITERIONINDEX", parameterNames)) {
          rgpControl$enableComplexityCriterionIndex <- des$ENABLECOMPLEXITYCRITERIONINDEX[k]
        }
        if (is.element("ENABLEAGECRITERIONINDEX", parameterNames)) {
          rgpControl$enableAgeCriterionIndex <- des$ENABLEAGECRITERIONINDEX[k]
        }

        # read output parameters from des...
        if (is.element("CONFIG", parameterNames)) {
          configNumber <- des$CONFIG[k]
        } else {
          configNumber <- k
        }
        if (is.element("STEP", parameterNames)) {
          spotStep <- des$STEP[k]
        } else {
          spotStep <- NA
        }
        seed <- des$SEED[k] + i - 1 # random seed passed to RGP	

        # start RGP run...
        rgpControl$seed <- seed
        rgpResult <- spotSymbolicRegressionRgp(rgpControl, spotConfig)

        # write results...
        res <- list(
          Y = rgpResult$bestFitness, # TODO change quality measure? 
          FUNCTIONSETINDEX = rgpControl$functionSetIndex,
          POPULATIONSIZEINDEX = rgpControl$populationSizeIndex,
          LAMBDAINDEX = rgpControl$lambdaIndex,
          CROSSOVERPROBABILITYINDEX = rgpControl$crossoverProbabilityIndex,
          NEWINDIVIDUALSPERGENERATIONINDEX = rgpControl$newIndividualsPerGenerationIndex,
          ENABLECOMPLEXITYCRITERIONINDEX = rgpControl$enableComplexityCriterionIndex,
          ENABLEAGECRITERIONINDEX = rgpControl$enableAgeCriterionIndex,
          SUBSAMPLINGSHARE = rgpControl$subSamplingShare,
          NUMBEROFFITNESSEVALUATIONS = rgpControl$numberOfFitnessEvaluations,
					PINST = problemInstanceId,          
					SEED = seed,
					CONFIG = configNumber)
        if (is.element("STEP", parameterNames)){
          res <- c(res, STEP = spotStep)
        } 
        resDf <- data.frame(res) # convert res list to data.frame

        if (spotConfig$spot.fileMode) {
          colNames <- !(file.exists(spotConfig$io.resFileName))
          write.table(resDf, file = spotConfig$io.resFileName, row.names = FALSE,
                      col.names = colNames, sep = " ", append = !colNames, quote = FALSE)
        }
        spotConfig$alg.currentResult <- rbind(spotConfig$alg.currentResult, resDf)
      }#, mc.cores = spotControl$numberOfCpuCores) # (parallel) loop on repeats
    } # for loop on configs
  } # for loop on problem instances

  # reset random seed
  if (!is.null(SAVESEED)) {
    assign(".Random.seed", SAVESEED, envir=globalenv())
  }

  #spotWriteLines(spotConfig$io.verbosity, 2, "Leaving spotAlgStartKtfgRgp")
  return (spotConfig)
}


makeKrigingTestFunctionGenerator <- function(testFunction = spotBraninFunction, dimensions = 2,
                                             lowerBounds = c(-5, 0), upperBounds = c(10, 15),
                                             samples = 20, deltaThetaSd = 1, trainingSeed = 1) { 
  set.seed(trainingSeed)
  x <- latinHypercubeDesign(dimensions, samples, lowerBounds, upperBounds)
  y <- as.matrix(apply(x, 1, testFunction))
  originalFit <- forrBuilder(x, y)
  
  krigingTestFunctionGenerator <- function(runSeed) {
    set.seed(runSeed)
    newFit <- originalFit
    newFit$Theta <- newFit$Theta + rnorm(length(newFit$Theta), sd = deltaThetaSd)
    newFit$dmodeltheta <- 10 ^ newFit$Theta
    krigingTestProblem <- function(x) forrRegPredictor(x, newFit, pred.all = FALSE)$f
    return (krigingTestProblem)
  }

  return (krigingTestFunctionGenerator)
}


spotSymbolicRegressionRgp <- function(control, spotConfig) {
  # decode factorial parameter settings...
  functionSetString <- if (1 == control$functionSetIndex) {
    'c("+", "-", "*", "/")'
  } else if (2 == control$functionSetIndex) {
    'c("+", "-", "*", "/", "sin", "cos")'
  } else if (3 == control$functionSetIndex) {
    'c("+", "-", "*", "/", "exp", "log", "sqrt")'
  } else if (4 == control$functionSetIndex) {
    'c("+", "-", "*", "/", "sin", "cos", "exp", "log", "sqrt")'
  }
  populationSize <- if (1 == control$populationSizeIndex) {
    10
  } else if (2 == control$populationSizeIndex) {
    50 
  } else if (3 == control$populationSizeIndex) {
    100 
  } else if (4 == control$populationSizeIndex) {
    500 
  }
  lambda <- if (1 == control$lambdaIndex) {
    ceiling(0.01 * populationSize)
  } else if (2 == control$lambdaIndex) {
    ceiling(0.1 * populationSize)
  } else if (3 == control$lambdaIndex) {
    ceiling(0.25 * populationSize)
  } else if (4 == control$lambdaIndex) {
    ceiling(0.5 * populationSize)
  }
  crossoverProbability <- if (1 == control$crossoverProbabilityIndex) {
    0.0
  } else if (2 == control$crossoverProbabilityIndex) {
    0.25
  } else if (3 == control$crossoverProbabilityIndex) {
    0.75
  } else if (4 == control$crossoverProbabilityIndex) {
    1.0 
  }
  newIndividualsPerGeneration <- if (1 == control$newIndividualsPerGenerationIndex) {
    ceiling(0.01 * populationSize)
  } else if (2 == control$newIndividualsPerGenerationIndex) {
    ceiling(0.05 * populationSize)
  } else if (3 == control$newIndividualsPerGenerationIndex) {
    ceiling(0.25 * populationSize)
  } else if (4 == control$newIndividualsPerGenerationIndex) {
    ceiling(0.5 * populationSize)
  }
  enableComplexityCriterion <- (control$enableComplexityCriterionIndex == 2)
  enableAgeCriterion <- (control$enableAgeCriterionIndex == 2) 

  # perform GP run and return results...
  set.seed(control$seed)

  funSet <- do.call(functionSet, as.list(eval(parse(text = functionSetString))))
  inVarSet <- do.call(inputVariableSet, as.list(paste("x", 1:control$dimensions, sep = ""))) 
  constSet <- numericConstantSet

  populationFactory <- function(populationSize, funSet, inVarSet, maxfuncdepth, constMin, constMax) { 
    Map(function(i) makeClosure(.Call("initialize_expression_grow_R",
                                      as.list(funSet$nameStrings),
                                      as.integer(funSet$arities),
                                      as.list(inVarSet$nameStrings),
                                      constMin, constMax,
                                      0.8, 0.2,
                                      as.integer(maxfuncdepth)),
                                as.list(inVarSet$nameStrings)), 1:populationSize)
  }
 
  searchHeuristic <- makeAgeFitnessComplexityParetoGpSearchHeuristic(lambda = lambda,
                                                                     crossoverProbability = crossoverProbability,
                                                                     newIndividualsPerGeneration = newIndividualsPerGeneration,
                                                                     enableComplexityCriterion = enableComplexityCriterion,
                                                                     enableAgeCriterion = enableAgeCriterion)
  
  mutationFunction <- function(ind) {
    subtreeMutantBody <- mutateSubtreeFast(body(ind), funSet, inVarSet, -1, 1, 0.33, 0.75, 1.0, 0.5, 2) 
    functionMutantBody <- mutateFuncFast(subtreeMutantBody, funSet, mutatefuncprob = 0.33)
    constantMutantBody <- mutateNumericConstFast(functionMutantBody, mutateconstprob = 0.33, mu = 0, sigma = 1)
    mutant <- makeClosure(constantMutantBody, inVarSet$all, envir = funSet$envir)
    return (mutant)
  }

  crossoverFunction <- function(func1, func2, crossoverprob = 1,
                                 breedingFitness = function(individual) TRUE,
                                 breedingTries = 1) {
    childBody <- crossoverexprFast(body(func1), body(func2))
    child <- makeClosure(childBody, inVarSet$all, envir = funSet$envir)
    return (child)
  }

  population <- populationFactory(populationSize, funSet, inVarSet, 8, -10.0, 10.0)

  srFormula <- formula(paste("y ~", do.call(paste, c(as.list(paste("x", 1:control$dimensions, sep = "")), sep = "+"))))

  sr <- symbolicRegression(srFormula,
                           data = control$fitnessCases,
                           functionSet = funSet,
                           errorMeasure = smse,
                           stopCondition = makeEvaluationsStopCondition(control$numberOfFitnessEvaluations),
                           population = population,
                           populationSize = populationSize,
                           individualSizeLimit = 128, # individuals with more than 128 nodes (inner and leafs) get fitness Inf
                           subSamplingShare = control$subSamplingShare,
                           mutationFunction = mutationFunction,
                           crossoverFunction = crossoverFunction,
                           searchHeuristic = searchHeuristic,
                           envir = environment(),
                           verbose = FALSE)

  writeLines(paste("RGP run finished with best fitness      :", sr$bestFitness,
                                                   "\n                      function set #    :", control$functionSetIndex,
                                                   "\n                      crossover prob.   :", crossoverProbability,
                                                   "\n                      population size   :", populationSize,
                                                   "\n                      lambda            :", lambda,
                                                   "\n                      new inds. per gen.:", newIndividualsPerGeneration, 
                                                   "\n                      compl. criterion  :", enableComplexityCriterion, 
                                                   "\n                      age criterion     :", enableAgeCriterion))

  return (sr)
}

