% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SAMCPLUS.R
\name{SAMCPLUS}
\alias{SAMCPLUS}
\title{SAMC Sampler with C++}
\usage{
SAMCPLUS(nv, energy, data = NA, options = list())
}
\arguments{
\item{nv}{number of variables.}

\item{energy}{a \code{CPP} function pointer for negative log density.}

\item{data}{extra data to be supplemented. It should be a vector, a matrix, or a list.}

\item{options}{a list specifying parameters/options for SAMC algorithm,
\tabular{lll}{
PARAMETER        \tab SPECIFICATION \tab DESCRIPTION \cr
\code{domain}    \tab vector(\eqn{2}) or matrix(\eqn{(nv\times 2)}) \tab domain of sample space \cr
\code{partition} \tab vector(\eqn{m}) \tab energy partition \cr
\code{vecpi}     \tab vector(\eqn{m-1}) \tab desired sampling distribution \cr
\code{tau}       \tab positive number \tab temperature \cr
\code{niter}     \tab positive integer \tab number of iterations to be run \cr
\code{t0}        \tab \eqn{(0.5,1]}  \tab gain factor sequence value \cr
\code{xi}        \tab positive number \tab gain factor sequence exponent \cr
\code{stepsize}  \tab positive number \tab stepsize for random-walk sampler \cr
\code{trange}    \tab vector(\eqn{2}) or matrix(\eqn{m\times 2}) \tab domain of estimates for \eqn{\log(g_i /\pi_i)}
}}
}
\value{
a named list containing \describe{
\item{samples}{an \eqn{(niter\times nv)} samples generated.}
\item{frequency}{length-\eqn{m} vector of visiting frequency for energy partition.}
\item{theta}{length-\eqn{m} vector of estimates of \eqn{\log(g_i / \pi_i)}}
}
}
\description{
The function \code{SAMCPLUS} is a generic SAMC sampler for distributions on continuous domain. Instead of an \code{R} function, 
\code{SAMCPLUS} requires a function pointer to be provided for faster sampling, with all other values and parameters being equal 
to its cousin \code{\link{SAMC}}. We limited the flexibility of the function pointer to be passed. See the below for more details or 
the vignette.
}
\section{Note on writing your own C++ function}{

First, the output should be returned as \code{SEXP} rather than \code{double} in evaluating the negative log density. Second, the variable and 
extra data should be provided as \code{arma::vec} and \code{arma::mat} type, with an exception for \code{Rcpp::List} for list-valued data. This means, for the \code{data} Even 
though we could let data to be passed freely, we believe using \pkg{RcppArmadillo}, which is a templated C++ linear algebra library, 
enables easier writing of one's own C++ code in a style of R or MATLAB while providing sufficient computational power. Furthermore, limiting extra data to one of 3 types (vector, matrix, and list) 
reduces potential type-matching issue in encapsulating of the current environment by removing unexpected errors a user might have incurred.
}

\examples{
\donttest{
##### Two-Dimensional Multimodal sampling
## Step 1 : Define negative log-density function as a CPP function
cppscript = "SEXP funcH(arma::vec x){
double x1 = x(0);
double x2 = x(1);
double val1 = (-std::pow((x1*sin(20*x2)+x2*sin(20*x1)),2))*cosh(sin(10*x1)*x1);
double val2 = (-std::pow((x1*cos(10*x2)-x2*sin(10*x1)),2))*cosh(cos(20*x2)*x2);
return Rcpp::wrap(val1+val2);
}"
func_ptr = RcppXPtrUtils::cppXPtr(cppscript,depends="RcppArmadillo") # as a pointer

## Step 2 : Prepare a setting option
myoption = list()
myoption$partition = c(-Inf,seq(from=-8,to=0,length.out=41))
myoption$tau       = 1.0
myoption$domain    = c(-1.1,1.1)
myoption$vecpi     = as.vector(rep(1/41,41))
myoption$niter     = 200000
myoption$stepsize  = 0.25

## Step 3 : Run The Code
res = SAMCPLUS(2,func_ptr,options=myoption)

## Step 4 : Visualize
select = seq(from=101,to=myoption$niter,by=100) # 100 burn-in, 1/100 thinning 
par(mfrow=c(1,2))
plot(res$samples[select,1], res$samples[select,2],xlab='x',ylab='y',main='samples')
barplot(as.vector(res$frequency/sum(res$frequency)),
        main="visiting frequency by energy partition",
        names.arg=myoption$partition[-1], xlab="energy")

##### (2) Use Extra Data
## Define negative log-density function as CPP function
cppscript2 = "SEXP funcH2(arma::vec x, arma::vec data){
double x1 = x(0);
double x2 = x(1);
double par1 = data(0);
double par2 = data(1);

double val1 = (-std::pow((x1*sin(par2*x2)+x2*sin(par2*x1)),2))*cosh(sin(par1*x1)*x1);
double val2 = (-std::pow((x1*cos(par1*x2)-x2*sin(par1*x1)),2))*cosh(cos(par2*x2)*x2);
return Rcpp::wrap(val1+val2);
}"
func_ptr2 = RcppXPtrUtils::cppXPtr(cppscript2,depends="RcppArmadillo") # as a pointer

## Run The Code
vecdata = as.vector(c(10,20)) 
res2 = SAMCPLUS(2,func_ptr2,data=vecdata, options=myoption)
select = seq(from=101,to=ex_niter,by=100) # 100 burn-in, 1/100 thinning 
par(mfrow=c(1,2))
plot(res2$samples[select,1], res2$samples[select,2],xlab='x',ylab='y',main='samples')
barplot(as.vector(res2$frequency/sum(res2$frequency)),
        main="visiting frequency by energy partition",
        names.arg=ex_part[2:(m+1)], xlab="energy")
}

}
\references{
\insertRef{SAMC}{SAMCpack}
}
\author{
Kisung You
}
