% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/FeatureSelection.R
\name{FS.greedy.heuristic.reduct.RST}
\alias{FS.greedy.heuristic.reduct.RST}
\title{The greedy heuristic algorithm for computing decision reducts and approximate decision reducts}
\usage{
FS.greedy.heuristic.reduct.RST(decision.table,
  attrDescriptions = attr(decision.table, "desc.attrs"),
  decisionIdx = ncol(decision.table), qualityF = X.gini, nAttrs = NULL,
  epsilon = 0, inconsistentDecisionTable = FALSE)
}
\arguments{
\item{decision.table}{an object of a \code{"DecisionTable"} class representing a decision table. See \code{\link{SF.asDecisionTable}}.}

\item{attrDescriptions}{a list containing possible values of attributes (columns) in \code{decision.table}. It usually corresponds to \code{attr(decision.table, "desc.attrs")}.}

\item{decisionIdx}{an integer value representing an index of the decision attribute.}

\item{qualityF}{a function used for computation of the quality of attribute subsets.
Currently, the following functions are included:
\itemize{
\item \code{X.entropy}: See \code{\link{X.entropy}}.
\item \code{X.gini}: See \code{\link{X.gini}}.
\item \code{X.nOfConflicts}: See \code{\link{X.nOfConflicts}}.
}}

\item{nAttrs}{an integer between 1 and the number of conditional attributes. It indicates
the attribute sample size for the Monte Carlo selection of candidating attributes.
If set to \code{NULL} (default) all attributes are used and the algorithm changes
to a standard greedy method for computation of decision reducts.}

\item{epsilon}{a numeric value between [0, 1) representing an approximate threshold. It
indicates whether to compute approximate reducts or not. If it equals 0 (the default)
a standard decision reduct is computed.}

\item{inconsistentDecisionTable}{logical indicating whether the decision table is suspected
       to be inconsistent.}
}
\value{
A class \code{"FeatureSubset"} that contains the following components:
\itemize{
\item \code{reduct}: a list representing a single reduct. In this case, it could be a superreduct or just a subset of features.
\item \code{type.method}: a string representing the type of method which is \code{"greedy.heuristic"}.
\item \code{type.task}: a string showing the type of task which is \code{"feature selection"}.
\item \code{model}: a string representing the type of model. In this case, it is \code{"RST"} which means rough set theory.
\item \code{epsilon}: the approximation threshold.
}
}
\description{
This function implements a greedy heuristic algorithm for computing decision reducts
(or approximate decision reducts) based on RST.
}
\details{
In this implementation, we provided some attribute subset quality measures which can be
passed to the algorithm by the parameter \code{qualityF}. Those measures
guide the computations in the search for a decision/approximated reduct. They are used to
assess amount of information gained after addition of an attribute. For example,
\code{X.entropy} corresponds to the information gain measure.

Additionally, this function can use the value of \code{epsilon} parameter in order to compute
\eqn{\epsilon}-approximate reducts. The \eqn{\epsilon}-approximate can be defined as an
irreducable subset of attributes \code{B}, such that:

\eqn{Quality_{\mathcal{A}}(B) \ge (1 - \epsilon)Quality_{\mathcal{A}}(A)},

where \eqn{Quality_{\mathcal{A}}(B)} is the value of a quality measure (see possible values
of the parameter \code{qualityF}) for an attribute subset \eqn{B} in decision table \eqn{\mathcal{A}}
and \eqn{\epsilon} is a numeric value between 0 and 1 expressing the approximation threshold.
A lot of monographs provide comprehensive explanations about this topics, for example
(Janusz and Stawicki, 2011; Slezak, 2002; Wroblewski, 2001) which are used as the references of this function.

Finally, this implementation allows to restrain the computational complexity of greedy
searching for decision reducts by setting the value of the parameter \code{nAttrs}. If this
parameter is set to a positive integer, the Monte Carlo method of selecting candidating
attributes will be used in each iteration of the algorithm.
}
\examples{
###################################################
## Example 1: Evaluate reduct and generate
##            new decision table
###################################################
data(RoughSetData)
decision.table <- RoughSetData$hiring.dt

## evaluate a single reduct
res.1 <- FS.greedy.heuristic.reduct.RST(decision.table, qualityF = X.entropy,
                                        epsilon = 0.0)

## generate a new decision table corresponding to the reduct
new.decTable <- SF.applyDecTable(decision.table, res.1)
}
\author{
Andrzej Janusz
}
\references{
A. Janusz and S. Stawicki, "Applications of Approximate Reducts to the Feature Selection Problem",
Proceedings of International Conference on Rough Sets and Knowledge Technology ({RSKT}), vol. 6954, p. 45 - 50 (2011).

D. Ślęzak, "Approximate Entropy Reducts", Fundamenta Informaticae, vol. 53, no. 3 - 4, p. 365 - 390 (2002).

J. Wróblewski, "Ensembles of Classifiers Based on Approximate Reducts", Fundamenta Informaticae, vol. 47, no. 3 - 4, p. 351 - 360 (2001).
}
\seealso{
\code{\link{FS.DAAR.heuristic.RST}} and \code{\link{FS.reduct.computation}}.
}

