#' @title Plots a response surface of a polynomial equation of second degree
#'
#' @description
#' Plots an RSA object, or a response surface with specified parameters
#'
#' @details
#' Each plot type has its distinctive advantages. The two-dimensional contour plot gives a clear view of the position of the principal axes and the stationary point. The 3d plot gives a three dimensional impression of the surface, allows overplotting of the original data points (in case an RSA object is provided), and allows the interactive adjustment of regression weights in the \code{\link{demoRSA}} function. The interactive plot allows rotating and exploring a three-dimensional surface with the mouse (nice for demonstration purposes).
#' If you want to export publication-ready plots, it is recommended to export it with following commands:
#' \code{p1 <- plot(r1, bw=TRUE)
#' trellis.device(device="cairo_pdf", filename="RSA_plot.pdf")
#' print(p1)
#' dev.off()}
#'
#' @aliases plotRSA
#'
#' @export
#' @param x Either an RSA object (returned by the \code{RSA} function), or the coefficient for the X predictor
#' @param y Y coefficient
#' @param x2 X^2 coefficient
#' @param y2 Y^2 coefficient
#' @param xy XY interaction coefficient
#' @param w W coefficient (for (un)constrained absolute difference model)
#' @param wx WX coefficient (for (un)constrained absolute difference model)
#' @param wy WY coefficient (for (un)constrained absolute difference model)
#' @param y3 Y^3 coefficient
#' @param x3 X^3 coefficient
#' @param xy2 XY^2 coefficient
#' @param x2y X^2Y coefficient
#' @param b0 Intercept
#' @param xlim Limits of the x axis
#' @param ylim Limits of the y axis
#' @param zlim Limits of the z axis
#' @param xlab Label for x axis
#' @param ylab Label for y axis
#' @param zlab Label for z axis
#' @param surface Method for the calculation of the surface z values. "predict" takes the predicted values from the model, "smooth" uses a thin plate smoother (function \code{Tps} from the \code{fields} package) of the raw data
#' @param lambda lambda parameter for the smoother. Default (NULL) means that it is estimated by the smoother function. Small lambdas around 1 lead to rugged surfaces, big lambdas to very smooth surfaces.
#' @param rotation Rotation of the 3d surface plot (when type == "3d")
#' @param label.rotation Rotation of the axis labls (when type == "3d")
#' @param gridsize Number of grid nodes in each dimension
#' @param bw Print surface in black and white instead of colors?
#' @param legend Print color legend for z values?
#' @param cex Font size factor for axes labels
#' @param type \code{3d} for 3d surface plot, \code{contour} for 2d contour plot, "interactive" for interactive rotatable plot. Shortcuts (i.e., first letter of string) are sufficient
#' @param points Should the original data points be overplotted?
#' @param model If x is an RSA object: from which model should the response surface be computed?
#' @param demo Do not change that parameter (internal use only)
#' @param fit Do not change that parameter (internal use only)
#' @param param Should the surface parameters a1 to a4 be shown on the plot? In case of a 3d plot a1 to a4 are printed on the upper left side; in case of a contour plot the principal axes are plotted.
#' @param axes A vector of strings specifying the axes that should be plotted. Can be any combination of c("LOC", "LOIC", "PA1", "PA2"). LOC = line of congruence, LOIC = line of incongruence, PA1 = first principal axis, PA2 = second principal axis
#' @param project Should the LOC, LOIC, etc. (as defined in parameter \code{axes}) be also plotted as a projection on the bottom of the cube?
#' @param link Link function to transform the z axes. Implemented are "identity" (no transformation; default), "probit", and "logit"
#' @param border Should a thicker border around the surface be plotted? Sometimes this border leaves the surrounding box, which does not look good. In this case the border can be suppressed by setting \code{border=FALSE}.
#' @param contour Should the contour lines be plotted on the 3d wireframe plot? (Parameter only relevant for \code{type="3d"})
#' @param hull Plot the convex hull enclosing the raw data points
#' @param SP.CI Plot the CI of the stationary point (only relevant for \code{type="contour"})
#' @param distance A vector of three values defining the distance of labels to the axes
#' @param tck A vector of three values defining the position of labels to the axes (see ?wireframe)
#' @param pal A palette for shading
#' @param ... Additional parameters passed to the plotting function (e.g., main="Title"). A useful title might be the R squared of the plotted model: main = as.expression(bquote(R^2==.(round(getPar(x, "r2", model="full"), 3))))
#'
#' @seealso \code{\link{demoRSA}}, \code{\link{RSA}}
#'
#' @examples
#' # Plot response surfaces from known parameters
#' # example of Edwards (2002), Figure 3
#' plotRSA(x=.314, y=-.118, x2=-.145, y2=-.102, xy=.299, b0=5.628)
#' plotRSA(x=.314, y=-.118, x2=-.145, y2=-.102, xy=.299, b0=5.628, type="c")
#'
#' # Plot response surface from an RSA object
#' set.seed(0xBEEF)
#' n <- 300
#' err <- 2
#' x <- rnorm(n, 0, 5)
#' y <- rnorm(n, 0, 5)
#' df <- data.frame(x, y)
#' df <- within(df, {
#' 	diff <- x-y
#' 	absdiff <- abs(x-y)
#' 	sqdiff <- (x-y)^2
#' 	z.diff <- diff + rnorm(n, 0, err)
#' 	z.abs <- absdiff + rnorm(n, 0, err)
#' 	z.sq <- sqdiff + rnorm(n, 0, err)
#' 	z.add <- diff + 0.4*x + rnorm(n, 0, err)
#' 	z.complex <- 0.4*x + - 0.2*x*y + + 0.1*x^2 - 0.03*y^2 + rnorm(n, 0, err)
#' })
#' 
#' r1 <- RSA(z.sq~x*y, df, models=c("sqdiff", "full", "IA"))
#' plot(r1)
#' plot(r1, points=TRUE, model="sqdiff")



#b0=0; x=0; y=0; x2=0; y2=0; xy=0; w=0; wx=0; wy=0;  zlim=NULL; xlim=c(-2, 2); ylim=c(-2, 2); rotation=list(x=-45, y=45, z=35); legend=TRUE; cex=1.2; type="3d"; points=TRUE; demo=FALSE; model="full"; 

#b0=-9; x=0; y=0; x2=0; y2=0; xy=0; w=0; wx=1; wy=-1;  zlim=NULL; xlim=c(-2, 2); ylim=c(-2, 2); rotation=list(x=-45, y=45, z=35); legend=TRUE; cex=1.2; type="3d"; points=TRUE; demo=FALSE; model="full"; fit=NULL; link="identity"; param=TRUE; gridsize=21;bw=FALSE; pal=NULL; axes=c("LOC", "LOIC", "PA1", "PA2"); distance=c(1, 1, 1); tck=c(1, 1, 1); xlab="X"; ylab="Y"; zlab="Z"; border=TRUE;

## old rotation
# rotation=list(x=-45, y=45, z=35), label.rotation=list(x=45, y=-25, z=94)
# distance=c(1, 1, 1), tck=c(1, 1, 1)

plotRSA <- function(x=NULL, y=0, x2=0, y2=0, xy=0, w=0, wx=0, wy=0, x3=0, xy2=0, x2y=0, y3=0, b0=0, xlim=NULL, ylim=NULL, zlim=NULL, xlab=NULL, ylab=NULL, zlab=NULL, surface="predict", lambda=NULL, rotation=list(x=-63, y=32, z=15), label.rotation=list(x=19, y=-40, z=92), gridsize=21, bw=FALSE, legend=TRUE, param=TRUE, axes=c("LOC", "LOIC", "PA1", "PA2"), project=FALSE,  cex=1.2, type="3d", points=FALSE, model="full", demo=FALSE, fit=NULL, link="identity", tck=c(1.5, 1.5, 1.5), distance=c(1.3, 1.3, 1.4), border=TRUE, contour=FALSE, hull=FALSE, SP.CI=FALSE, pal=NULL, ...) {
	
	if (!identical(xlim, ylim)) {warning("Axes dimensions are not equal. The visual diagonal is *not* the line of congruence! Consider choosing the same values for xlim and ylim.")}
	
	type <- match.arg(type, c("interactive", "3d", "contour"))
	surface <- match.arg(surface, c("predict", "smooth"))

	# take parameters from function parameters, or from model object
	if (demo == FALSE) {
		if (is.null(x)) {
			x <- 0
			fit <- NULL
			if (is.null(xlab)) xlab <- "X"
			if (is.null(ylab)) ylab <- "Y"
			if (is.null(zlab)) zlab <- "Z"
			
			if (is.null(xlim)) {xlim <- c(-2.1, 2.1)}
			if (is.null(ylim)) {ylim <- c(-2.1, 2.1)}
			
			hull <- FALSE
		} else	if (!is.null(x) & !is.null(attr(x, "class"))) {
			if (attr(x, "class") == "RSA" & demo==FALSE) {
				fit <- x
				C <- coef(fit$models[[model]])
				if (fit$models[[model]]@Options$estimator != "DWLS") {
					b0 <- as.numeric(ifelse(is.na(C[paste0(fit$DV, "~1")]), b0, C[paste0(fit$DV, "~1")]))
					} else {
						# the threshold is the negative of the intercept ...
						b0 <- -as.numeric(ifelse(is.na(C[paste0(fit$DV, "|t1")]), b0, C[paste0(fit$DV, "|t1")]))
					}
				x <- as.numeric(ifelse(is.na(C["b1"]), 0, C["b1"]))
				y <- as.numeric(ifelse(is.na(C["b2"]), y, C["b2"]))
				x2 <- as.numeric(ifelse(is.na(C["b3"]), x2, C["b3"]))
				y2 <- as.numeric(ifelse(is.na(C["b5"]), y2, C["b5"]))
				xy <- as.numeric(ifelse(is.na(C["b4"]), xy, C["b4"]))
				w <- as.numeric(ifelse(is.na(C["b6"]), w, C["b6"]))
				wx <- as.numeric(ifelse(is.na(C["b7"]), wx, C["b7"]))
				wy <- as.numeric(ifelse(is.na(C["b8"]), wy, C["b8"]))
				
				# cubic parameters
				x3 <- as.numeric(ifelse(is.na(C["b9"]), x3, C["b9"]))
				xy2 <- as.numeric(ifelse(is.na(C["b10"]), xy2, C["b10"]))
				x2y <- as.numeric(ifelse(is.na(C["b11"]), x2y, C["b11"]))
				y3 <- as.numeric(ifelse(is.na(C["b12"]), y3, C["b12"]))
		
				if (is.null(xlim)) {
					xlim <- c(min(fit$data[, fit$IV1], na.rm=TRUE), max(fit$data[, fit$IV1], na.rm=TRUE))
					# expand range by 20% at each end
					xlim[1] <- xlim[1]*ifelse(xlim[1]<0, 1.1, 0.9)
					xlim[2] <- xlim[2]*ifelse(xlim[2]<0, 0.9, 1.1)
				}
				
				if (is.null(ylim)) {
					ylim <- c(min(fit$data[, fit$IV2], na.rm=TRUE), max(fit$data[, fit$IV2], na.rm=TRUE))
					ylim[1] <- ylim[1]*ifelse(ylim[1]<0, 1.1, 0.9)
					ylim[2] <- ylim[2]*ifelse(ylim[2]<0, 0.9, 1.1)
				}
								
				# for the correct visual diagonal: same range for X and Y
				xlim[1] <- ylim[1] <- min(xlim[1], ylim[1])
				xlim[2] <- ylim[2] <- max(xlim[2], ylim[2])
			} else {
				stop("Unknown object provided.")
				fit <- NULL
			}
		} else {
			fit <- NULL
			hull <- FALSE
			if (is.null(xlab)) xlab <- "X"
			if (is.null(ylab)) ylab <- "Y"
			if (is.null(zlab)) zlab <- "Z"
			
			if (is.null(xlim)) {xlim <- c(-2.1, 2.1)}
			if (is.null(ylim)) {ylim <- c(-2.1, 2.1)}
		}
	}
	
	if (!is.null(fit)) {
		if (is.null(xlab)) xlab <- fit$IV1
		if (is.null(ylab)) ylab <- fit$IV2
		if (is.null(zlab)) zlab <- fit$DV
	}
	
	if (is.null(fit) & surface == "smooth") {
		warning("Smoothing only works if an RSA object is provided! Reverting to surface = 'predict'")
		surface <- "predict"
	}
	
	C <- c(x, y, x2, y2, xy, w, wx, wy,x3, xy2, x2y, y3)
	
	if (!model %in% c("absunc", "absdiff")) {
		if (!is.null(fit) & model != "cubic" & model != "null") {
			SP <- RSA.ST(fit, model=model)
		} else {
			SP <- RSA.ST(x=x, y=y, xy=xy, x2=x2, y2=y2)
		}
		SP.text <- paste0("a", 1:4, ": ", round(SP$SP$estimate, 2), sig2star(SP$SP$p.value), collapse="    ")
	} else {
		SP <- NULL
		param <- FALSE
		SP.text <- ""
	}
		
	# build data set
	grid <- gridsize
	new <- data.frame(x = rep(seq(xlim[1], xlim[2], length.out=grid), grid), y = rep(seq(ylim[1], ylim[2], length.out=grid), each=grid))
	new2 <- add.variables(z~x+y, new)
		
	# calculate z values
	if (surface == "predict") {
		new2$z <- b0 + colSums(C*t(new2[, c(1:5, 9:11, 15:18)]))
	}
	if (surface == "smooth") {
		library(fields)
		tpsfit <- Tps(fit$data[, c(fit$IV1, fit$IV2)], fit$data[, fit$DV], scale.type="unscaled", lambda=lambda)
		new2$z <- predict(tpsfit, new[, c("x", "y")])
	}
	
	# impose link functions
	logit <- function (x) {log(x/(1-x))}
	invlogit <- function (x) {1/(1+exp(-x))}
	link <- match.arg(link, c("identity", "logit", "probit"))
	if (link == "probit") {
		z.trans <- 1.7 * new2$z
		new2$z <- invlogit(z.trans)
	}
	if (link == "logit") {
		new2$z <- invlogit(new2$z)
	}

	# determine zlim
	if (!is.null(fit) & demo==FALSE & is.null(zlim)) {
		zlim <- c(min(min(new2$z, na.rm=TRUE), min(fit$data[, fit$DV], na.rm=TRUE)), max(max(new2$z, na.rm=TRUE), max(fit$data[, fit$DV], na.rm=TRUE)))
	} else {
		if (is.null(zlim)) zlim <- c(min(new2$z), max(new2$z))
	}
	zlim.final <- zlim
	
	## Plots
	
	if (bw == FALSE) {
		# RdYlGn palette
		if (is.null(pal)) {
			pal <- c("#A50026","#D73027","#F46D43","#FDAE61","#FEE08B","#FFFFBF","#D9EF8B","#A6D96A","#66BD63","#1A9850","#006837")
		}
		gridCol <- ifelse(contour==TRUE, "grey60", "grey30")
	} else {
		if (is.null(pal)) {
			pal <- colorRampPalette(c("#FFFFFF", "#888888", "#333333"), bias=0.8)(7)
		}
		gridCol <- ifelse(contour==TRUE, "grey80", "grey30")
	}
	if (length(pal) < 2) {legend <- FALSE}
		
		
	# calculate hull
	if (hull==TRUE) {
		h1 <- chull(data.frame(x=fit$data[, fit$IV1], y=fit$data[, fit$IV2]))
		HULL <- data.frame(X=fit$data[h1, fit$IV1], Y=fit$data[h1, fit$IV2], Z0=fit$data[h1, fit$DV])
		HULL <- rbind(HULL, HULL[1, ])
		# calculate predicted values for hull points (so thge hull lies on the surface)
		HULL3 <- add.variables(Z0~X+Y, HULL[, c("X", "Y")])
		HULL3$Z <- b0 + colSums(C*t(HULL3[, c(1:5, 9:11, 15:18)]))
	}
	
		
	## ======================================================================
	## Interactive plot
	## ======================================================================
	
	if (type == "interactive") {
		library(rgl)
		
		P <- list(x=seq(xlim[1], xlim[2], length.out=grid), y=seq(ylim[1], ylim[2], length.out=grid))
		DV2 <- matrix(new2$z, nrow=grid, ncol=grid, byrow=FALSE)
		R <- range(DV2)
		col2 <- as.character(cut(1:(R[2] - R[1] + 1), breaks=length(pal), labels=pal))
		
		open3d()
		rgl.viewpoint(-30, -90)
		rgl.light(theta = 0, phi = 90, viewpoint.rel = TRUE, ambient = "#FF0000", diffuse = "#FFFFFF", specular = "#FFFFFF")
		persp3d(P$x, P$y, DV2, xlab = xlab, ylab = ylab, zlab = zlab, color=col2[DV2 - R[1] + 1])

	    contours <- contourLines(P, z=DV2)
	     for (i in 1:length(contours)) {
			 with(contours[[i]], lines3d(x, y, level, col="black"))
		 }
		
		if (points==TRUE) {
			points3d(fit$data[, c(fit$IV1, fit$IV2, fit$DV)])
		}
		
		p1 <- NULL	# no plot object is returned	
	}	
	
	
	
	## ======================================================================
	## Wireframe plot
	## ======================================================================
	
		
	if (type == "3d") {
		library(lattice)
		library(gridExtra)
				
			mypanel2 <- function(x, y, z, xlim, ylim, zlim, xlim.scaled, ylim.scaled, zlim.scaled, axes, x.points=NULL, y.points=NULL, z.points=NULL, SPs="", ...) {
				
				
			   # rescale absolute x, y, and z values so that they fit into the box
			   RESCALE <- function(n) {
				  X2 <- xlim.scaled[1] + diff(xlim.scaled) * (n$X - xlim[1]) / diff(xlim)
	              Y2 <- ylim.scaled[1] + diff(ylim.scaled) * (n$Y - ylim[1]) / diff(ylim)
	              Z2 <- zlim.scaled[1] + diff(zlim.scaled) * (n$Z - zlim[1]) / diff(zlim)
				  df <- data.frame(X=X2, Y=Y2, Z=Z2)
				  df <- df[df$X > min(xlim.scaled) & df$X < max(xlim.scaled) & df$Y > min(ylim.scaled) & df$Y < max(ylim.scaled) &  df$Z > min(zlim.scaled) & df$Z < max(zlim.scaled), ]
				  return(df)
			   }
				
			
			# ---------------------------------------------------------------------
			# 1. Projection on bottom of cube
			  if (project==TRUE) {
				  for (a in axes) {
					  #print(paste("Plotting projection of", names(a)))
					  a0 <- RESCALE(getIntersect2(p0=a$p0, p1=a$p1, Z=zlim.final[1] + .01))
		              panel.3dscatter(x = a0$X, y = a0$Y, z = a0$Z, xlim = xlim, ylim = ylim, zlim = zlim,
		                              xlim.scaled = xlim.scaled, ylim.scaled = ylim.scaled, zlim.scaled = zlim.scaled, type="l", col.line=a$col, lty=a$lty, lwd=2, ...)
				  } 
			  }
		  	
				
			   # ---------------------------------------------------------------------
			   # 2. Borders, back part
			   
			   
					if (border==TRUE) {
						  # Make boundary of grid a bit thicker
						  box1 <- new2[new2$y == max(new2$y), ]
						  box2 <- new2[new2$y == min(new2$y), ]
						  box3 <- new2[new2$x == max(new2$x), ]
						  box4 <- new2[new2$x == min(new2$x), ]
						  box <- rbind(data.frame(box1, side=1), data.frame(box2, side=2), data.frame(box3, side=3), data.frame(box4, side=4))
						  
			              x.box <- xlim.scaled[1] + diff(xlim.scaled) * (box$x - xlim[1]) / diff(xlim)
			              y.box <- ylim.scaled[1] + diff(ylim.scaled) * (box$y - ylim[1]) / diff(ylim)
			              z.box <- zlim.scaled[1] + diff(zlim.scaled) * (box$z - zlim[1]) / diff(zlim)
						  
						  # plot the back lines of the border
			              panel.3dscatter(x = x.box[box$side==1], y = y.box[box$side==1], z = z.box[box$side==1], xlim = xlim, ylim = ylim, zlim = zlim, xlim.scaled = xlim.scaled, ylim.scaled = ylim.scaled, zlim.scaled = zlim.scaled, type="l", col.line=gridCol, lwd=4, ...)
						  panel.3dscatter(x = x.box[box$side==3], y = y.box[box$side==3], z = z.box[box$side==3], xlim = xlim, ylim = ylim, zlim = zlim, xlim.scaled = xlim.scaled, ylim.scaled = ylim.scaled, zlim.scaled = zlim.scaled, type="l", col.line=gridCol, lwd=4, ...)
					  }
				
					  # ---------------------------------------------------------------------
					  # 3. the surface
						  panel.3dwire(x = x, y = y, z = z, xlim = xlim, ylim = ylim, zlim = zlim,
		                           xlim.scaled = xlim.scaled, ylim.scaled = ylim.scaled, zlim.scaled = zlim.scaled,
								   col=gridCol,  lwd=0.3, ...)
								   
								   
					   
					# ---------------------------------------------------------------------
					# 4. plot of LOC and LOIC, and other axes
						  for (a in axes) {
							  a0 <- RESCALE(getIntersect2(p0=a$p0, p1=a$p1))
				              panel.3dscatter(x = a0$X, y = a0$Y, z = a0$Z, xlim = xlim, ylim = ylim, zlim = zlim,
				                              xlim.scaled = xlim.scaled, ylim.scaled = ylim.scaled, zlim.scaled = zlim.scaled,
											  type="l", col.line=a$col, lty=a$lty, lwd=2, ...)
						  }   
						  
						  
					# ---------------------------------------------------------------------
					# 4b: The convex hull, if requested
					
					if (hull==TRUE) {
					  hull.rescale <- RESCALE(HULL3)
		              panel.3dscatter(x = hull.rescale$X, y = hull.rescale$Y, z = hull.rescale$Z, xlim = xlim, ylim = ylim, zlim = zlim,
		                              xlim.scaled = xlim.scaled, ylim.scaled = ylim.scaled, zlim.scaled = zlim.scaled,
									  type="l", col.line="grey20", lty="dashed", lwd=2, ...)
					}	  

					# ---------------------------------------------------------------------
					# 5. Borders, front part	  
									   
					   if (border==TRUE) {
 						  # plot the front boundary lines
 			              panel.3dscatter(x = x.box[box$side==2], y = y.box[box$side==2], z = z.box[box$side==2], xlim = xlim, ylim = ylim, zlim = zlim, xlim.scaled = xlim.scaled, ylim.scaled = ylim.scaled, zlim.scaled = zlim.scaled, type="l", col.line=gridCol, lwd=3, ...)
						  panel.3dscatter(x = x.box[box$side==4], y = y.box[box$side==4], z = z.box[box$side==4], xlim = xlim, ylim = ylim, zlim = zlim, xlim.scaled = xlim.scaled, ylim.scaled = ylim.scaled, zlim.scaled = zlim.scaled, type="l", col.line=gridCol, lwd=3, ...)
					   }
										  
						  if (param == TRUE) {
							  grid.text(SPs, .02, .95, just="left")
						  }  
						  
						  
					# ---------------------------------------------------------------------
					# 6. Raw data points scatter plot	  
			              x2 <- xlim.scaled[1] + diff(xlim.scaled) * (x.points - xlim[1]) / diff(xlim)
			              y2 <- ylim.scaled[1] + diff(ylim.scaled) * (y.points - ylim[1]) / diff(ylim)
			              z2 <- zlim.scaled[1] + diff(zlim.scaled) * (z.points - zlim[1]) / diff(zlim)
				  		if (points==TRUE & !is.null(fit)) {
  			              panel.3dscatter(x = x2, y = y2, z = z2, xlim = xlim, ylim = ylim, zlim = zlim,
  			                              xlim.scaled = xlim.scaled, ylim.scaled = ylim.scaled, zlim.scaled = zlim.scaled,
  										  pch="o", col="black", ...)
				  		}
						
						
						
					# ---------------------------------------------------------------------
					# 7. plot contour lines:
						if (contour == TRUE) {
							cs <- ggplot(new2, aes_string(x="x", y="y", fill="z", z="z")) + stat_contour(bins=ifelse(length(pal)>1, length(pal)+1, 8))
							cLines <- ggplot_build(cs)
						
							C0 <- cLines$data[[1]][, c("x", "y", "level", "group")]
							colnames(C0) <- c("X", "Y", "Z", "group")
							for (cL in C0$group) {
							  C1 <- RESCALE(C0[C0$group==cL, c("X", "Y", "Z")])
				              panel.3dscatter(x = C1$X, y = C1$Y, z = C1$Z, xlim = xlim, ylim = ylim, zlim = zlim,
				                              xlim.scaled = xlim.scaled, ylim.scaled = ylim.scaled, zlim.scaled = zlim.scaled,
											  type="l", col.line="grey40", lty="solid", lwd=1, ...)
							}
						}
						
				}
			


				# local function: compute the surface line, defined by a line on the X-Y plane (p0 = intercept, p1=slope)
				getIntersect2 <- function(p0, p1, Z=NULL) {
					X <- seq(min(xlim), max(xlim), length.out=grid*2)
					Y <- p0 + p1*X
					n <- data.frame(X, Y)
					n2 <- add.variables(z~X+Y, n)
					n2$Z <- b0 + colSums(c(x, y, x2, y2, xy)*t(n2[, c(1:5)]))
					if (!is.null(Z)) n2$Z <- Z
					return(n2[, c("X", "Y", "Z")])
				}
				
				axesList <- list()
				if ("LOC" %in% axes) {axesList[["LOC"]]  <- list(p0=0, p1=1, lty="solid", col="grey")}
				if ("LOIC" %in% axes) {axesList[["LOIC"]] <- list(p0=0, p1=-1, lty="dotted", col="grey")}
				if ("PA1" %in% axes) {
					if (x2 == y2) {
						# print("Adjusting x2 to print PA1")
						SP2 <- RSA.ST(x=x, y=y, xy=xy, x2=x2 + .001, y2=y2)
						axesList[["PA1"]] <- list(p0=SP2$p10, p1=SP2$p11, lty="solid", col="black")
						} else {
							axesList[["PA1"]] <- list(p0=SP$p10, p1=SP$p11, lty="solid", col="black")
						}					
				}
				if ("PA2" %in% axes) {
					if (x2 == y2) {
						# print("Adjusting x2 to print PA2")
						SP2 <- RSA.ST(x=x, y=y, xy=xy, x2=x2 + .001, y2=y2)
						axesList[["PA2"]] <- list(p0=SP2$p20, p1=SP2$p21, lty="dotted", col="black")
					} else {
						axesList[["PA2"]] <- list(p0=SP$p20, p1=SP$p21, lty="dotted", col="black")	
					}
				}			
				
				pad <- 0	# pad controls them margin around the figure
				
			if (points==FALSE) {
				p1 <- wireframe(z ~ x*y, new2,  drape=TRUE, 
					scales 	= list(arrows = FALSE, cex=cex, col = "black", font = 1, tck=tck, distance=distance), 
					xlab	= list(cex=cex, label=xlab, rot=label.rotation[["x"]]), 
					ylab	= list(cex=cex, label=ylab, rot=label.rotation[["y"]]), 
					zlab	= list(cex=cex, label=zlab, rot=label.rotation[["z"]]), zlim=zlim, 
					screen	= rotation, colorkey=legend, 
					at		= seq(zlim[1], zlim[2], length.out=length(pal)-1), col.regions=pal, 
					par.settings = list(
						axis.line = list(col = "transparent"), 
						layout.heights = list(top.padding=pad, bottom.padding=pad), 
						layout.widths=list(left.padding=pad, right.padding=pad)), 
					axes	= axesList, 
					SPs		= SP.text, 
					panel.3d.wireframe = mypanel2, ...)
								
				#p1
			} else {
				p1 <- wireframe(z ~ x*y, new2,  drape=TRUE, 
					scales 	= list(arrows = FALSE, cex=cex, col = "black", font = 1, tck=tck, distance=distance), 
					xlab	= list(cex=cex, label=xlab, rot=label.rotation[["x"]]), 
					ylab	= list(cex=cex, label=ylab, rot=label.rotation[["y"]]), 
					zlab	= list(cex=cex, label=zlab, rot=label.rotation[["z"]]), zlim=zlim, 
					screen	= rotation, colorkey=legend, 
					at		= seq(zlim[1], zlim[2], length.out=length(pal)-1), col.regions=pal,
					par.settings = list(
						axis.line = list(col = "transparent"), 
						layout.heights = list(top.padding=pad, bottom.padding=pad), 
						layout.widths=list(left.padding=pad, right.padding=pad)), 
					axes	= axesList, 
					SPs		= SP.text, 
					panel.3d.wireframe = mypanel2,
					x.points=fit$data[, fit$IV1], y.points=fit$data[, fit$IV2], z.points=fit$data[, fit$DV], ...)
			}
				
	}  # of type == "3d"
	
	
	
	## ======================================================================
	## Contour plot
	## ======================================================================
	
	if (type == "contour") {
		if (!all(C == 0)) {
			library(ggplot2)
			library(RColorBrewer)
			
			p1 <- ggplot(new2, aes_string(x="x", y="y", fill="z", z="z")) + geom_tile() + scale_fill_gradientn(zlab, colours=pal, limits=c(zlim[1], zlim[2])) + theme_bw() + theme(aspect.ratio=1) + xlab(xlab) + ylab(ylab)
			
			if (legend==FALSE) {
				p1 <- p1 + guides(fill=FALSE)
			}
	
			p1 <- p1 + stat_contour(bins=40, alpha=.4)
				
			# (in)congruence lines
			if ("LOC" %in% axes) {
				p1 <- p1 + geom_abline(aes(xintercept=0, slope=1), color="grey")
			}
			if ("LOIC" %in% axes) {
				p1 <- p1 + geom_abline(aes(xintercept=0, slope=-1), linetype="dotted", color="grey50")
			}
			if (("PA1" %in% axes) & !any(is.na(SP[c("p10", "p11")]))) {
				p1 <- p1 + geom_abline(data=data.frame(SP[c("p10", "p11")]), aes_string(intercept="p10", slope="p11"))
			}
			if (("PA2" %in% axes) & !any(is.na(SP[c("p20", "p21")]))) {
				p1 <- p1+ geom_abline(data=data.frame(SP[c("p20", "p21")]), aes_string(intercept="p20", slope="p21"), linetype="dotted")
			}
		
			if (param==TRUE & !any(is.na(SP[c("X0", "Y0")]))) {
				p1 <- p1 + annotate("point", x=SP$X0, y=SP$Y0, z=max(new2$z))
			}
				
				
			if (points==TRUE & !is.null(fit)) {
				p1 <- p1 + annotate("point", x=fit$data[, fit$IV1], y=fit$data[, fit$IV2], color="grey20", size=1.5)
			}
			
			if (hull==TRUE & !is.null(fit)) {
				p1 <- p1 + annotate("path", x=HULL$X, y=HULL$Y, linetype="dashed", color="grey10")
			}
			
			# plot CI of SP
			if (param==TRUE & SP.CI==TRUE & !is.null(fit)) {
				PAR <- getPar(fit, "coef", model=model)
				p1 <- p1 + annotate("errorbar", x=SP$X0, y=SP$Y0, ymin=PAR[PAR$label=="Y0", "ci.lower"], ymax=PAR[PAR$label=="Y0", "ci.upper"], z=max(new2$z), width=.3)
				p1 <- p1 + annotate("errorbarh", x=SP$X0, y=SP$Y0, xmin=PAR[PAR$label=="X0", "ci.lower"], xmax=PAR[PAR$label=="X0", "ci.upper"], z=max(new2$z), height=.3)
			}
			
		}
	}
	
	
	return(p1)
}


#' @S3method plot RSA

plot.RSA <- function(x, ...) {
	plotRSA(x, ...)
}

#plotRSA(r1, type="i")
#plotRSA(x=.05, x2=.1, xy=.20, type="3", gridsize=21)

#demoRSA(x=.625, y=.519, x2=-.196, xy=.285, y2=-.167)
#demoRSA(x=.625, y=.519, x2=-.196, xy=.285, y2=-.167, type="c")
#
#plotRSA(fit=r1, type="3d", points=TRUE, rotation=list(x=-58, y=50, z=26))
#plotRSA(x=.05, x2=.1, xy=.20, rotation=list(x=-50, y=58, z=36), legend=FALSE, type="c")
#plotRSA(x=.05, x2=.1, xy=.20, rotation=list(x=-50, y=58, z=36), legend=FALSE, type="3d")
#
## exampe of Edwards, 2002, Figure 3
#p1 <- plotRSA(b0=5.628, x=.314, y=-.118, x2=-.145, y2=-.102, xy=.299, legend=FALSE, type="3d")
#plotRSA(b0=5.628, x=.314, y=-.118, x2=-.145, y2=-.102, xy=.299, legend=FALSE, type="c")

#RSA.ST(coef=c(x=.314, y=-.118, x2=-.145, xy=.299, y2=-.102))


## Minimal example
#x <- y <- z <- c(1,2,3)
#df <- data.frame(x, y, z)
#wireframe(z ~ x*y, df,  scales = list(arrows = FALSE, col = "black", font = 1, tck = c(1, 1, 1), distance =c(1, 1, 1)))
#
#
#xyplot(z ~ x, df,  scales = list(arrows = FALSE, cex=cex, col = "black", font = 1, tck=tck), par.settings=list(axis.components=list(left=list(pad1=2))))