\name{ROSE.eval}
\alias{ROSE.eval}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Evaluation of learner accuracy by ROSE 
}
\description{
Given a classifier and a set of data, this function exploits ROSE 
generation of synthetic samples to provide holdout, bootstrap or leave-one-out 
cross-validation estimates of a specified accuracy measure.
}
\usage{
ROSE.eval(formula, data, learner, acc.measure="auc", extr.pred=NULL, 
          method.assess="holdout", B=100, control.rose=list(),
          control.learner=list(), control.predict=list(), 
          control.accuracy=list(), trace=FALSE, subset, na.action, seed)}
\arguments{
  \item{formula}{ 
	An object of class \code{\link{formula}} (or one that can be coerced to that class). 
	The specification of the formula must be suited for the selected classifier. 
	See \code{\link{ROSE}} and the ``Note'' below for information about interaction 
   among predictors or their transformations.}
  \item{data}{ 
	An optional data frame, list or environment (or object
	coercible to a data frame by \code{as.data.frame}) in which 
   to preferentially interpret ``formula''. 
   If not specified, the variables are taken from ``environment(formula)''.
    }
  \item{learner}{
	A standard defined R function to fit a classifier having 
	\code{\link{formula}} as a mandatory argument and returning an object whose 
	class is associated to a \code{\link{predict}} method. see Examples below.
	}  
  \item{acc.measure}{
  One among \code{c("auc", "precision", "recall", "F")}, it defines the 
  accuracy measure to be estimated. Function \code{\link{roc.curve}} is internally called 
  when \code{auc="auc"} while the other options entail an internal call of function \cr 
  \code{\link{accuracy.meas}}. Default value is \code{"auc"}.  
  }
  \item{extr.pred}{
    An optional function that extracts from the output of a \code{predict}
    function the vector of predicted values. 
    If not specified, the value returned by ``predict'' is used. See Examples below.
    }
  \item{method.assess}{
  One among \code{c("holdout", "L1OCV", "BOOT")}, it is the method used 
  for model assessment. When \code{"holdout"} is chosen, the learner is fitted
  on one ROSE sample and tested on the data provided in \code{formula}. 
  \code{"L1OCV"} stands for ``leave-1-out cross validation": for each observation
  \eqn{i}, a learner is estimated on a ROSE sample built on the provided data but
  row \eqn{i} and tested on row \eqn{i}. \code{"BOOT"} estimates the accuracy measure by fitting a
  learner on \code{B} ROSE samples and testing each of them on the provided data.  
  }
  \item{B}{
  The number of bootstrap replications to set when \code{method.assess="BOOT"}. 
  Ignored otherwise. Default value is 100.
  }
 \item{control.learner}{
  Further arguments to be passed to \code{learner}}
  \item{control.rose}{
  Optional arguments to be passed to \code{\link{ROSE}}. 
  }
  \item{control.predict}{
  Further arguments to be passed to \code{\link{predict}}.
  }
  \item{control.accuracy}{
  Optional arguments to be passed to either \code{\link{roc.curve}} or \code{\link{accuracy.meas}}
  depending on the selected accuracy measure.
  }
  \item{trace}{
  logical, if \code{TRUE} traces information on the progress of model 
  assessment (number of bootstrap or cross validation iterations performed). 
  }
  \item{subset}{
 An optional vector specifying a subset of observations to be used in the sampling and learning process.
 } 
 \item{na.action}{
 A function which indicates what should happen when the data contain 'NA's.  
 The default is set by the 'na.action' setting of 'options'.
}
  \item{seed}{
A single value, interpreted as an integer, recommended to specify seeds and keep trace of the generated ROSE 
  sample/es.
  }
}
\details{
This function estimates a measure of accuracy of a classifier specified by the user
by using either holdout, cross-validation, or bootstrap estimators.
Operationally, the classifier is trained over synthetic data generated by ROSE and
then evaluated on the original data. 

Whatever accuracy measure and estimator are chosen, the \emph{true} accuracy depends 
on the probability distribution underlying the training data. This is clearly affected by the imbalance 
and its estimation is then regulated by argument \code{control.rose}. 
A default setting of the arguments (that is, \code{p=0.5}) entails the estimation of the learner accuracy 
conditional to a balanced training set. In order to estimate the accuracy of a learner fitted on unbalanced data, 
the user may set argument \code{p} of \code{control.rose} to the proportion of
positive examples in the observed sample. See Example 2 below and, for further details, Menardi and Torelli (2012).
 
To the aim of a grater flexibility, the function is not linked to the
 use of a specific learner and works virtually with any classifier called by 
 \code{nameclassifier(formula=formula, data)} and returning an object whose
 class is associated to a \code{predict} method. However, since \code{predict}
 returns are very etherogeneous, the user is allowed to define some function 
 \code{extr.pred} which extracts from the output of \code{predict} 
 the desired vectors of predicted values. See ``Examples'' below.}

\value{
A list whose elements are:
\item{Call}{The matched call.}
\item{method}{The selected method for model assessment.}
\item{measure}{The selected measure to evaluate accuracy.}
\item{acc }{The vector of the estimated measure of accuracy. It has length \eqn{1} if \cr \code{method.assess="holdout"}, 
or \code{method.assess="L1OCV"} and length \code{B} \cr
if \code{method.assess="BOOT"}, corresponding to the bootstrap distribution of the accuracy
estimator. 
}
%% ...
}
\references{
Menardi, G. and Torelli, N. (2012). Training and assessing classification rules with imbalanced data. \emph{Data Mining and Knowledge Discovery
}, DOI:10.1007/s10618-012-0295-5, to appear.
}

\section{Note}{
The function allows the user to include in the formula transformations of predictors or 
interactions among them. ROSE samples are generated on the original data and tranformations 
or interactions are ignored. These are then retrieved in fitting the classifier, provided that 
the selected learner function can handle them. See also ``Warning'' in \code{\link{ROSE}}.
}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
\code{\link{ROSE}}, \code{\link{roc.curve}}, \code{\link{accuracy.meas}}.
}
\examples{

#2-dimensional data
#loading data
data(sefihy)

# in the following examples 
# use of a small subset of observations only --> argument subset
# and use of two predictors only for illustrative purpose
dat <- sefihy.train[,c(1,2,11)]

table(dat$cls)

##Example 1
# classification with logit model
# arguments to glm are passed through control.learner
# leave-one-out cross-validation estimate of auc of classifier trained on balanced data
ROSE.eval(cls~., data=dat, glm, subset=c(1:60, 950:1000), method.assess="L1OCV",
          control.learner=list(family=binomial), seed=1)

\dontrun{
##Example 2
# classification with decision tree 
# require package rpart
library(rpart)

# function is needed to extract predicted probability of cls 1 
f.pred.rpart <- function(x) x[,2]

# holdout estimate of auc of two classifiers

# first classifier trained on ROSE unbalanced sample
# proportion of rare events in original data
p <- (table(dat$cls)/sum(table(dat$cls)))[2]
ROSE.eval(cls~., data=dat, rpart, subset=c(1:60, 950:1000), control.rose=list(p = p),
          extr.pred=f.pred.rpart, seed=1)

# second classifier trained on ROSE balanced sample
# optional arguments to plot the roc.curve are passed through control.accuracy 
ROSE.eval(cls~., data=dat, rpart, subset=c(1:60, 950:1000), control.rose=list(p = 0.5),
          control.accuracy = list(add.roc = TRUE, col = 2), extr.pred=f.pred.rpart, seed=1)

##Example 3
# classification with linear discriminant analysis
library(MASS)

# function is needed to extract the predicted values from predict.lda
f.pred.lda <- function(z) z$posterior[,2]

# bootstrap estimate of precision of learner trained on balanced data
prec.distr <- ROSE.eval(cls~., data=dat, lda, subset=c(1:60, 950:1000), 
                        extr.pred=f.pred.lda, acc.measure="precision",
                        method.assess="BOOT", B=100, trace=TRUE)

summary(prec.distr)

##Example 4
# compare auc of classification with neural network
# with auc of classification with tree 
# require package nnet
# require package tree

library(nnet)
library(tree)

# optional arguments to nnet are passed through control.learner 
ROSE.eval(cls~., data=dat, nnet, subset=c(1:60, 950:1000), method.assess="holdout",
          control.learn=list(size=1), seed=1)

# optional arguments to plot the roc.curve are passed through control.accuracy
# function is needed to extract predicted probability of class 1 
f.pred.rpart <- function(x) x[,2] 
f.pred.tree <- function(x) x[,2] 
ROSE.eval(cls~., data=dat, tree, subset=c(1:60, 950:1000), method.assess="holdout",
          extr.pred=f.pred.tree, control.acc=list(add=TRUE, col=2), seed=1)
}
}

\keyword{ bootstrap }

% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
%\keyword{ ~kwd1 }
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line

