% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/redcap-project-info-read.R
\name{redcap_project_info_read}
\alias{redcap_project_info_read}
\title{Export project information.}
\usage{
redcap_project_info_read(
  redcap_uri,
  token,
  http_response_encoding = "UTF-8",
  locale = readr::default_locale(),
  verbose = TRUE,
  config_options = NULL,
  handle_httr = NULL
)
}
\arguments{
\item{redcap_uri}{The
\href{https://en.wikipedia.org/wiki/Uniform_Resource_Identifier}{uri}/url
of the REDCap server
typically formatted as "https://server.org/apps/redcap/api/".
Required.}

\item{token}{The user-specific string that serves as the password for a
project.  Required.}

\item{http_response_encoding}{The encoding value passed to
\code{\link[httr:content]{httr::content()}}.  Defaults to 'UTF-8'.}

\item{locale}{a \code{\link[readr:locale]{readr::locale()}} object to specify preferences like
number, date, and time formats.  This object is passed to
\code{\link[readr:read_delim]{readr::read_csv()}}.  Defaults to \code{\link[readr:locale]{readr::default_locale()}}.}

\item{verbose}{A boolean value indicating if \code{message}s should be printed
to the R console during the operation.  The verbose output might contain
sensitive information (\emph{e.g.} PHI), so turn this off if the output might
be visible somewhere public. Optional.}

\item{config_options}{A list of options passed to \code{\link[httr:POST]{httr::POST()}}.
See details at \code{\link[httr:httr_options]{httr::httr_options()}}. Optional.}

\item{handle_httr}{The value passed to the \code{handle} parameter of
\code{\link[httr:POST]{httr::POST()}}.
This is useful for only unconventional authentication approaches.  It
should be \code{NULL} for most institutions.  Optional.}
}
\value{
Currently, a list is returned with the following elements:
\itemize{
\item \code{data}: An R \code{\link[tibble:tibble]{tibble::tibble()}} of all data access groups of the project.
\item \code{success}: A boolean value indicating if the operation was apparently
successful.
\item \code{status_codes}: A collection of
\href{https://en.wikipedia.org/wiki/List_of_HTTP_status_codes}{http status codes},
separated by semicolons.  There is one code for each batch attempted.
\item \code{outcome_messages}: A collection of human readable strings indicating the
operations' semicolons.  There is one code for each batch attempted.  In an
unsuccessful operation, it should contain diagnostic information.
\item \code{elapsed_seconds}: The duration of the function.
}
}
\description{
This function exports some of the basic attributes of a given
REDCap project, such as the project's title, if it is longitudinal,
if surveys are enabled, the time the project was created and moved to
production.  Returns a \code{\link[tibble:tibble]{tibble::tibble()}}.
}
\details{
\strong{Timezones}

Several datetime variables are returned, such as the project's
\code{creation_time}.  For the time to be meaningful, you'll need to set
the time zone because this function uses \code{\link[readr:read_delim]{readr::read_csv()}},
which assigns
\href{https://en.wikipedia.org/wiki/Coordinated_Universal_Time}{UTC}
if no timezone is specified.  Find your server's location listed in
\code{\link[base:timezones]{base::OlsonNames()}}, and pass it to readr's \code{\link[readr:locale]{readr::locale()}} function,
and then pass that to \code{redcap_project_info_read()}.  See the examples below

For more timezone details see the
\href{https://readr.tidyverse.org/articles/locales.html#timezones}{Timezones}
section of readr's
\href{https://readr.tidyverse.org/articles/locales.html}{locales} vignette.

\strong{Columns not yet added}
This function casts columns into data types that we think are most natural to
use.  For example, \code{in_production} is cast from a 0/1 to a
boolean TRUE/FALSE.  All columns not (yet) recognized by REDCapR will be
still be returned to the client, but cast as a character.
If the REDCap API adds a new column in the future,
please alert us in a
\href{https://github.com/OuhscBbmc/REDCapR/issues}{REDCapR issue} and we'll
expand the casting specifications.

\strong{External Modules}

A project's \code{external_modules} cell contains all its approved EMs,
separated by a column.  Consider using a function like
\code{\link[tidyr:separate_rows]{tidyr::separate_rows()}} to create a long data structure.
}
\examples{
\dontrun{
# Specify your project uri and token(s).
uri                  <- "https://bbmc.ouhsc.edu/redcap/api/"
token_simple         <- "9A81268476645C4E5F03428B8AC3AA7B"
token_longitudinal   <- "0434F0E9CF53ED0587847AB6E51DE762"

# ---- Simple examples
REDCapR::redcap_project_info_read(uri, token_simple      )$data
REDCapR::redcap_project_info_read(uri, token_longitudinal)$data

# ---- Specify timezone
# Specify the server's timezone, for example, US Central
server_locale <- readr::locale(tz = "America/Chicago")
d3 <-
  REDCapR::redcap_project_info_read(
    uri,
    token_simple,
    locale     = server_locale
  )$data
d3$creation_time

# Alternatively, set timezone to the client's location.
client_locale <- readr::locale(tz = Sys.timezone())

# ---- Inspect multiple projects in the same tibble
# Stack all the projects on top of each other in a (nested) tibble,
#   starting from a csv of REDCapR test projects.
# The native pipes in this snippet require R 4.1+.
d_all <-
  system.file("misc/example.credentials", package = "REDCapR") |>
  readr::read_csv(
    comment     = "#",
    col_select  = c(redcap_uri, token),
    col_types   = readr::cols(.default = readr::col_character())
  ) |>
  dplyr::filter(32L == nchar(token)) |>
  purrr::pmap_dfr(REDCapR::redcap_project_info_read, locale = server_locale)

# Inspect values stored on the server.
d_all$data
# or: View(d_all$data)

# Inspect everything returned, including values like the http status code.
d_all
}
}
\references{
The official documentation can be found on the 'API Help Page'
and 'API Examples' pages on the REDCap wiki (\emph{i.e.},
https://community.projectredcap.org/articles/456/api-documentation.html
and
https://community.projectredcap.org/articles/462/api-examples.html).
If you do not have an account for the wiki, please ask your campus REDCap
administrator to send you the static material.
}
\author{
Will Beasley, Stephan Kadauke
}
