% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/delaunay.R
\name{delaunay}
\alias{delaunay}
\title{2D Delaunay triangulation}
\usage{
delaunay(points, edges = NULL)
}
\arguments{
\item{points}{a numeric matrix with two columns}

\item{edges}{the edges for the constrained Delaunay triangulation, 
an integer matrix with two columns; \code{NULL} for no constraint}
}
\value{
A list. It has two fields for an unconstrained Delaunay trianguation: 
 \strong{triangles}, an integer matrix with three columns, it provides 
 the indices of the vertices of the Delaunay triangles, and a field 
 \strong{allEdges}, which is an integer matrix with two columns, providing 
 the indices of all the edges of the triangulation. For a constrained 
 Delaunay tessellation, there is an additional field \strong{borderEdges}, 
 an integer matrix with two columns, providing the indices of the edges 
 given as constraints.
}
\description{
Performs a (constrained) Delaunay triangulation of a set of 
  2d points.
}
\examples{
# random points in a square
set.seed(314)
library(RCDT)
library(uniformly)
square <- rbind(
  c(-1, 1), c(1, 1), c(1, -1), c(-1, -1)
)
ptsin <- runif_in_cube(10L, d = 2L)
pts <- rbind(square, ptsin)
del <- delaunay(pts)
opar <- par(mar = c(0, 0, 0, 0))
plotDelaunay(
  del, xlab = NA, ylab = NA, asp = 1, color = "random", luminosity = "dark"
)
par(opar)

# a constrained Delaunay triangulation ####
# points
nangles <- 12L
angles <- seq(0, 2*pi, length.out = nangles+1L)[-1L]
points <- cbind(cos(angles), sin(angles))
points <- rbind(points, points/1.5)

# constraint edges
indices <- 1L:nangles
edges <- cbind(
  indices,
  c(indices[-1L], indices[1L])
) 
edges <- rbind(edges, edges + nangles)

# Delaunay
del <- delaunay(points, edges) 

# plot
opar <- par(mar = c(1, 1, 1, 1))
plotDelaunay(del, color = "yellow", lwd_borders = 2, asp = 1, 
             axes = FALSE, xlab = NA, ylab = NA)
par(opar)

# another constrained Delaunay triangulation: a face ####
V <- read.table(
  system.file("extdata", "face_vertices.txt", package = "RCDT")
)
E <- read.table(
  system.file("extdata", "face_edges.txt", package = "RCDT")
)
del <- delaunay(
  points = as.matrix(V)[, c(2L, 3L)], edges = as.matrix(E)[, c(2L, 3L)]
)
opar <- par(mar = c(1, 1, 1, 1))
plotDelaunay(
  del, col_edges = NULL, color = "salmon", col_borders = "black", asp = 1,
  axes = FALSE, xlab = NA, ylab = NA
)
par(opar)
}
