% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/preddist.R
\name{preddist}
\alias{preddist}
\alias{preddist.betaMix}
\alias{preddist.normMix}
\alias{preddist.gammaMix}
\title{Predictive Distributions for Mixture Distributions}
\usage{
preddist(mix, ...)

\method{preddist}{betaMix}(mix, n = 1, ...)

\method{preddist}{normMix}(mix, n = 1, sigma, ...)

\method{preddist}{gammaMix}(mix, n = 1, ...)
}
\arguments{
\item{mix}{mixture distribution}

\item{...}{includes arguments which depend on the specific prior-likelihood pair, see description below.}

\item{n}{predictive sample size, set by default to 1}

\item{sigma}{The fixed reference scale of a normal mixture. If left
unspecified, the default reference scale of the mixture is assumed.}
}
\value{
The function returns for a normal, beta or gamma mixture
the matching predictive distribution for \eqn{y_n}.
}
\description{
Predictive distribution for mixture of conjugate distributions
(beta, normal, gamma).
}
\details{
Given a mixture density (either a posterior or a prior)

\deqn{p(\theta,\mathbf{w},\mathbf{a},\mathbf{b})}{p(\theta,w,a,b)}

and a data likelihood of

\deqn{y|\theta \sim f(y|\theta),}{y|\theta ~ f(y|\theta),}

the predictive distribution of a one-dimensional summary \eqn{y_n}
of $n$ future observations is distributed as

\deqn{y_n \sim \int p(\theta,\mathbf{w},\mathbf{a},\mathbf{b}) \, f(y_n|\theta) \, d\theta .}{y_n ~ \int p(u,w,a,b) \, f(y_n|u) du .}

This distribution is the marginal distribution of the data under
the mixture density. For binary and Poisson data \eqn{y_n =
\sum_{i=1}^n y_i} is the sum over future events. For normal data,
it is the mean\eqn{\bar{y}_n = 1/n \sum_{i=1}^n y_i}.
}
\section{Methods (by class)}{
\itemize{
\item \code{betaMix}: Obtain the matching predictive distribution
for a beta distribution, the BetaBinomial.

\item \code{normMix}: Obtain the matching predictive distribution
for a Normal with constant standard deviation. Note that the
reference scale of the returned Normal mixture is meaningless as the
individual components are updated appropriatley.

\item \code{gammaMix}: Obtain the matching predictive distribution
for a Gamma. Only Poisson likelihoods are supported.
}}

\section{Supported Conjugate Prior-Likelihood Pairs}{


\tabular{lccc}{
\strong{Prior/Posterior} \tab \strong{Likelihood} \tab \strong{Predictive} 
 \tab \strong{Summaries} \cr
Beta \tab Binomial \tab Beta-Binomial \tab \code{n}, \code{r} \cr
Normal \tab Normal (\emph{fixed \eqn{\sigma}}) \tab Normal \tab \code{n}, \code{m}, \code{se}  \cr
Gamma \tab Poisson \tab Gamma-Poisson \tab  \code{n}, \code{m} \cr
Gamma \tab Exponential \tab Gamma-Exp (\emph{not supported}) \tab \code{n}, \code{m}
}
}

\examples{

# Example 1: predictive distribution from uniform prior.
bm <- mixbeta(c(1,1,1))
bmPred <- preddist(bm, n=10)
# predictive proabilities and cumulative predictive probabilities
x <- 0:10
d <- dmix(bmPred, x)
names(d) <- x
barplot(d)
cd <- pmix(bmPred, x)
names(cd) <- x
barplot(cd)
# median
mdn <- qmix(bmPred,0.5)
mdn

# Example 2: 2-comp Beta mixture

bm <- mixbeta( inf=c(0.8,15,50),rob=c(0.2,1,1))
plot(bm)
bmPred <- preddist(bm,n=10)
plot(bmPred)
mdn <- qmix(bmPred,0.5)
mdn
d <- dmix(bmPred,x=0:10)
\donttest{
n.sim <- 100000
r <-  rmix(bmPred,n.sim)
d
table(r)/n.sim
}

# Example 3: 3-comp Normal mixture

m3 <- mixnorm( c(0.50,-0.2,0.1),c(0.25,0,0.2), c(0.25,0,0.5), sigma=10)
print(m3)
summary(m3)
plot(m3)
predm3 <- preddist(m3,n=2)
plot(predm3)
print(predm3)
summary(predm3)

}
