% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MIM.search2.R
\name{MIM.search2}
\alias{MIM.search2}
\title{QTL search by MIM with Seletive Genotyping}
\usage{
MIM.search2(
  QTL,
  marker,
  geno,
  y,
  yu = NULL,
  sele.g = "n",
  tL = NULL,
  tR = NULL,
  method = "EM",
  type = "RI",
  D.matrix = NULL,
  ng = 2,
  cM = TRUE,
  speed = 1,
  QTLdist = 15,
  link = TRUE,
  crit = 10^-3,
  console = TRUE
)
}
\arguments{
\item{QTL}{matrix. A q*2 matrix contains the QTL information, where the
row dimension 'q' represents the number of QTLs in the chromosomes. The
first column labels the chromosomes where the QTLs are located, and the
second column labels the positions of QTLs (in morgan (M) or centimorgan
(cM)).}

\item{marker}{matrix. A k*2 matrix contains the marker information,
where the row dimension 'k' represents the number of markers in the
chromosomes. The first column labels the chromosomes where the markers
are located, and the second column labels the positions of markers (in
morgan (M) or centimorgan (cM)). It's important to note that chromosomes
and positions must be sorted in order.}

\item{geno}{matrix. A n*k matrix contains the genotypes of k markers
for n individuals. The marker genotypes of P1 homozygote (MM),
heterozygote (Mm), and P2 homozygote (mm) are coded as 2, 1, and 0,
respectively, with NA indicating missing values.}

\item{y}{vector. A vector that contains the phenotype values of
individuals with genotypes.}

\item{yu}{vector. A vector that contains the phenotype values of
individuals without genotypes.}

\item{sele.g}{character. Determines the type of data being analyzed:
If sele.g="n", it considers the data as complete genotyping data. If
sele.g="f", it treats the data as selective genotyping data and utilizes
the proposed corrected frequency model (Lee 2014) for analysis; If
sele.g="t", it considers the data as selective genotyping data and uses
the truncated model (Lee 2014) for analysis; If sele.g="p", it treats
the data as selective genotyping data and uses the population
frequency-based model (Lee 2014) for analysis. Note that the 'yu'
argument must be provided when sele.g="f" or "p".}

\item{tL}{numeric. The lower truncation point of phenotype value when
sele.g="t". When sele.g="t" and tL=NULL, the 'yu' argument must be
provided. In this case, the function will consider the minimum of 'yu'
as the lower truncation point.}

\item{tR}{numeric. The upper truncation point of phenotype value when
sele.g="t". When sele.g="t" and tR=NULL, the 'yu' argument must be
provided. In this case, the function will consider the maximum of 'yu'
as the upper truncation point.}

\item{method}{character. When method="EM", it indicates that the interval
mapping method by Lander and Botstein (1989) is used in the analysis.
Conversely, when method="REG", it indicates that the approximate regression
interval mapping method by Haley and Knott (1992) is used in the analysis.}

\item{type}{character. The population type of the dataset. Includes
backcross (type="BC"), advanced intercross population (type="AI"), and
recombinant inbred population (type="RI"). The default value is "RI".}

\item{D.matrix}{matrix. The design matrix of QTL effects is a g*p matrix,
where g is the number of possible QTL genotypes, and p is the number of
effects considered in the MIM model. It's important to note that the QTL
number of the design matrix must be the original QTL number plus one. The
design matrix can be easily generated by the function D.make(). If set to
NULL, it will automatically generate a design matrix with all additive and
dominant effects and without any epistasis effect.}

\item{ng}{integer. The generation number of the population type. For
instance, in a BC1 population where type="BC", ng=1; in an AI F3
population where type="AI", ng=3.}

\item{cM}{logical. Specify the unit of marker position. If cM=TRUE, it
denotes centimorgan; if cM=FALSE, it denotes morgan.}

\item{speed}{numeric. The walking speed of the QTL search (in cM).}

\item{QTLdist}{numeric. The minimum distance (in cM) among different
linked significant QTL. Positions near the known QTLs within this distance
will not be considered as candidate positions in the search process.}

\item{link}{logical. When set to False, positions on the same chromosomes
as the known QTLs will not be searched.}

\item{crit}{numeric. The convergence criterion of EM algorithm.
The E and M steps will iterate until a convergence criterion is met.
It must be a value between 0 and 1.}

\item{console}{logical. Determines whether the process of the algorithm
will be displayed in the R console or not.}
}
\value{
\item{effect}{The estimated effects, log-likelihood value, and LRT
statistics of all searched positions.}
\item{QTL.best}{The positions of the best QTL combination.}
\item{effect.best}{The estimated effects and LRT statistics of the best
QTL combination.}
\item{model}{The model of selective genotyping data in this analyze.}
}
\description{
Expectation-maximization algorithm for QTL multiple interval mapping to
find one more QTL in the presence of some known QTLs. It can handle
genotype data which is selective genotyping.
}
\examples{
# load the example data
load(system.file("extdata", "exampledata.RDATA", package = "QTLEMM"))

# make the seletive genotyping data
ys <- y[y > quantile(y)[4] | y < quantile(y)[2]]
yu <- y[y >= quantile(y)[2] & y <= quantile(y)[4]]
geno.s <- geno[y > quantile(y)[4] | y < quantile(y)[2],]

# run and result
QTL <- c(1, 23)
result <- MIM.search2(QTL, marker, geno.s, ys, yu, sele.g = "f",
 type = "RI", ng = 2, speed = 15, QTLdist = 50)
result$QTL.best
result$effect.best
}
\references{
KAO, C.-H. and Z.-B. ZENG 1997 General formulas for obtaining the maximum
likelihood estimates and the asymptotic variance-covariance matrix in QTL
mapping when using the EM algorithm. Biometrics 53, 653-665.

KAO, C.-H., Z.-B. ZENG and R. D. TEASDALE 1999 Multiple interval mapping
for Quantitative Trait Loci. Genetics 152: 1203-1216.

H.-I LEE, H.-A. HO and C.-H. KAO 2014 A new simple method for improving
QTL mapping under selective genotyping. Genetics 198: 1685-1698.
}
\seealso{
\code{\link[QTLEMM]{EM.MIM2}}
\code{\link[QTLEMM]{MIM.search}}
}
