#' Fix grammar and rewrite text using the Google PaLM model based on a query.
#'
#' This function sends a query with grammatical issues to the Google PaLM model and generates corrected text as a response.
#' It allows customization of the generated text using various parameters.
#'
#' @param palmParameter A character vector containing the API key, model version, and model type, as provided by Google.
#' The API key should be a 39-character string. Model version and type are specified by Google. See function `setupPALM()` for details.
#' @param inquery A character string representing the text with grammatical issues that you want to rewrite. The length of
#' the text should be between 1 and 8196 characters, inclusive.
#' @param temperature A numeric value between 0.0 and 1.0, inclusive (default: 0.7). Controls the randomness of the generated text.
#' A higher value (e.g., 0.9) results in more creative responses, while a lower value (e.g., 0.3) produces more straightforward text.
#' @param maxOutputTokens An integer value between 1 and 1024, inclusive (default: 1024). Specifies the maximum number of tokens to
#' include in the generated text.
#' @param topP A numeric value between 0.0 and 1.0, inclusive (default: 0.95). Defines the maximum cumulative probability of tokens
#' considered when sampling. It controls the diversity of the text generated.
#' @param topK An integer value between 1 and 1,000,000, inclusive (default: 40). Sets the maximum number of tokens to consider
#' when sampling.
#' @param htUnspecified Safety setting threshold for unspecified harm. The default threshold is "meda." Refer to \href{https://developers.generativeai.google/api/rest/generativelanguage/HarmCategory}{HarmCategory - Google PaLMr}.
#' Valid options include:
#'
#' - "unsp" (HARM_BLOCK_THRESHOLD_UNSPECIFIED)
#'
#' - "lowa" (BLOCK_LOW_AND_ABOVE)
#'
#' - "meda" (BLOCK_MEDIUM_AND_ABOVE)
#'
#' - "high" (BLOCK_ONLY_HIGH)
#'
#' - "none" (BLOCK_NONE)
#'
#' @param htDerogatory Safety setting threshold for derogatory harm. The default threshold is "meda." Refer to \href{https://developers.generativeai.google/api/rest/generativelanguage/HarmCategory}{HarmCategory - Google PaLMr}.
#' Valid options include:
#'
#' - "unsp" (HARM_BLOCK_THRESHOLD_UNSPECIFIED)
#'
#' - "lowa" (BLOCK_LOW_AND_ABOVE)
#'
#' - "meda" (BLOCK_MEDIUM_AND_ABOVE)
#'
#' - "high" (BLOCK_ONLY_HIGH)
#'
#' - "none" (BLOCK_NONE)
#'
#' @param htToxicity Safety setting threshold for toxicity harm. The default threshold is "meda." Refer to \href{https://developers.generativeai.google/api/rest/generativelanguage/HarmCategory}{HarmCategory - Google PaLMr}.
#' Valid options include:
#'
#' - "unsp" (HARM_BLOCK_THRESHOLD_UNSPECIFIED)
#'
#' - "lowa" (BLOCK_LOW_AND_ABOVE)
#'
#' - "meda" (BLOCK_MEDIUM_AND_ABOVE)
#'
#' - "high" (BLOCK_ONLY_HIGH)
#'
#' - "none" (BLOCK_NONE)
#'
#' @param htViolence Safety setting threshold for violence harm. The default threshold is "meda." Refer to \href{https://developers.generativeai.google/api/rest/generativelanguage/HarmCategory}{HarmCategory - Google PaLMr}.
#' Valid options include:
#'
#' - "unsp" (HARM_BLOCK_THRESHOLD_UNSPECIFIED)
#'
#' - "lowa" (BLOCK_LOW_AND_ABOVE)
#'
#' - "meda" (BLOCK_MEDIUM_AND_ABOVE)
#'
#' - "high" (BLOCK_ONLY_HIGH)
#'
#' - "none" (BLOCK_NONE)
#'
#' @param htSexual Safety setting threshold for sexual harm. The default threshold is "meda." Refer to \href{https://developers.generativeai.google/api/rest/generativelanguage/HarmCategory}{HarmCategory - Google PaLMr}.
#' Valid options include:
#'
#' - "unsp" (HARM_BLOCK_THRESHOLD_UNSPECIFIED)
#'
#' - "lowa" (BLOCK_LOW_AND_ABOVE)
#'
#' - "meda" (BLOCK_MEDIUM_AND_ABOVE)
#'
#' - "high" (BLOCK_ONLY_HIGH)
#'
#' - "none" (BLOCK_NONE)
#'
#' @param htMedical Safety setting threshold for medical harm. The default threshold is "meda." Refer to \href{https://developers.generativeai.google/api/rest/generativelanguage/HarmCategory}{HarmCategory - Google PaLMr}.
#' Valid options include:
#'
#' - "unsp" (HARM_BLOCK_THRESHOLD_UNSPECIFIED)
#'
#' - "lowa" (BLOCK_LOW_AND_ABOVE)
#'
#' - "meda" (BLOCK_MEDIUM_AND_ABOVE)
#'
#' - "high" (BLOCK_ONLY_HIGH)
#'
#' - "none" (BLOCK_NONE)
#'
#' @param htDangerous Safety setting threshold for dangerous harm. The default threshold is "meda." Refer to \href{https://developers.generativeai.google/api/rest/generativelanguage/HarmCategory}{HarmCategory - Google PaLMr}.
#' Valid options include:
#'
#' - "unsp" (HARM_BLOCK_THRESHOLD_UNSPECIFIED)
#'
#' - "lowa" (BLOCK_LOW_AND_ABOVE)
#'
#' - "meda" (BLOCK_MEDIUM_AND_ABOVE)
#'
#' - "high" (BLOCK_ONLY_HIGH)
#'
#' - "none" (BLOCK_NONE)
#'
#' @return A character string containing the rewritten text with corrected grammar, generated by the Google PaLM API
#' based on the provided query and parameters.
#'
#' @details
#' This function interacts with the Google PaLM model by sending a query with grammatical issues using the specified parameters.
#' It allows you to customize the generated text by adjusting the `temperature`, `maxOutputTokens`, `topP`, `topK`, and safety settings.
#'
#' If the function is successful, it returns a character string containing the rewritten text with corrected grammar.
#' If an error occurs during the API request, it will stop execution and provide an error message.
#'
#' The `palmParameter` argument should be a character vector with the API key, model version, and model type provided by
#' Google. You can obtain this information by following the instructions provided by Google for using the PaLM API.
#'
#' The safety settings control the content's safety level based on different harm categories. Harm thresholds are
#' specified as per Google's guidelines and can be customized to control the content generated.
#'
#' For more information on safety settings, harm categories, and harm thresholds, refer to the official Google PaLM API
#' documentation: \href{https://developers.generativeai.google/api/rest/generativelanguage/SafetySetting}{Safety Setting - Google PaLMr}
#'
#' @examples
#' \dontrun{
#' # Set up the PaLM parameters
#' # Replace your_api_key_here with the API key you get from Google
#' palmParameter <- c("your_api_key_here", "v1beta3", "text-bison-001")
#'
#' # Fix grammar and rewrite text based on a query
#'
#' # Example 1
#' inquery <- "Yesterday, I will buy a book for my younger sister as his birthday
#'             gift. They were very happen when seeing this gift earlier today."
#' print(fixGrammarPALM(palmParameter, inquery, temperature = 0.7))
#'
#' # Example output:
#' # Yesterday, I bought a book for my younger sister as her birthday gift.
#' # She was very happy when she saw it earlier today.
#'
#' # Example 2
#' inquery <- "Dora begun to understan how the Teacher come up with the rules.
#'            By the end of the year, Dora though Mrs. Davis was the best Teacher
#'            she evere had!"
#' print(fixGrammarPALM(palmParameter, inquery, temperature = 1))
#'
#' # Example output:
#' # Dora began to understand how the teacher came up with the rules. By the end
#' # of the year, Dora thought Mrs. Davis was the best teacher she had ever had!
#' }
#'
#' @seealso
#' \href{https://developers.generativeai.google/api/rest/generativelanguage/SafetySetting}{Safety Setting - Google PaLMr}
#'
#' \href{https://developers.generativeai.google/api/rest/generativelanguage/HarmCategory}{HarmCategory - Google PaLMr}
#'
#' @export
#'
#' @importFrom PaLMr checkModelSelection
#' @importFrom PaLMr generateSafetySettings
#' @importFrom PaLMr generateOutput
fixGrammarPALM = function(palmParameter, inquery, temperature = 0.7,
                          maxOutputTokens = 1024, topP = 0.95, topK = 40,
                          htUnspecified = "meda",
                          htDerogatory = "meda",
                          htToxicity = "meda",
                          htViolence = "meda",
                          htSexual = "meda",
                          htMedical = "meda",
                          htDangerous = "meda") {
  checkModelSelection(palmParameter[2], palmParameter[3])

  # Define the API URL
  apiURL = paste0("https://generativelanguage.googleapis.com/",
                  palmParameter[2],
                  "/models/", palmParameter[3],
                  ":generateText?key=", palmParameter[1])

  # Create the request body as a JSON object
  inquery = gsub('"', '\\"', inquery)
  requestBody = list(
    prompt = list(
      text = paste0("Rewrite the following text and fix any grammar issues:\n ",
                    "# Text starts #\n",
                    inquery, "\n",
                    "# Text ends #\n")
    ),
    safetySettings = generateSafetySettings(htUnspecified, htDerogatory, htToxicity,
                                            htViolence, htSexual, htMedical, htDangerous),
    temperature = temperature,
    maxOutputTokens = as.integer(maxOutputTokens),
    topP = topP,
    topK = as.integer(topK)
  )

  # Convert the request body to JSON
  requestBodyJSON = jsonlite::toJSON(requestBody, auto_unbox = TRUE)

  # Get response from the API
  response = httr::POST(url = apiURL,
                        body = requestBodyJSON,
                        httr::add_headers("Content-Type" = "application/json"))
  responseJSON = httr::content(response, "parsed")

  # Generate output
  generateOutput(responseJSON)
}
