% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/cornish_fisher.r
\name{AS269}
\alias{AS269}
\title{Higher order Cornish Fisher approximation.}
\usage{
AS269(z,cumul,order.max=NULL,all.ords=FALSE)
}
\arguments{
\item{z}{the quantiles of the normal distribution. an atomic vector.}

\item{cumul}{the standardized cumulants of order 3, 4, ..., k. an atomic
vector.}

\item{order.max}{the maximum order approximation, must be greater than
\code{length(cumul)+2}.
We assume the cumulants have been adjusted to reflect that the random
variable has unit variance ('standardized cumulants')}

\item{all.ords}{a logical value. If \code{TRUE}, then results are returned
as a matrix, with a column for each order of the approximation. Otherwise
the results are a matrix with a single column of the highest order
approximation.}
}
\value{
A matrix, which is, depending on \code{all.ords}, either with one column per
order of the approximation, or a single column giving the maximum order
approximation. There is one row per value in \code{z}.

Invalid arguments will result in return value \code{NaN} with a warning.
}
\description{
Lee and Lin's Algorithm AS269 for higher order Cornish Fisher quantile approximation.
}
\details{
The Cornish Fisher approximation is the Legendre
inversion of the Edgeworth expansion of a distribution, but ordered
in a way that is convenient when used on the mean of a number of
independent draws of a random variable.

Suppose \eqn{x_1, x_2, \ldots, x_n}{x_1, x_2, ..., x_n} are \eqn{n} independent
draws from some probability distribution.
Letting
\deqn{X = \frac{1}{\sqrt{n}} \sum_{1 \le i \le n} x_i,}{X = (x_1 + x_2 + ... x_n) / sqrt(n),}
the Central Limit Theorem assures us that, assuming finite variance,
\deqn{X \rightarrow \mathcal{N}(\sqrt{n}\mu, \sigma),}{X ~~ N(sqrt(n) mu, sigma),}
with convergence in \eqn{n}.

The Cornish Fisher approximation gives a more detailed picture of the
quantiles of \eqn{X}{X}, one that is arranged in decreasing powers of
\eqn{\sqrt{n}}{sqrt(n)}. The quantile function is the function \eqn{q(p)}{q(p)}
such that \eqn{P\left(X \le q(p)\right) = q(p)}{P(x <= q(p)) = p}. The
Cornish Fisher expansion is
\deqn{q(p) = \sqrt{n}\mu + \sigma \left(z + \sum_{3 \le j} c_j f_j(z)\right),}{q(p) = sqrt{n}mu + sigma (z + sum_{3 <= j} c_j f_j(z)),}
where \eqn{z = \Phi^{-1}(p)}{z = qnorm(p)}, and \eqn{c_j}{c_j} involves
standardized cumulants of the distribution of \eqn{x_i}{x_i} of order
up to \eqn{j}. Moreover, the \eqn{c_j}{c_j} include decreasing
powers of \eqn{\sqrt{n}}{sqrt(n)}, giving some justification for truncation.
When \eqn{n=1}{n=1}, however, the ordering is somewhat arbitrary.
}
\note{
A warning will be thrown if any of the z are greater than
3.719017274 in absolute value; the traditional AS269 errors out in this
case.
}
\examples{
foo <- AS269(seq(-2,2,0.01),c(0,2,0,4))
# test with the normal distribution:
s.cumul <- c(0,0,0,0,0,0,0,0,0)
pv <- seq(0.001,0.999,0.001)
zv <- qnorm(pv)
apq <- AS269(zv,s.cumul,all.ords=FALSE)
err <- zv - apq

# test with the exponential distribution
rate <- 0.7
n <- 18
# these are 'raw' cumulants'
cumul <- (rate ^ -(1:n)) * factorial(0:(n-1))
# standardize and chop
s.cumul <- cumul[3:length(cumul)] / (cumul[2]^((3:length(cumul))/2))
pv <- seq(0.001,0.999,0.001)
zv <- qnorm(pv)
apq <- cumul[1] + sqrt(cumul[2]) * AS269(zv,s.cumul,all.ords=TRUE)
truq <- qexp(pv, rate=rate)
err <- truq - apq
colSums(abs(err))

# an example from Wikipedia page on CF,
# \\url{https://en.wikipedia.org/wiki/Cornish\%E2\%80\%93Fisher_expansion}
s.cumul <- c(5,2)
apq <- 10 + sqrt(25) * AS269(qnorm(0.95),s.cumul,all.ords=TRUE)
}
\author{
Steven E. Pav \email{shabbychef@gmail.com}
}
\references{
Lee, Y-S., and Lin, T-K. "Algorithm AS269: High Order Cornish Fisher
Expansion." Appl. Stat. 41, no. 1 (1992): 233-240.
\url{http://www.jstor.org/stable/2347649}

Lee, Y-S., and Lin, T-K. "Correction to Algorithm AS269: High Order
Cornish Fisher Expansion." Appl. Stat. 42, no. 1 (1993): 268-269.
\url{http://www.jstor.org/stable/2347433}

AS 269. \url{http://lib.stat.cmu.edu/apstat/269}

Jaschke, Stefan R. "The Cornish-Fisher-expansion in the context of
Delta-Gamma-normal approximations." No. 2001, 54. Discussion Papers,
Interdisciplinary Research Project 373: Quantification and Simulation of
Economic Processes, 2001.
\url{http://www.jaschke-net.de/papers/CoFi.pdf}
}
\seealso{
\code{\link{qapx_cf}}
}
\keyword{distribution}

