% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ARMA_optim.R
\name{NNS.ARMA.optim}
\alias{NNS.ARMA.optim}
\title{NNS ARMA Optimizer}
\usage{
NNS.ARMA.optim(
  variable,
  h = NULL,
  training.set = NULL,
  seasonal.factor,
  negative.values = FALSE,
  obj.fn = expression(mean((predicted - actual)^2)/(NNS::Co.LPM(1, predicted, actual,
    target_x = mean(predicted), target_y = mean(actual)) + NNS::Co.UPM(1, predicted,
    actual, target_x = mean(predicted), target_y = mean(actual)))),
  objective = "min",
  linear.approximation = TRUE,
  pred.int = 0.95,
  print.trace = TRUE,
  plot = FALSE
)
}
\arguments{
\item{variable}{a numeric vector.}

\item{h}{integer; \code{NULL} (default) Number of periods to forecast out of sample.  If \code{NULL}, \code{h = length(variable) - training.set}.}

\item{training.set}{integer; \code{NULL} (default) Sets the number of variable observations as the training set.  See \code{Note} below for recommended uses.}

\item{seasonal.factor}{integers; Multiple frequency integers considered for \link{NNS.ARMA} model, i.e. \code{(seasonal.factor = c(12, 24, 36))}}

\item{negative.values}{logical; \code{FALSE} (default) If the variable can be negative, set to
\code{(negative.values = TRUE)}.  It will automatically select \code{(negative.values = TRUE)} if the minimum value of the \code{variable} is negative.}

\item{obj.fn}{expression;
\code{expression(cor(predicted, actual, method = "spearman") / sum((predicted - actual)^2))} (default) Rank correlation / sum of squared errors is the default objective function.  Any \code{expression(...)} using the specific terms \code{predicted} and \code{actual} can be used.}

\item{objective}{options: ("min", "max") \code{"max"} (default) Select whether to minimize or maximize the objective function \code{obj.fn}.}

\item{linear.approximation}{logical; \code{TRUE} (default) Uses the best linear output from \code{NNS.reg} to generate a nonlinear and mixture regression for comparison.  \code{FALSE} is a more exhaustive search over the objective space.}

\item{pred.int}{numeric [0, 1]; 0.95 (default) Returns the associated prediction intervals for the final estimate.  Constructed using the maximum entropy bootstrap \link{NNS.meboot} on the final estimates.}

\item{print.trace}{logical; \code{TRUE} (default) Prints current iteration information.  Suggested as backup in case of error, best parameters to that point still known and copyable!}

\item{plot}{logical; \code{FALSE} (default)}
}
\value{
Returns a list containing:
\itemize{
\item{\code{$period}} a vector of optimal seasonal periods
\item{\code{$weights}} the optimal weights of each seasonal period between an equal weight or NULL weighting
\item{\code{$obj.fn}} the objective function value
\item{\code{$method}} the method identifying which \link{NNS.ARMA} method was used.
\item{\code{$shrink}} whether to use the \code{shrink} parameter in \link{NNS.ARMA}.
\item{\code{$nns.regress}} whether to smooth the variable via \link{NNS.reg} before forecasting.
\item{\code{$bias.shift}} a numerical result of the overall bias of the optimum objective function result.  To be added to the final result when using the \link{NNS.ARMA} with the derived parameters.
\item{\code{$errors}} a vector of model errors from internal calibration.
\item{\code{$results}} a vector of length \code{h}.
\item{\code{$lower.pred.int}} a vector of lower prediction intervals per forecast point.
\item{\code{$upper.pred.int}} a vector of upper prediction intervals per forecast point.
}
}
\description{
Wrapper function for optimizing any combination of a given \code{seasonal.factor} vector in \link{NNS.ARMA}.  Minimum sum of squared errors (forecast-actual) is used to determine optimum across all \link{NNS.ARMA} methods.
}
\note{
\itemize{
\item{} Typically, \code{(training.set = 0.8 * length(variable)} is used for optimization.  Smaller samples could use \code{(training.set = 0.9 * length(variable))} (or larger) in order to preserve information.

\item{} The number of combinations will grow prohibitively large, they should be kept as small as possible.  \code{seasonal.factor} containing an element too large will result in an error.  Please reduce the maximum \code{seasonal.factor}.

}
}
\examples{

## Nonlinear NNS.ARMA period optimization using 2 yearly lags on AirPassengers monthly data
\dontrun{
nns.optims <- NNS.ARMA.optim(AirPassengers[1:132], training.set = 120,
seasonal.factor = seq(12, 24, 6))

## To predict out of sample using best parameters:
NNS.ARMA.optim(AirPassengers[1:132], h = 12, seasonal.factor = seq(12, 24, 6))

## Incorporate any objective function from external packages (such as \code{Metrics::mape})
NNS.ARMA.optim(AirPassengers[1:132], h = 12, seasonal.factor = seq(12, 24, 6),
obj.fn = expression(Metrics::mape(actual, predicted)), objective = "min")
}

}
\references{
Viole, F. and Nawrocki, D. (2013) "Nonlinear Nonparametric Statistics: Using Partial Moments"
\url{https://www.amazon.com/dp/1490523995/ref=cm_sw_su_dp}
}
\author{
Fred Viole, OVVO Financial Systems
}
