% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Simulations.R
\name{Spot.Price.Simulate}
\alias{Spot.Price.Simulate}
\title{Spot.Price.Simulate}
\usage{
Spot.Price.Simulate(
  X.0,
  parameters,
  t = 1,
  dt = 1,
  n = 2,
  antithetic = TRUE,
  verbose = FALSE
)
}
\arguments{
\item{X.0}{Initial values of the state vector.}

\item{parameters}{A named vector of parameter values of a specified N-factor model. Function \code{NFCP.Parameters} is recommended.}

\item{t}{the number of years to simulate}

\item{dt}{discrete time step of simulation}

\item{n}{total number of simulations}

\item{antithetic}{\code{logical}. Should antithetic price paths be simulated?}

\item{verbose}{\code{logical}. Should simulated state variables be output? see \bold{returns}}
}
\value{
\code{Spot.Price.Simulate} returns a list when \code{verbose = T} and a matrix of simulated price paths when \code{verbose = F}. The returned objects in the list are:

\code{Prices} A matrix of simulated price paths. Each column represents one simulated price path and each row represents one simulated observation.

\code{Factors} A matrix of simulated state variables for each factor is returned when \code{verbose = T}. The number of factors returned corresponds to the number of factors in the specified N-factor model.
}
\description{
Simulate risk-neutral price paths of an an N-factor commodity pricing model through Monte Carlo Simulation.
}
\details{
\loadmathjax
The \code{Spot.Price.Simulate} function is able to quickly simulate a large number of risk-neutral price paths of a commodity following the N-factor model.
Simulating risk-neutral price paths of a commodity under an N-factor model through Monte Carlo simulations allows for the
valuation of commodity related investments and derivatives, such as American Options and Real Options through dynamic programming methods.
The \code{Spot.Price.Simulate} function quickly and efficiently simulates an N-factor model over a specified number of years, simulating antithetic price paths as a simple variance reduction technique.
The \code{Spot.Price.Simulate} function uses the \code{mvrnorm} function from the \code{MASS} package to draw from a multivariate normal distribution for the simulation shocks.

The N-factor model stochastic differential equation is given by:

Brownian Motion processes (ie. factor one when \code{GBM = T}) are simulated using the following solution:

\mjdeqn{x_{1,t+1} = x_{1,t} + \mu^*\Delta t + \sigma_1 \Delta t Z_{t+1}}{x[1,t+1] = x[1,t] + mu^* * Delta t + sigma[1] * Delta t * Z[t+1]}

Where \mjeqn{\Delta t}{Delta t} is the discrete time step, \mjeqn{\mu^*}{mu^*} is the risk-neutral growth rate and \mjeqn{\sigma_1}{sigma[1]} is the instantaneous volatility. \mjeqn{Z_t}{Z[t]} represents the
independent standard normal at time \mjeqn{t}{t}.

Ornstein-Uhlenbeck Processes are simulated using the following solution:

\mjdeqn{x_{i,t} = x_{i,0}e^{-\kappa_it}-\frac{\lambda_i}{\kappa_i}(1-e^{-\kappa_it})+\int_0^t\sigma_ie^{\kappa_is}dW_s}{x[i,t] = x[i,0] * e^(-kappa[i] * t) - lambda[i]/kappa[i] * (1 - e^(-kappa[i] * t)) + int_0^t (sigma[i] *
e^(kappa[i] * s) dW[s])}

Where a numerical solution is obtained by numerically discretising and approximating the integral term using the Euler-Maruyama integration scheme:
\mjdeqn{\int_0^t\sigma_ie^{\kappa_is}dW_s = \sum_{j=0}^t \sigma_ie^{\kappa_ij}dW_s}{int_0^t ( sigma[i] e^(kappa[i] * s) dw[s])}
}
\examples{
# Example 1
###Simulate a Geometric Brownian Motion (GBM) process:
## Starting price of 20, with a growth of 5\% p.a. and
## volatility of 20\% p.a.
Simulated.Spot.Prices <- Spot.Price.Simulate(
 X.0 = log(20),
 parameters = c(mu_star = (0.05 - (1/2) * 0.2^2), sigma_1 = 0.2),
 t = 1,
 dt = 1/12,
 n = 1e3)

# Example 2
###Simulate future spot price paths under Risk-Neutrality and under the
###Schwartz - Smith two factor model:

##Step 1 - Run the Kalman Filter for the Two-Factor Oil Model:
Schwartz.Smith.Oil <- NFCP.Kalman.filter(parameter.values = SS.Oil$Two.Factor,
                                      parameters = names(SS.Oil$Two.Factor),
                                      log.futures = log(SS.Oil$Stitched.Futures),
                                      dt = SS.Oil$dt,
                                      TTM = SS.Oil$Stitched.TTM,
                                      verbose = TRUE)

#Step 2 - Simulate spot prices:
##100 antithetic simulations of one year of monthly observations
Simulated.Spot.Prices <- Spot.Price.Simulate(
 X.0 = Schwartz.Smith.Oil$X.t,
 parameters = SS.Oil$Two.Factor,
 t = 1,
 dt = 1/12,
 n = 1e3,
 antithetic = TRUE,
 verbose = TRUE)

}
\references{
Schwartz, E. S., and J. E. Smith, (2000). Short-Term Variations and Long-Term Dynamics in Commodity Prices. \emph{Manage. Sci.}, 46, 893-911.

Cortazar, G., and L. Naranjo, (2006). An N-factor Gaussian model of oil futures prices. \emph{Journal of Futures Markets: Futures, Options, and Other Derivative Products}, 26(3), 243-268.
}
