% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matchitmice.R
\name{matchitmice}
\alias{matchitmice}
\title{Selects Matched Samples from Multiply Imputed Datasets}
\usage{
matchitmice(Formula, Datasets, Distance = "logit",
  Distance.options = list(), Discard = "none", Reestimate = FALSE,
  Replace = FALSE, Ratio = 1, Caliper = 0, Calclosest = FALSE,
  Mahvars = NULL, Merging = FALSE, MergingData = NULL, By = "ID",
  Weighting = FALSE, ...)
}
\arguments{
\item{Formula}{This argument takes the usual syntax of the R formula, "treat ~ x1 + x2", where "treat" is a binary treatment indicator and "x1" and "x2" are the pre-treatment covariates. Both the treatment indicator and pre-treatment covariates must be contained in all of the multiply imputed datasets, which is specified as "Datasets" (see below). All of the usual R syntax for "Formula" works. For example, "x1:x2" represents the first order interaction term between "x1" and "x2", and "I(x1^2)" represents the square term of "x1". See "help(formula)" for details.}

\item{Datasets}{This argument specifies the datasets containing the variables called in the "Formula". This must be an object of class "mids", which is typically produced by a previous call to "mice()" or "mice.mids()" functions (from the "mice" package).}

\item{Distance}{This argument specifies the method used to estimate the distance (propensity score) measures. The default is logistic regression, "logit". A variety of other methods are available.}

\item{Distance.options}{This optional argument specifies the optional arguments that are passed to the model for estimating the distance measures. The input to this argument should be a list.}

\item{Discard}{This argument specifies whether to discard units that fall outside some measures of support of the distance scores before matching, and not allow them to be used at all in the matching procedure. Note that discarding units may change the quantity of interest being estimated. The options are: "none" (default), which discards no units before matching, "both", which discards all units (treatment and control) that are outside the support of the distance measure, "control", which discards only control units outside the support of the distance measure of the treatment units, and "treat", which discards only treatment units outside the support of the distance measure of the control units.}

\item{Reestimate}{This argument specifies whether the model for distance measure should be re-estimated after units are discarded. The input must be a logical value. The default is "FALSE".}

\item{Replace}{This argument specifies whether each control unit can be matched to more than one treatment unit. The input must be a logical value. The default is "FALSE".}

\item{Ratio}{This argument specifies the number of control units to match to each treatment unit. The input must be a numeric value. The default is 1.}

\item{Caliper}{This argument specifies the number of standard deviations of the distance measure within which to draw control units. The input must be a numeric value. The default is 0.}

\item{Calclosest}{When "Caliper" is greater than 0, this argument specifies whether to take the nearest available match if no matches are available within the "Caliper". The input must be a logical value. The default is "FALSE".}

\item{Mahvars}{When "Caliper" is greater than 0, this argument specifies variables on which to perform Mahalanobis-metric matching within each "Caliper". The default is "NULL".}

\item{Merging}{This argument specifies whether a dataset should be merged with each imputed dataset or not. The input must be a logical value. The default is "FALSE".}

\item{MergingData}{This optional argument specifies the dataset that should be merged with each imputed dataset. The input to this argument should be a dataframe.}

\item{By}{This optional argument specifies the variable based on which the "MergingData" and imputed datasets should be merged. The input to this argument should be the name of the dataset column. The default is "ID".}

\item{Weighting}{This argument specifies whether the matched samples should be combined or the inverse propensity score weighted samples. The default is "FALSE", meaning that the matched samples will be returned.}

\item{...}{Additional arguments to be passed to the matching method.}
}
\value{
This function returns an object of the "mids" class (multiply imputed datasets), either combined from the matched samples (when the "Weighting" is "FALSE") or from the inverse propensity score weighted samples (when the "Weighting" is "TRUE").
}
\description{
Selects Matched Samples from Multiply Imputed Datasets
}
\examples{

\donttest{
#### Loading Packages

#The `mice` and the `MatchIt.mice` packages are loaded.

library(mice)
library(MatchIt.mice)

#### Loading the Datasets

#The dataset `HandOsteoarthritis` contains data of 4,704 individuals with (`HANDOA = 1`) or without
#(`HANDOA = 0`) hand osteoarthritis. The recorded data has missing values in body mass index
#(`BMI`, a quantitative variable), hand use habits (`HANDUSE`, a binary qualitative variable), and
#smoking status (`SMOKING`, a categorical qualitative variable).

data(HandOsteoarthritis)
factorized <- c("SIDE", "SEX", "HANDUSE", "SMOKING", "HANDOA")
HandOsteoarthritis[factorized] <- lapply(HandOsteoarthritis[factorized], factor)

#The dataset `KneeOsteoarthritis` contains data of 4,796 individuals (including the 4,704
#individuals of the `HandOsteoarthritis` dataset) with (`KNEEOA = 1`) or without (`KNEEOA = 0`) knee
#osteoarthritis.

data(KneeOsteoarthritis)
factorized <- c("KNEEOA")
KneeOsteoarthritis[factorized] <- lapply(KneeOsteoarthritis[factorized], factor)

#### Imputing the Missing Data Points

#The `mice()` function from the `mice` package is called to impute the missing data in the
#`HandOsteoarthritis` dataset.

datasets <- mice(HandOsteoarthritis, m = 5, maxit = 10,
                 method = c("", "", "", "", "mean", "polyreg", "logreg", ""))

#### Matching the Imputed Datasets

#The `matchitmice()` function from the `MatchIt.mice` package is called to select matched samples
#(for age, gender, and BMI) from the control (`HANDOA = 0`) and treatment (`HANDOA = 1`) groups of
#the each imputed dataset (in a 1:3 ratio and with a calliper of 5\% of the standard deviation of the
#estimated propensity score values).

matcheddatasets <- matchitmice(HANDOA ~ SEX + AGE + BMI, datasets, Ratio = 3, Caliper  =  0.05)

#The `matcheddatasets` is an object of the `mids` class and can be used in other studies, using the
#`with()` and `pool()` functions (from the `mice`) package).

results <- with(data = matcheddatasets,
                exp = glm(HANDOA ~ SMOKING, na.action = na.omit, family = binomial))
print(pool(results))

#### Estimating Weights of Individuals in the Imputed Datasets

#The `matchitmice()` function is applied to estimate inverse propensity score weights (from the
#model with age, gender, and BMI as the covariates) for each individual in the control and treatment
#groups of each imputed dataset.

matcheddatasets <- matchitmice(HANDOA ~ SEX + AGE + BMI, datasets, Weighting = TRUE)

#Similarly, the `matcheddatasets` can be used to estimate treatment effects on each imputed dataset.

results <- with(data = matcheddatasets,
                exp = glm(HANDOA ~ SMOKING, na.action = na.omit, family = binomial,
                weights = weight))
print(pool(results))

#### Merging Imputed Dataset with Other Datasets

#The `matchitmice()` function is applied to both select matched samples and to merge the imputed
#datasets with the `KneeOsteoarthritis` dataset, using the column name specified by the `By`.

matcheddatasets <- matchitmice(HANDOA ~ SEX + AGE + BMI, datasets, Ratio = 3, Caliper  =  0.05,
                               Merging = TRUE, MergingData = KneeOsteoarthritis, By = "ID")

#Similarly, the `matcheddatasets` can be used to estimate treatment effects on each imputed dataset.

results <- with(data = matcheddatasets,
                exp = glm(HANDOA ~ KNEEOA, na.action = na.omit, family = binomial))
print(pool(results))
}

#Loading the 'mice' and `MatchIt.mice` packages:

library(mice)
library(MatchIt.mice)

#Loading and polishing the `HandOsteoarthritis` dataset:

data(HandOsteoarthritis)
factorized <- c("SIDE", "SEX", "HANDUSE", "SMOKING", "HANDOA")
HandOsteoarthritis[factorized] <- lapply(HandOsteoarthritis[factorized], factor)

#Imputing the missing data points in 1 dataset of the `datasets` object after 1 iteration (In
#practice, usually there are 5 imputed datasets after 50 iterations, please see the
#https://github.com/FarhadPishgar/MatchIt.mice/ for more details):

datasets <- mice(HandOsteoarthritis, m = 1, maxit = 1,
                 method = c("", "", "", "", "mean", "polyreg", "logreg", ""))

#Selecting matched samples (for age, gender, and body mass index) from the control and treatment
#groups of each imputed dataset:

matcheddatasets <- matchitmice(HANDOA ~ SEX + AGE + BMI, datasets)
}
