#' One-step estimator for unlabeled data (multi‐dist)
#'
#' Fits the one-step estimator by maximizing the unlabeled likelihood via TMB,
#' automatically differentiating the objective, gradient, and Hessian.
#'
#' @param Y Numeric response vector.
#' @param Xhat Numeric matrix of regressors *excluding* the intercept. The
#'             **first** column must be the ML-generated regressor to correct.
#' @param homoskedastic Logical; if `TRUE`, assume a single error variance.
#' @param distribution Character: one of `"normal"`, `"t"`, `"laplace"`, `"gamma"`, or `"beta"`. Specifies which conditional density to use for residuals in the likelihood estimation.
#' @param nu Numeric; degrees of freedom (only used if `distribution = "t"`).
#' @param gshape,gscale Numeric; shape & scale for Gamma (only if `distribution = "gamma"`).
#' @param ba,bb Numeric; alpha & beta for Beta (only if `distribution = "beta"`).
#' @param intercept Logical; if `TRUE`, an intercept column of 1’s is prepended.
#'
#' @return An object of class `mlbc_fit` and subclass `mlbc_onestep` with:
#'   * `coef`: Named numeric vector of estimated coefficients.
#'   * `cov` : Variance–covariance matrix.
#'
#' @importFrom TMB MakeADFun
#' @references Battaglia, Christensen, Hansen, and Sacher (2025). "Inference for Regression with Variables Generated by AI or Machine Learning".
#' @examples
#' set.seed(2025)
#'
#' # 1) Simulate “unlabeled” data
#' n     <- 200
#' p     <- 0.3
#' Xtrue <- rbinom(n, 1, p)
#' # ML regressor with 10% false positives
#' Xhat  <- ifelse(runif(n) < 0.10, 1 - Xtrue, Xtrue)
#' Y     <- 1 + 2 * Xtrue + rnorm(n)
#'
#' # 2) Small validation set to estimate fpr
#' m        <- 50
#' Xval_t   <- rbinom(m, 1, p)
#' Xval_h   <- ifelse(runif(m) < 0.10, 1 - Xval_t, Xval_t)
#' fpr_hat  <- mean(Xval_h == 1 & Xval_t == 0)
#'
#' # 3) One‐step TMB estimator (Normal), with intercept
#' fit <- one_step(
#'   Y            = Y,
#'   Xhat         = matrix(Xhat, ncol = 1, dimnames = list(NULL, "Xhat")),
#'   homoskedastic = FALSE,
#'   distribution  = "normal",
#'   intercept     = TRUE
#' )
#' print(fit)
#' @export

one_step <- function(Y,
                     Xhat,
                     homoskedastic = FALSE,
                     distribution  = c("normal","t","laplace","gamma","beta"),
                     nu            = 4,
                     gshape        = 2, gscale = 1,
                     ba            = 2, bb = 2,
                     intercept     = TRUE) {

  distribution <- match.arg(distribution)
  dist_code <- switch(distribution,
                      normal  = 1L,
                      t       = 2L,
                      laplace = 3L,
                      gamma   = 4L,
                      beta    = 5L
  )


  if (intercept) {
    Xhat    <- cbind(Intercept = 1, Xhat)
    gen_idx <- 2L
  } else {
    gen_idx <- 1L
  }


  data <- list(
    Y            = as.numeric(Y),
    Xhat         = as.matrix(Xhat),
    homoskedastic= as.integer(homoskedastic),
    dist_code    = dist_code,
    nu           = nu,
    gshape       = gshape,
    gscale       = gscale,
    ba           = ba,
    bb           = bb
  )


  theta_init <- initial_guess(Y, Xhat, homoskedastic)


  obj <- MakeADFun(
    data       = data,
    parameters = list(theta = theta_init),
    DLL        = "MLBC",
    silent     = TRUE
  )


  opt <- nlminb(
    start     = obj$par,
    objective = obj$fn,
    gradient  = obj$gr,
    control   = list(iter.max = 200)
  )


  H      <- obj$he(opt$par)
  d      <- ncol(Xhat)
  b_raw  <- opt$par[1:d]
  V_raw  <- solve(H)[1:d, 1:d]


  if (intercept) {
    perm   <- c(gen_idx, setdiff(seq_len(d), gen_idx))
    b      <- b_raw[perm]
    V      <- V_raw[perm, perm]
    names(b) <- colnames(Xhat)[perm]
  } else {
    b      <- b_raw
    V      <- V_raw
    names(b) <- colnames(Xhat)
  }

  out <- list(coef = b, cov = V)
  class(out) <- c("mlbc_fit", "mlbc_onestep")
  out
}
