% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_ThermalLifetime.R
\name{calc_ThermalLifetime}
\alias{calc_ThermalLifetime}
\title{Calculates the Thermal Lifetime using the Arrhenius equation}
\usage{
calc_ThermalLifetime(
  E,
  s,
  T = 20,
  output_unit = "Ma",
  profiling = FALSE,
  profiling_config = list(),
  verbose = TRUE,
  plot = TRUE,
  ...
)
}
\arguments{
\item{E}{\link{numeric} (\strong{required}):
vector of trap depths in eV,
if \code{profiling = TRUE} only the first two elements are considered}

\item{s}{\link{numeric} (\strong{required}):
vector of frequency factor in 1/s,
if \code{profiling = TRUE} only the first two elements are considered}

\item{T}{\link{numeric} (\emph{with default}):
temperature in deg. C for which the lifetime(s) will be calculated.
A vector can be provided.}

\item{output_unit}{\link{character} (\emph{with default}):
output unit of the calculated lifetimes, accepted
entries are: \code{"Ma"}, \code{"ka"}, \code{"a"}, \code{"d"}, \code{"h"}, \code{"min"}, \code{"s"}}

\item{profiling}{\link{logical} (\emph{with default}):
this option allows to estimate uncertainties based on
given E and s parameters and their corresponding standard error
(cf. details and examples section)}

\item{profiling_config}{\link{list} (\emph{optional}):
allows to set configuration parameters used for the profiling
(and only have an effect here). Supported parameters are:
\itemize{
\item \code{n} (number of MC runs),
\item \code{E.distribution} (distribution used for the re-sampling for E) and
\item \code{s.distribution} (distribution used for the re-sampling for s).
}

Currently only the normal distribution is supported
(e.g., \code{profiling_config = list(E.distribution = "norm")}}

\item{verbose}{\link{logical} (\emph{with default}):
enable/disable output to the terminal.}

\item{plot}{\link{logical} (\emph{with default}):
enable/disable the plot output, currently only in combination with
\code{profiling = TRUE}.}

\item{...}{further arguments that can be passed in combination with the plot output.
Standard plot parameters are supported (\link{plot.default})}
}
\value{
A \linkS4class{RLum.Results} object is returned a along with a plot (for
\code{profiling = TRUE}). The output object contain the following slots:

\strong{\verb{@data}}

\tabular{lll}{
\strong{Object} \tab \strong{Type} \tab \strong{Description} \cr
\code{lifetimes} \tab \link{array} or \link{numeric} \tab calculated lifetimes \cr
\code{profiling_matrix} \tab \link{matrix} \tab profiling matrix used for the MC runs
}

\strong{\verb{@info}}

\tabular{lll}{
\strong{Object} \tab \strong{Type} \tab \strong{Description} \cr
\code{call} \tab \code{call} \tab the original function call
}
}
\description{
The function calculates the thermal lifetime of charges for given E (in eV), s (in 1/s) and
T (in deg. C.) parameters. The function can be used in two operational modes:
}
\details{
\strong{Mode 1 \code{(profiling = FALSE)}}

An arbitrary set of input parameters (E, s, T) can be provided and the
function calculates the thermal lifetimes using the Arrhenius equation for
all possible combinations of these input parameters. An array with 3-dimensions
is returned that can be used for further analyses or graphical output (see example 1)

\strong{Mode 2 \code{(profiling = TRUE)}}

This mode tries to profile the variation of the thermal lifetime for a chosen
temperature by accounting for the provided E and s parameters and their corresponding
standard errors, e.g., \code{E = c(1.600, 0.001)}
The calculation based on a Monte Carlo simulation, where values are sampled from a normal
distribution (for E and s).

\strong{Used equation (Arrhenius equation)}

\deqn{\tau = 1/s exp(E/kT)}
where:
\eqn{\tau} in s as the mean time an electron spends in the trap for a given \eqn{T},
\eqn{E} trap depth in eV,
\eqn{s} the frequency factor in 1/s,
\eqn{T} the temperature in K and \eqn{k} the Boltzmann constant in eV/K (cf. Furetta, 2010).
}
\note{
The profiling is currently based on re-sampling from a normal distribution, this
distribution assumption might be, however, not valid for given E and s parameters.
}
\section{Function version}{
 0.1.0
}

\examples{

##EXAMPLE 1
##calculation for two trap-depths with similar frequency factor for different temperatures
E <- c(1.66, 1.70)
s <- 1e+13
T <- 10:20
temp <- calc_ThermalLifetime(
  E = E,
  s = s,
  T = T,
  output_unit = "Ma"
)
graphics::contour(x = E, y = T, z = temp$lifetimes[1,,],
        ylab = "Temperature [\u00B0C]",
        xlab = "Trap depth [eV]",
        main = "Thermal Lifetime Contour Plot"
)
mtext(side = 3, "(values quoted in Ma)")

##EXAMPLE 2
##profiling of thermal life time for E and s and their standard error
E <- c(1.600, 0.003)
s <- c(1e+13,1e+011)
T <- 20
calc_ThermalLifetime(
  E = E,
  s = s,
  T = T,
  profiling = TRUE,
  output_unit = "Ma"
)

} 

\section{How to cite}{
Kreutzer, S., 2025. calc_ThermalLifetime(): Calculates the Thermal Lifetime using the Arrhenius equation. Function version 0.1.0. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., Colombo, M., Steinbuch, L., Boer, A.d., 2025. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 1.1.0. https://r-lum.github.io/Luminescence/
}

\references{
Furetta, C., 2010. Handbook of Thermoluminescence, Second Edition. World Scientific.
}
\seealso{
\link[graphics:matplot]{graphics::matplot}, \link[stats:Normal]{stats::rnorm}, \link{get_RLum}
}
\author{
Sebastian Kreutzer, Institute of Geography, Heidelberg University (Germany)
, RLum Developer Team}
\keyword{datagen}
