% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/interval2state.R
\name{interval2state}
\alias{interval2state}
\title{Adds a state column to a dataset from interval data}
\usage{
interval2state(
  dataset,
  State.interval.dataset,
  Datetime.colname = Datetime,
  State.colname = State,
  Interval.colname = Interval,
  Id.colname.dataset = Id,
  Id.colname.interval = Id,
  overwrite = FALSE,
  output.dataset = TRUE
)
}
\arguments{
\item{dataset}{A light logger dataset. Expects a \code{dataframe}. If not imported
by \link{LightLogR}, take care to choose a sensible variable for the
\code{Datetime.colname}.}

\item{State.interval.dataset}{Name of the dataset that contains \code{State} and
\code{Interval} columns. Interval data can be created, e.g., through
\code{\link[=sc2interval]{sc2interval()}}.}

\item{Datetime.colname}{column name that contains the datetime. Defaults to
\code{"Datetime"} which is automatically correct for data imported with
\link{LightLogR}. Expects a \code{symbol}. Needs to be part of the \code{dataset}.}

\item{State.colname, Interval.colname}{Column names of the \code{State} and
\code{Interval} in the \code{State.interval.dataset}. Expects a \code{symbol}. \code{State}
can't be in the \code{dataset} yet or the function will give an error. You can
also set \code{overwrite = TRUE}.}

\item{Id.colname.dataset, Id.colname.interval}{Column names of the
participant's \code{Id} in both the \code{dataset} and the \code{State.interval.dataset}.
On the off-chance that there are inconsistencies, the names can be
different. If the datasets where imported and preprocessed with
\link{LightLogR}, this just works. Both datasets need an \code{Id}, because the
states will be added based not only on the \code{Datetime}, but also depending
on the dataset.}

\item{overwrite}{If \code{TRUE} (defaults to \code{FALSE}), the function will
overwrite the \code{State.colname} column if it already exists.}

\item{output.dataset}{should the output be a \code{data.frame} (Default \code{TRUE}) or
a vector with \code{hms} (\code{FALSE}) times? Expects a \code{logical} scalar.}
}
\value{
One of
\itemize{
\item a \code{data.frame} object identical to \code{dataset} but with the state column added
\item a \code{vector} with the states
}
}
\description{
This function can make use of \code{Interval} data that contain \code{States} (like
\code{"sleep"}, \code{"wake"}, \code{"wear"}) and add a column to a light logger \code{dataset},
where the \code{State} of  every \code{Datetime} is specified, based on the
participant's \code{Id}.
}
\examples{
#create a interval dataset
library(tibble)
library(dplyr)
library(lubridate)
library(rlang)
library(purrr)
states <- tibble::tibble(Datetime = c("2023-08-15 6:00:00",
                                      "2023-08-15 23:00:00",
                                      "2023-08-16 6:00:00",
                                      "2023-08-16 22:00:00",
                                      "2023-08-17 6:30:00",
                                      "2023-08-18 1:00:00",
                                      "2023-08-18 6:00:00",
                                      "2023-08-18 22:00:00",
                                      "2023-08-19 6:00:00",
                                      "2023-08-19 23:00:00",
                                      "2023-08-20 6:00:00",
                                      "2023-08-20 22:00:00"),
                         State = rep(c("wake", "sleep"), 6),
                         Wear = rep(c("wear", "no wear"), 6),
                         Performance = rep(c(100, 0), 6),
                         Id = "Participant")
intervals <- sc2interval(states)

#create a dataset with states
dataset_with_states <-
sample.data.environment \%>\%
interval2state(State.interval.dataset = intervals)

#visualize the states - note that the states are only added to the respective ID in the dataset
library(ggplot2)
ggplot(dataset_with_states, aes(x = Datetime, y = MEDI, color = State)) +
 geom_point() +
 facet_wrap(~Id, ncol = 1)

#import multiple State columns from the interval dataset
#interval2state will only add a single State column to the dataset, 
#which represents sleep/wake in our case
dataset_with_states[8278:8283,]

#if we want to add multiple columns we can either perfom the function 
#multiple times with different states:
dataset_with_states2 <- 
dataset_with_states \%>\%
interval2state(State.interval.dataset = intervals, State.colname = Wear)
dataset_with_states2[8278:8283,]

#or we can use `purrr::reduce` to add multiple columns at once
dataset_with_states3 <-
syms(c("State", "Wear", "Performance")) \%>\% 
reduce(\(x,y) interval2state(x, State.interval.dataset = intervals, State.colname = !!y), 
.init = sample.data.environment)

#Note: 
# - the State.colnames have to be provided as symbols (`rlang::syms`)
# - the reduce function requires a two argument function `\(x,y)`, where `x` 
#   is the dataset to be continiously modified and `y` is the symbol of the
#   State column name to be added
# - the `!!` operator from `rlang` is used to exchange `y` with each symbol
# - the `.init` argument is the initial dataset to be modified

#this results in all states being applied
dataset_with_states3[8278:8283,]
}
