#' Print kHTML Objects
#' 
#' By default, we \code{cat} out kHTML objects as we typically
#' intend to embed them in R Markdown documents. This is mainly used for
#' printing of items in the environment \code{html}.
#' @param ... a set of kHTML objects (strings).
#' @method print kHTML
#' @seealso \code{\link{html}}
#' @export
#' @examples
#' data(html)
#' html$br()
print.kHTML <- function(...) {
  cat( ... )
}

#' Make HTML Elements
#' 
#' Creates a function that returns a function that can be used to generate
#' HTML elements. See examples for usage.
#' 
#' This function returns a function that can be called as an HTML tag
#' generating function. For example, by calling
#' \code{p <- makeHTMLTag("p")}, we can generate a function that interprets
#' all named arguments as attributes, and all unnamed arguments as
#' 'data', which is generated for a \code{p} HTML tag.
#' 
#' @export
#' @param tag the HTML tag to use
#' @param ... a collection of named and unnamed arguments;
#'  named arguments are parsed as attributes of the tag,
#'  unnamed arguments are pasted together into the inner data of the tag.
#' @seealso \code{\link{html}}
#' @examples
#' div <- makeHTMLTag("div")
#' my_class = "orange"
#' x <- "some text"
#' div( class=my_class, id="hello", "This is ", x )
makeHTMLTag <- function(tag, ...) {
  
  ## force tag into a closure
  force(tag)
  
  return( function(...) {
    
    ## get dot args
    dotArgs <- list(...)
    
    ## process attributes from named arguments
    preparsedArgs <- as.list( match.call(expand.dots=FALSE)$`...` )
    namedArgs <- dotArgs[ names( preparsedArgs ) != "" ]
    
    if( length( namedArgs ) > 0 ) {
      attrs <- paste( sep="", " ", paste( names(namedArgs), 
                      paste( sep="",  "'",
                              unlist( namedArgs ),
                              "'" ),
                      sep = "=", collapse = " " ) )
    } else {
      attrs <- NULL
    }
    
    ## process data from unnamed arguments
    if( is.null( names( preparsedArgs ) ) ) {
      unnamedArgs <- dotArgs
    } else {
      unnamedArgs <- dotArgs[ names(preparsedArgs) == "" ]
    }
   
    data <- paste( unnamedArgs, sep="", collapse="" )
    
    if( length( unnamedArgs ) == 0 ) {
      out <- paste( "<", tag, attrs, " />", sep="", collapse="" )
    } else {
      out <- paste( "<", tag, attrs, ">", data, "</", tag, ">",
                    sep="", collapse="")
    }
    
    class(out) <- "kHTML"
    return(out)
    
  } )
  
}

#' Attach Common, Non-Masking HTML Functions
#' 
#' Calling this function will attach HTML tagging functions that 
#' do not mask any functions available in the 'base' R packages.
#' 
#' @seealso \code{\link{detachHTML}}
#' @export
attachHTML <- function() {
  
  if( ".html.sub" %in% search() ) {
    warning("HTML functions already attached!")
    invisible()
  } else {
    data("html")
    myFuns <- c(
      paste( sep="", "h", 1:6),
      "tr", "td", "th", "ul", "ol", "li", "p", "span", "br", "strong", "em", "a",
      "hr", "div", "link", "script", "pre"
    )
    .html.sub <- new.env()
    for( fun in myFuns ) {
      assign( fun, get(fun, envir=get("html")), envir=.html.sub )
    }
    
    attach(.html.sub)
  }
  
}

#' Detach Common, Non-Masking HTML Functions
#' 
#' Calling this function will detach the environment generated by
#' \code{attachHTML()}.
#' 
#' @seealso \code{\link{attachHTML}}
#' @export
detachHTML <- function() {
  
  if( ".html.sub" %in% search() ) {
    detach(".html.sub")
  } else {
    stop("HTML functions not attached.")
  }  
}

# Generate HTML tag environment
# 
# Generates the environment used for the HTML utility functions. Included
# as means of documentation for where the HTML utility functions come from.
# html <- new.env()
# html_tags <- scan( what=character(), sep="\n", strip.white=TRUE,
#                    "./data/html.txt" )
# tags <- gsub( "(<)(.*?)(>)", "\\2", html_tags, perl=TRUE )
# for( tag in tags ) {
#   assign( tag, makeHTMLTag(tag), envir=html )
# }
# save( html, file=paste( sep="",  getwd(), "/data/html.rda" ) )