% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/isoscape.R
\name{isoscape}
\alias{isoscape}
\alias{print.isoscape}
\alias{summary.isoscape}
\title{Predicts the spatial distribution of isotopic values}
\usage{
isoscape(elevation.raster, isofit, verbose = interactive())
}
\arguments{
\item{elevation.raster}{The elevation raster (\var{RasterLayer}) created by
\code{\link{relevate}}}

\item{isofit}{The fitted isoscape created by \code{\link{isofit}}}

\item{verbose}{A \var{logical} indicating whether information about the
progress of the procedure should be displayed or not while the function is
running. By default verbose is \var{TRUE} if users use an interactive R
session and \var{FALSE} otherwise.}
}
\value{
This function returns a \var{list} of class \var{isoscape}
containing a stack of all 8 raster layers mentioned above (all being of
class \var{RasterLayer}), and the location of the sources as spatial points.
}
\description{
This function produces the isoscape, i.e. a spatial prediction (i.e. map) of
the distribution of isotopic delta values. Predictions are computed using
the fitted isoscape for each raster cell of the elevation raster. All shape
files can be exported and loaded into any Geographic Information System
(GIS) if needed.
}
\details{
This function computes the predictions (\code{mean}), prediction variances
(\code{mean.predVar}), residual variances (\code{mean.residVar}) and
response variances (\code{mean.respVar}) for the isotopic values at a
resolution equal to the one of the elevation raster. It also computes the
same information for the residual dispersion variance (\code{disp.pred},
\code{disp.predVar}, \code{disp.residVar}, or \code{disp.respVar}).

The predictions of isotopic values across the landscape are performed by
calling the function \code{\link[spaMM]{predict}} from the package
\pkg{\link[spaMM]{spaMM}} on the fitted isoscape produced by \code{\link{isofit}}.

Let us detail the meaning of \code{mean}, \code{mean.predVar} and
\code{mean.respVar}:

Our model assumes that that there is a single true unknown isoscape, which
is fixed but which is represented by the mixed-effect model as a random draw
from possible realizations of isoscapes (random draws of the
Matern-correlated process and of the uncorrelated random effects if
considered). We infer this realized isoscape by fitting the model to a
limited amount of data, with some uncertainty since different random draws
of the unknown isoscape may give the same observed data. There is thus a
conditional distribution of possible true isoscapes given the data. For
linear mixed-effects models, the mean prediction, technically called the
best linear unbiased prediction (BLUP), is the mean of this conditional
distribution. The prediction variance is ideally the mean square difference
between the true unknown value of the linear predictor and the BLUP at a
given location. The response variance has a different meaning. It estimates
the variance of new observations drawn from the true unknown isoscape at a
given location. The response variance is simply equal to the sum of the
prediction variance and the residual variance (note that the residual
variance considered assume that a single observation is being observed per
location).

The isoscape can be plotted using the function \code{\link{plot.isoscape}}
(see examples).
}
\examples{

## The examples below will only be run if sufficient time is allowed
## You can change that by typing e.g. IsoriX.options(example_maxtime = XX)
## if you want to allow for examples taking up to ca. XX seconds to run
## (so don't write XX but put a number instead!)

if(IsoriX.getOption("example_maxtime") > 30) {

## We prepare the data
GNIPDataDEagg <- prepdata(data = GNIPDataDE)

## We fit the models
GermanFit <- isofit(iso.data = GNIPDataDEagg,
                    mean.model.fix = list(elev = TRUE, lat.abs = TRUE))

## We build the isoscapes
isoscape <- isoscape(elevation.raster = ElevRasterDE,
                     isofit = GermanFit)

isoscape

## We build the plots
plot.mean <- plot(x = isoscape, which = "mean", plot = FALSE)

plot.mean.predVar <- plot(x = isoscape, which = "mean.predVar", plot = FALSE)

plot.mean.residVar <- plot(x = isoscape, which = "mean.residVar", plot = FALSE)

plot.mean.respVar <- plot(x = isoscape, which = "mean.respVar", plot = FALSE)

## We display the plots
if(require(lattice)) {
 print(plot.mean, split = c(1, 1, 2, 2), more = TRUE)
 print(plot.mean.predVar,   split = c(2, 1, 2, 2), more = TRUE)
 print(plot.mean.residVar,  split = c(1, 2, 2, 2), more = TRUE)
 print(plot.mean.respVar,   split = c(2, 2, 2, 2), more = FALSE)
}

}

}
\seealso{
\code{\link{isofit}} for the function fitting the isoscape

\code{\link{plot.isoscape}} for the function plotting the isoscape model

\code{\link{IsoriX}} for the complete work-flow
}
\keyword{models}
\keyword{predict}
\keyword{prediction}
\keyword{regression}
