% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.fossils.R
\name{plot.fossils}
\alias{plot.fossils}
\title{Plot simulated fossils}
\usage{
\method{plot}{fossils}(x, tree, show.fossils = TRUE, show.tree = TRUE,
  show.ranges = FALSE, show.strata = FALSE, strata = 1,
  max.age = NULL, interval.ages = NULL, binned = FALSE,
  show.axis = TRUE, show.proxy = FALSE, proxy.data = NULL,
  show.preferred.environ = FALSE, preferred.environ = NULL,
  show.taxonomy = FALSE, taxonomy = NULL, root.edge = TRUE,
  hide.edge = FALSE, edge.width = 1, show.tip.label = FALSE,
  align.tip.label = FALSE, fossil.col = 1, range.col = rgb(0, 0, 1),
  extant.col = NULL, cex = 1.2, pch = 18, ...)
}
\arguments{
\item{x}{Fossils object.}

\item{tree}{Phylo object.}

\item{show.fossils}{If TRUE plot fossils (default = TRUE).}

\item{show.tree}{If TRUE plot the tree  (default = TRUE).}

\item{show.ranges}{If TRUE plot stratigraphic ranges (default = FALSE). If show.taxonomy = FALSE all occurrences along a single edge are grouped together (i.e. function assumes all speciation is symmetric).}

\item{show.strata}{If TRUE plot strata  (default = FALSE).}

\item{strata}{Number of stratigraphic intervals (default = 1).}

\item{max.age}{Maximum age of a set of equal length intervals. If no value is specified (max = NULL), the function uses a maximum age based on tree height.}

\item{interval.ages}{Vector of stratigraphic interval ages, starting with the minimum age of the youngest interval and ending with the maximum age of the oldest interval.}

\item{binned}{If TRUE fossils are plotted at the mid point of each interval.}

\item{show.axis}{If TRUE plot x-axis (default = TRUE).}

\item{show.proxy}{If TRUE add profile of sampling data to plot (e.g rates in time-dependent rates model) (default = FALSE).}

\item{proxy.data}{Vector of sampling proxy data (default = NULL). Should be as long as the number of stratigraphic intervals.}

\item{show.preferred.environ}{If TRUE add species preferred environmental value (e.g. water depth) (default = FALSE). Only works if combined with \code{show.proxy = TRUE}.}

\item{preferred.environ}{Preferred environmental value (e.g. water depth). Currently only one value can be shown.}

\item{show.taxonomy}{If TRUE highlight species taxonomy.}

\item{taxonomy}{Taxonomy object.}

\item{root.edge}{If TRUE include the root edge (default = TRUE).}

\item{hide.edge}{If TRUE hide the root edge but still incorporate it into the automatic timescale (default = FALSE).}

\item{edge.width}{A numeric vector giving the width of the branches of the plotted phylogeny. These are taken to be in the same order as the component edge of \code{tree}. If fewer widths are given than the number of edges, then the values are recycled.}

\item{show.tip.label}{Whether to show the tip labels on the phylogeny (defaults to FALSE).}

\item{align.tip.label}{A logical value or an integer. If TRUE, the tips are aligned and dotted lines are drawn between the tips of the tree and the labels. If an integer, the tips are aligned and this gives the type of the lines (following \code{lty}).}

\item{fossil.col}{Colour of fossil occurrences.}

\item{range.col}{Colour of stratigraphic ranges.}

\item{extant.col}{Colour of extant samples. If show.taxonomy = TRUE extant.col will be ignored.}

\item{cex}{Numeric value giving the factor used to scale the points representing the fossils when \code{show.fossils = TRUE}.}

\item{pch}{Numeric value giving the symbol used for the points representing the fossils when \code{show.fossils = TRUE}.}

\item{...}{Additional parameters to be passed to \code{plot.default}.}
}
\description{
This function is adapted from the \emph{ape} function \code{plot.phylo} used to plot phylogenetic trees.
The function can be used to plot simulated fossils (\code{show.fossils = TRUE}), with or without the corresponding tree (\code{show.tree = TRUE}),
stratigraphic intervals (\code{show.strata = TRUE}), stratigraphic ranges (\code{show.ranges = TRUE}) and sampling proxy data (\code{show.proxy = TRUE}).
Interval ages can be specified as a vector (\code{interval.ages}) or a uniform set of interval ages can be specified using the
number of intervals (\code{strata}) and maximum interval age (\code{max}), where interval length \eqn{= max.age/strata}.
If no maximum age is specified, the function calculates a maximum interval age slightly older than the root edge (or root age if \code{root.edge = FALSE}),
using the function \code{tree.max()}.
}
\examples{
set.seed(123)

## simulate tree
t = TreeSim::sim.bd.taxa(8, 1, 1, 0.3)[[1]]

## simulate fossils under a Poisson sampling process
f = sim.fossils.poisson(rate = 3, tree = t)
plot(f, t)
# add a set of equal length strata
plot(f, t, show.strata = TRUE, strata = 4)
# show stratigraphic ranges
plot(f, t, show.strata = TRUE, strata = 4, show.ranges = TRUE)

## simulate fossils and highlight taxonomy
s = sim.taxonomy(t, 0.5, 1)
f = sim.fossils.poisson(rate = 3, taxonomy = s)
plot(f, t, taxonomy = s, show.taxonomy = TRUE, show.ranges = TRUE)


## simulate fossils under a non-uniform model of preservation
# assign a max interval based on tree height
max.age = tree.max(t)
times = c(0, 0.3, 1, max.age)
rates = c(4, 1, 0.1)
f = sim.fossils.intervals(t, interval.ages = times, rates = rates)
plot(f, t, show.strata = TRUE, interval.ages = times)
# add proxy data
plot(f, t, show.strata = TRUE, interval.ages = times, show.proxy = TRUE, proxy.data = rates)

}
