% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/depletion.R
\name{depletion}
\alias{depletion}
\alias{plot.depletion}
\alias{summary.depletion}
\alias{coef.depletion}
\alias{anova.depletion}
\alias{confint.depletion}
\alias{depletion.formula}
\alias{depletion.default}
\alias{rSquared.depletion}
\title{Computes the Leslie or DeLury population estimate from catch and effort data.}
\usage{
depletion(catch, ...)

\method{depletion}{formula}(
  catch,
  data,
  method = c("Leslie", "DeLury", "Delury"),
  Ricker.mod = FALSE,
  ...
)

\method{depletion}{default}(
  catch,
  effort,
  method = c("Leslie", "DeLury", "Delury"),
  Ricker.mod = FALSE,
  ...
)

\method{summary}{depletion}(
  object,
  parm = c("all", "both", "No", "q", "lm"),
  verbose = FALSE,
  as.df = FALSE,
  ...
)

\method{coef}{depletion}(object, parm = c("all", "both", "No", "q", "lm"), as.df = FALSE, ...)

\method{confint}{depletion}(
  object,
  parm = c("all", "both", "No", "q", "lm"),
  level = conf.level,
  conf.level = 0.95,
  incl.est = FALSE,
  as.df = FALSE,
  ...
)

\method{anova}{depletion}(object, ...)

\method{rSquared}{depletion}(object, digits = getOption("digits"), percent = FALSE, ...)

\method{plot}{depletion}(
  x,
  xlab = NULL,
  ylab = NULL,
  pch = 19,
  col.pt = "black",
  col.mdl = "gray70",
  lwd = 1,
  lty = 1,
  pos.est = "topright",
  cex.est = 0.95,
  ...
)
}
\arguments{
\item{catch}{A numeric vector of catches of fish at each time, or a formula of the form \code{catch~effort}.}

\item{\dots}{Additional arguments for methods.}

\item{data}{A data.frame from which the variables in the \code{catch} formula can be found. Not used if \code{catch} is not a formula.}

\item{method}{A single string that indicates which depletion method to use}

\item{Ricker.mod}{A single logical that indicates whether to use the modification proposed by Ricker (=TRUE) or not (=FALSE, default).}

\item{effort}{A numeric vector of efforts expended at each time.}

\item{object}{An object saved from the \code{removal} call (i.e., of class \code{depletion}).}

\item{parm}{A specification of which parameters are to be given confidence intervals, either a vector of numbers or a vector of names. If missing, all parameters are considered.}

\item{verbose}{A logical that indicates whether a reminder of the method used should be printed with the summary results.}

\item{as.df}{A logical that indicates whether the results of \code{coef}, \code{confint}, or \code{summary} should be returned as a data.frame. Ignored in \code{summary} if \code{parm="lm"}.}

\item{level}{Same as \code{conf.level} but used for compatibility with generic \code{confint} function.}

\item{conf.level}{A single number that represents the level of confidence to use for constructing confidence intervals.}

\item{incl.est}{A logical that indicated whether the parameter point estimate should be included in the results from \code{confint}. Defaults to \code{FALSE}.}

\item{digits}{The number of digits to round the \code{rSquared} result to.}

\item{percent}{A logical that indicates if the \code{rSquared} result should be returned as a percentage (\code{=TRUE}) or as a proportion (\code{=FALSE}; default).}

\item{x}{An object saved from the \code{depletion} call (i.e., of class \code{depletion}).}

\item{xlab}{A label for the x-axis.}

\item{ylab}{A label for the y-axis.}

\item{pch}{A numeric that indicates the type of plotting character.}

\item{col.pt}{A string that indicates the color of the plotted points.}

\item{col.mdl}{A string that indicates the color of the fitted line.}

\item{lwd}{A numeric that indicates the line width of the fitted line.}

\item{lty}{A numeric that indicates the type of line used for the fitted line.}

\item{pos.est}{A single string to identify where to place the estimated population estimate and catchability on the plot. Can be set to one of \code{"bottomright"}, \code{"bottom"}, \code{"bottomleft"}, \code{"left"}, \code{"topleft"}, \code{"top"}, \code{"topright"}, \code{"right"} or \code{"center"} for positioning the estimated mortality rates on the plot. Typically \code{"bottomleft"} (DEFAULT) and \code{"topright"} will be \dQuote{out-of-the-way} placements. Set \code{pos.est} to \code{NULL} to remove the estimated population size and catchability coefficient from the plot.}

\item{cex.est}{A single numeric that identifies the character expansion value for the estimated population estimate and catchability placed on the plot.}
}
\value{
A list with the following items:
 \itemize{
   \item method A string that indicates whether the \code{"Leslie"} or \code{"DeLury"} model was used.
   \item catch The original vector of catches.
   \item effort The original vector of efforts.
   \item cpe A computed vector of catch-per-unit-effort for each time.
   \item KorE A computed vector of cumulative catch (K; Leslie method) or effort (E; DeLury method).
   \item lm The \code{lm} object from the fit of CPE on K (Leslie method) or log(CPE) on E (DeLury method).
   \item est A 2x2 matrix that contains the estimates and standard errors for No and q.
 }
}
\description{
Computes the Leslie or DeLury estimates of population size and catchability coefficient from paired catch and effort data. The Ricker modification may also be used.
}
\details{
For the Leslie method, a linear regression model of catch-per-unit-effort on cumulative catch prior to the sample is fit. The catchability coefficient (q) is estimated from the negative of the slope and the initial population size (No) is estimated by dividing the intercept by the catchability coefficient. If \code{Ricker.mod=TRUE} then the cumulative catch is modified to be the cumulative catch prior to the sample plus half of the catch of the current sample.

For the DeLury method, a linear regression model of log (catch-per-unit-effort) on cumulative effort is fit. The catchability coefficient (q) is estimated from the negative of the slope and the initial population size (No) is estimated by dividing the intercept as an exponent of e by the catchability coefficient. If \code{Ricker.mod=TRUE} then the cumulative effort is modified to be the cumulative effort prior to the sample plus half of the effort of the current sample.

Standard errors for the catchability and population size estimates are computed from formulas on page 298 (for Leslie) and 303 (for DeLury) from Seber (2002). Confidence intervals are computed using standard large-sample normal distribution theory with the regression error df.
}
\section{testing}{
  The Leslie method without the Ricker modification and the DeLury method with the Ricker modification matches the results from \code{\link[fishmethods]{deplet}} in \pkg{fishmethods} for the \code{\link[fishmethods]{darter}} (from \pkg{fishmethods}), \code{\link[FSAdata]{LobsterPEI}} and \code{\link[FSAdata]{BlueCrab}} from \pkg{FSAdata}, and \code{\link{SMBassLS}} for N0 to whole numbers, the SE for No to one decimal, q to seven decimals, and the SE of q to at least five decimals.

The Leslie method matches the results of Seber (2002) for N0, q, and the CI for Q but not the CI for N (which was so far off that it might be that Seber's result is incorrect) for the lobster data and the q and CI for q but the NO or its CI (likely due to lots of rounding in Seber 2002) for the Blue Crab data.

The Leslie and DeLury methods match the results of Ricker (1975) for No and Q but not for the CI of No (Ricker used a very different method to compute CIs).
}

\section{IFAR Chapter}{
 10-Abundance from Depletion Data.
}

\examples{
## Leslie model examples
# no Ricker modification
l1 <- depletion(SMBassLS$catch,SMBassLS$effort,method="Leslie")
summary(l1)
summary(l1,verbose=TRUE)
summary(l1,parm="No")
rSquared(l1)
rSquared(l1,digits=1,percent=TRUE)
coef(l1)
confint(l1)
confint(l1,incl.est=TRUE)
confint(l1,incl.est=TRUE,parm="No")
confint(l1,incl.est=TRUE,parm="q")
confint(l1,incl.est=TRUE,parm="lm")
plot(l1)

# with Ricker modification
l2 <- depletion(SMBassLS$catch,SMBassLS$effort,method="Leslie",Ricker.mod=TRUE)
summary(l2)
confint(l1,incl.est=TRUE)
plot(l2)

## DeLury model examples with no Ricker modification
d1 <- depletion(SMBassLS$catch,SMBassLS$effort,method="DeLury")
summary(d1)
rSquared(d1)
confint(d1,incl.est=TRUE)
plot(d1)

# Leslie model using formula notation
l3 <- depletion(catch~effort,data=SMBassLS)
summary(l3)

# Leslie model by group (requires dplyr package)
# Dummy example data (lake=="A" is SMBassLS example ... just FYI)
tmpdf <- data.frame(ct=c(131,69,99,78,56,76,49,42,63,47,
                         117,75,87,67,58,67,42),
                    ft=c(7,7,7,7,7,7,7,7,7,7,
                         5,7,5,5,4,6,5),
                    lake=as.factor(c(rep("A",10),rep("B",7))))
                                   
if (require(dplyr)) {
  res <- tmpdf \%>\%
    dplyr::group_by(lake) \%>\%
      dplyr::group_modify(~confint(depletion(ct~ft,data=.x),
                                   incl.est=TRUE,as.df=TRUE)) \%>\%
      as.data.frame() # removes tibble and grouping structure
  res
}

}
\references{
Ogle, D.H. 2016. \href{https://fishr-core-team.github.io/fishR/pages/books.html#introductory-fisheries-analyses-with-r}{Introductory Fisheries Analyses with R}. Chapman & Hall/CRC, Boca Raton, FL.

Ricker, W.E. 1975. Computation and interpretation of biological statistics of fish populations. Technical Report Bulletin 191, Bulletin of the Fisheries Research Board of Canada. [Was (is?) from http://www.dfo-mpo.gc.ca/Library/1485.pdf.]

Seber, G.A.F. 2002. The Estimation of Animal Abundance. Edward Arnold, Second edition (reprinted).
}
\seealso{
See \code{\link{removal}} for related functionality and \code{\link[fishmethods]{deplet}} in \pkg{fishmethods} for similar functionality.
}
\author{
Derek H. Ogle, \email{DerekOgle51@gmail.com}
}
\keyword{hplot}
\keyword{manip}
